﻿// <copyright file="ControlController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class ControlController
* Base class for controllers of controls
*/
define(['when', 'common/Enums', 'common/Error', 'common/Utilites', 'common/Appearence', 'base/EventTarget',
    'core/trigger/TriggerHandler', 'core/ActionFactory',
     'core/Event', 'controls/Scheme/SchemePreloader', 'core/Validator', 'helpers/contextResolver', 'helpers/PermissionChecker', 'core/connection/ParameterConnections', 'libs/hammer', 'core/L10n'],
function (when, Enums, Error, Utilites, Appearence, EventTarget,
    TriggerHandler, ActionFactory, Event, SchemePreloader, Validator, contextResolver, PermissionChecker,
    ParameterConnections, hammer, L10n) {

    var ABSOLUTE_X_CHANGED = 'absolute x changed',
		ABSOLUTE_Y_CHANGED = 'absolute y changed',
		ACTUAL_WIDTH_CHANGED = 'actual width changed',
		ACTUAL_HEIGHT_CHANGED = 'actual height changed',
		ACTUAL_VISIBLE_CHANGED = 'actual visible changed',
		ACTUAL_ENABLED_CHANGED = 'actual enabled changed';


    var ControlController = Class.extend({

        eventTarget: null,
        actions: null,
        triggerHandler: null,
        flashClass: 'flash',

        init: function () {
            this.eventTarget = new EventTarget();
            this.events = {};
            this.validator = null;
            this.loaded = false;
            this.modelChanged = {};
            this.modelChanged[Enums.ParameterRoles.WIDTH] = this.onWidthChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.WIDTH_UNITS] = this.onWidthUnitsChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.HEIGHT] = this.onHeightChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.HEIGHT_UNITS] = this.onHeightUnitsChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.Z] = this.onZChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.SKIN] = this.onSkinChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.ANGLE] = this.onAngleChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.SCALE_X] = this.onScaleXChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.SCALE_Y] = this.onScaleYChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.COLSPAN] = this.onColspanChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.ROWSPAN] = this.onRowspanChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.IS_ENABLED] = this.onIsEnabledChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.IS_VISIBLE] = this.onIsVisibleChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.FLASH] = this.onFlashChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.OPACITY] = this.onOpacityChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.SHADOW_SIZE] = this.onShadowSizeChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.SHADOW_COLOR] = this.onShadowColorChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.BACKGROUND_COLOR] = this.onBackgroundColorChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.BACKGROUND_TILE] = this.onBackgroundTileChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.RESOURCE] = this.onResourceChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.BORDER_COLOR] = this.applyBorder.bind(this);
            this.modelChanged[Enums.ParameterRoles.BORDER_STYLE] = this.applyBorder.bind(this);
            this.modelChanged[Enums.ParameterRoles.BORDER_THIKNESS] = this.applyBorder.bind(this);
            this.modelChanged[Enums.ParameterRoles.CORNER_RADIUS] = this.onCornerRadiusChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.THEME] = this.onThemeChanged.bind(this);
            this.modelChanged[Enums.ParameterRoles.ID] = this.renderId.bind(this);
            this.modelChanged[Enums.ParameterRoles.TAB_INDEX] = this.onTabIndexChanged.bind(this);
        },

        //
        //agility events handlers
        //

        /**
         *  @method Fired upon object creation         
         */
        onCreate: function () {
            this.mvc.model.subscribePropertyChanged(this.modelPropertyChanged, this);

            mergeActions.call(this, this.mvc.model.actions);
            this.triggerHandler = new TriggerHandler(this.actions);
            this.triggerHandler.add(this.mvc.model.triggers, this);
            this.attachEvents();
            this.onRender();
        },


        onBeforeDestroy: function () {
        },

        /**
         *  @method Fired before object is destroyed.
         */
        onDestroy: function () {
            var eventActions;
            for (var eventType in this.events) {
                eventActions = this.events[eventType].getActions();
                eventActions.forEach(function (eventAction) {
                    this.actions[eventAction].destroy();
                }, this);
            }

            this.mvc.model.unsubscribePropertyChanged(this.modelPropertyChanged);
            this.modelChanged = null;
        },
        getParentObject: function(){
            var parent=this.mvc._parent;
            while (!parent.model.objectId && parent && this._parent) {
                parent = parent._parent;
            }
            return parent.model.objectId;
        },

        buildPath: function () {
            var parents = [this.mvc.model.getId()],
                parent = this.mvc._parent;

            if (parent === null || (this.mvc.model.ClassName === "WindowModel" && parent === null)) {
                return '/';
            }

            while (parent !== null && parent.model.getId !== undefined && parent.model.ClassName != "WindowModel") {
                parents.push(parent.model.getId());
                parent = parent._parent;
            }

            return '/' + parents.reverse().join('/') + '/';
        },

        buildFullPath: function () {
            var parents = [this.mvc.model.getId()],
                parent = this.mvc._parent;

            if (parent === null) {
                return '/';
            }

            while (parent !== null && parent.model.getId !== undefined && parent.model.ClassName != "SchemeModel") {
                parents.push(parent.model.getId());
                parent = parent._parent;
            }

            return '/' + parents.reverse().join('/') + '/';
        },

        /**
         *  @method Fired when a new Agility object is added to the object's container.
         */
        onChildAdded: function (child) {

        },

        /**
         *  @method Fired with an Agility object is removed from the object's container.
         */
        onChildRemoved: function (child) {

        },

        /*
        * @method Fired when control added to parent container
        */
        onAddedToParent: function () {

            //хитрый ход: запускаем onAddedToDOM child-элементов, 
            //только когда корневое окно appendChild к Scheme,
            //а контейнеры уже запустят onAddedToDOM  у своих child
            //т.е onAddedToParent - снизу-вверх, onAddedToDOM - сверху-вниз
            var parents = this.mvc.parent().view.$().parents('html');
            if (parents.length > 0) {
                this.mvc.controller.onOneTimeConnectionsProcess()
                this.mvc.controller.onAddedToDOM();
            }
            var parentController = this.mvc.parent().controller;
            if (parentController) {
                 //subscribe on parent events here
                 if (parentController.subscribeActualVisibleChanged) {
                     parentController.subscribeActualVisibleChanged(
                         this.onParentActualVisibleChanged, this);
                 }
 
                 if (parentController.subscribeActualEnabledChanged) {
                     parentController.subscribeActualEnabledChanged(
                         this.onParentActualEnabledChanged, this);
                 }
 
                 if (parentController.subscribeAbsoluteXChanged) {
                     parentController.subscribeAbsoluteXChanged(
                         this.onParentAbsoluteXChanged, this);
                 }
 
                 if (parentController.subscribeAbsoluteYChanged) {
                     parentController.subscribeAbsoluteYChanged(
                         this.onParentAbsoluteYChanged, this);
                 }
 
                 if (parentController.subscribeActualWidthChanged && this.widthType!=Enums.SizeType.Pixel) {
                     parentController.subscribeActualWidthChanged(
                         this.onParentActualWidthChanged, this);
                 }
 
                 if (parentController.subscribeActualHeightChanged && this.heightType!=Enums.SizeType.Pixel) {
                     parentController.subscribeActualHeightChanged(
                         this.onParentActualHeightChanged, this);
                 }
             }

                this.applyWidth();
                this.applyHeight();
        },

        /*
        * @method Fired just before control removed from its container
        */
        onRemovingFromParent: function () {
            var parentController = this.mvc.parent().controller;

            //unsubscribe from parent events here
            if (parentController) {
                if (parentController.subscribeActualVisibleChanged) {
                    parentController.subscribeActualVisibleChanged(
                        this.onParentActualVisibleChanged, this);
                }

                if (parentController.subscribeActualEnabledChanged) {
                    parentController.subscribeActualEnabledChanged(
                        this.onParentActualEnabledChanged, this);
                }

                if (parentController.unsubscribeAbsoluteXChanged) {
                    parentController.unsubscribeAbsoluteXChanged(
                        this.onParentAbsoluteXChanged);
                }

                if (parentController.unsubscribeAbsoluteYChanged) {
                    parentController.unsubscribeAbsoluteYChanged(
                        this.onParentAbsoluteYChanged);
                }

                if (parentController.unsubscribeActualWidthChanged) {
                    parentController.unsubscribeActualWidthChanged(
                        this.onParentActualWidthChanged);
                }

                if (parentController.unsubscribeActualHeightChanged) {
                    parentController.unsubscribeActualHeightChanged(
                        this.onParentActualHeightChanged);
                }
            }
        },

        onAddedToDOM: function () {
            if (this.mvc.model.isPropertyExists(Enums.ParameterRoles.TOOL_TIP)) {
                this.mvc.view.$().attr('title', "");
                this.mvc.view.$().tooltip({
                    track: true,
                    content: ""
                });
                this.modelChanged[Enums.ParameterRoles.TOOL_TIP] = this.onToolTipChanged.bind(this);
                this.onToolTipChanged(this.mvc.model.get(Enums.ParameterRoles.TOOL_TIP));
            }
            if (this.mvc.model.get(Enums.ParameterRoles.OPACITY) == 0) {
                this.mvc.view.$().css('pointer-events', 'none');
            }

            _.forOwn(this.actions, function (action) {
                action.subscribePropertyChanged(function (ControlController, action) {
                    return function (event) {
                        // ControlController.buildPath()
                        var path = "/" + action.id + '/' + event.property;
                        this.model.overrideConnectionsOperation(path, action.constants.operation);
                        this.model.updateTypeConnection(path, this);
                        this.model.restoreConnectionsOperation(action.constants.operation);
                    }
                }(this, action), this.mvc);
            }, this);

            _.forOwn(this.events, function (ev) {
                ev.subscribePropertyChanged(function (ControlController, ev) {
                    return function (event) {
                        // ControlController.buildPath()
                        var path = "/" + ev.type + '/' + event.property;
                        this.model.updateTypeConnection(path, this);
                    }
                }(this, ev), this.mvc);
            }, this);
            if (this.mvc.model.isPropertyExists(Enums.ParameterRoles.SKIN) && this.mvc.model.get(Enums.ParameterRoles.SKIN)) {
                var value = this.mvc.model.get(Enums.ParameterRoles.SKIN);
                var resourceUrl;
                if (!value) {
                    resourceUrl = null;
                }
                else {
                    resourceUrl = this.mvc.model.resourceManager.resolve(value);
                }
                var embed = document.createElement('embed');
                embed.setAttribute("type", "image/svg+xml");
                embed.setAttribute("src", resourceUrl);
                var attrs = this.mvc.view.$()[0].attributes;
                for (var i = 0; i < attrs.length; i++) {
                    embed.setAttribute(attrs[i].name, attrs[i].value);
                }
                embed.style.border = '';
                embed.style.background = '';
                this.mvc.view.$().replaceWith(embed);
                this.mvc.view.$root = $(embed);
                this.modelChanged[Enums.ParameterRoles.BORDER_COLOR] = this.onSvgModelChanged.bind(this, Enums.ParameterRoles.BORDER_COLOR + ":values", Appearence.color.color2Matrix);
                this.modelChanged[Enums.ParameterRoles.BORDER_STYLE] = this.onSvgBorderChanged.bind(this);
                this.modelChanged[Enums.ParameterRoles.BORDER_THIKNESS] = this.onSvgBorderChanged.bind(this);
                this.modelChanged[Enums.ParameterRoles.BACKGROUND_COLOR] = this.onSvgModelChanged.bind(this, Enums.ParameterRoles.BACKGROUND_COLOR+":values", Appearence.color.color2Matrix);
                this.modelChanged[Enums.ParameterRoles.IS_ENABLED] = this.onSvgIsEnabledChanged.bind(this);
                //загрузка
                var svgContent = null;
                try {
                    svgContent = embed.getSVGDocument();
                }
                catch (ex) {
                    console.error(ex.message || ex);
                }

                //VictorM svg элементы подгружаются параллельно с подгрузкой основного документа. 
                //Поэтому может получиться, что к моменту вызова addSvgEventListener некоторые элементы уже загружены и по ним уже не вызовится onload

                if (svgContent != null) {
                    this.addSvgEventListener(embed);
                }
                else {
                    embed.addEventListener('load', function () {
                        this.loaded = true;
                        this.addSvgEventListener(embed);
                        this.fireAbsoluteXChanged();
                        this.fireAbsoluteYChanged();
                        this.mvc.view.$().tooltip({
                            track: true,
                            content: ""
                        });
                        this._resetSVG();
                        this.triggerHandler.init(this);
                        //    this.triggerHandler.applyAll();
                    }.bind(this), false);
                }
            }
            else {
                //применяем триггеры, когда тип и модель обработаны + 
                //построены и отработали все связи
                this.triggerHandler.init(this);
                //    this.triggerHandler.applyAll();
            }

            this.askAddPermissions(this.doNothing, this.mvc.model.setIsEnabled.bind(this.mvc.model, false), true);
            
        },
        grayScaleControl: function(){
            this.controlColorChanged(Appearence.color.makeDisabled(this.mvc.model.getBackgroundColor()), Appearence.color.makeDisabled(this.mvc.model.get(Enums.ParameterRoles.BORDER_COLOR)));
        },
        controlColorChanged: function(bg, border){
            Appearence.background.apply(this.mvc.view.$(), bg);
            border = {
                penType: this.mvc.model.get(Enums.ParameterRoles.BORDER_STYLE),
                size: this.mvc.model.get(Enums.ParameterRoles.BORDER_THIKNESS),
                color: border
            };
            Appearence.border.applyBorder(this.mvc.view.$(), border);
        },

        onSvgIsEnabledChanged: function(){
            if (!this.actualEnabled()) {
                this.mvc.view.$().attr('disabled', 'true');
                this.mvc.view.$().addClass('disabled');
                this.modelChanged[Enums.ParameterRoles.BACKGROUND_COLOR](Appearence.color.makeDisabled(this.mvc.model.getBackgroundColor()));
                this.modelChanged[Enums.ParameterRoles.BORDER_COLOR](Appearence.color.makeDisabled(this.mvc.model.get(Enums.ParameterRoles.BORDER_COLOR)));
            } else {
                this.mvc.view.$().removeAttr('disabled');
                this.mvc.view.$().removeClass('disabled');
                this.modelChanged[Enums.ParameterRoles.BACKGROUND_COLOR](this.mvc.model.getBackgroundColor());
                this.modelChanged[Enums.ParameterRoles.BORDER_COLOR](this.mvc.model.get(Enums.ParameterRoles.BORDER_COLOR));
            }
        },

        onSvgModelChanged: function (property, converter, value) {
            var newvalue;
            if (converter) {
                newvalue = converter(value);
            }
            else {
                newvalue = value;
            }
            this.onCustomParameterChanged(property, newvalue);
        },

        onCustomParameterChanged: function (path, value) {
            if (this.loaded) {
                this._setSvgParameter(path, value);
            }
        },
        getSVGElementById: function(id){
            var element = this.mvc.view.$()[0].getSVGDocument().getElementById(id);
            if (!element) {
                Error.warn(String.format("У контрола {0} {1} не найден элемент с id {2}", this.mvc.model.getRole(), this.mvc.model.getId(), id));
            }
            return element;

        },
        getSVGAttributeFromElement: function(element, attribute){
            if (element) {
                var value = element.getAttribute(attribute);
                if (!value) {
                    Error.warn(String.format("У контрола {0} {1} в svg параметре Образ не найден атрибут {2}", this.mvc.model.getRole(), this.mvc.model.getId(), attribute));
                }
                return value;
            }
        },
        _setSvgParameter: function (path, value) {
            var svgRoot = this._getSvgRoot();
            if (svgRoot === null) {
                console.warn('No svg root');
                return;
            }

            var tokenizedPath = path.split(':');
            var selector = this._getSelector(tokenizedPath[0]);

            if (tokenizedPath[1] === 'Content') {
                if (Utilites.isWebkit()) {
                    $(selector, svgRoot)[0].textContent = value;
                } else {
                    $(selector, svgRoot).html(value);
                }

            } else {
                this._getSetParameterMethod(tokenizedPath[2]).call($(selector, svgRoot), tokenizedPath[1], value);
            }
        },

        _getSelector: function (name) {
            if (this._isClassParameter(name)) {
                return name;
            } else {
                return '#' + name;
            }
        },
        _getSetParameterMethod: function (name) {
            if (name === null || name === undefined || name !== 'style') {
                return $.fn.attr;
            } else {
                return $.fn.css;
            }
        },

        _isClassParameter: function (name) {
            return name.indexOf('.') === 0;
        },

        _getSvgRoot: function () {
            try {
                if (this.mvc.view.$()[0].getSVGDocument() !== null && this.mvc.view.$()[0].getSVGDocument().readyState === 'complete') {
                    return this.mvc.view.$()[0].getSVGDocument().documentElement;
                } else {
                    return null;
                }
            } catch (e) {
                Error.debug(String.format('Reason: {0} \n Stack trace: \n {1}', e.message || e, e.stack));
                return null;
            }
        },

        addSvgEventListener: function (embed) {
            var self = this;
            var elem;
            if (embed.getSVGDocument().getElementById("body")) {
                elem = $(embed.getSVGDocument().getElementById("body"));
            } else {
                elem = $(embed.getSVGDocument().querySelector('svg'));
            }
            elem.mousedown(function (event) { self.onMouseDown(event); });
            elem.mouseup(function (event) { self.onMouseUp(event); });
            elem.mouseover(function (event) { self.onMouseMove(event); });
            elem.mouseenter(function (event) { self.onMouseEnter(event); });
            elem.mouseleave(function (event) { self.onMouseLeave(event); });

            elem.click(function (event) { self.onClick(event); });
            elem.dblclick(function (event) { self.onDoubleClick(event); });

            elem.focusin(function (event) { self.onFocus(event); });
            elem.focusout(function (event) { self.onFocusLost(event); });

            elem.keypress(function (event) { return self.onKeyPress(event); });
            elem.keydown(function (event) { return self.onKeyDown(event); });
            elem.keyup(function (event) { return self.onKeyUp(event); });
        },

        _resetSVG: function () {
            var data = this.mvc.model._data;
            for (dataKey in data) {
                this.modelPropertyChangedInternal({property: dataKey});
            }
            data = null;
        },

        onModelInit: function () {
            this.validator = new Validator(this.mvc.model.getId());
            mergeActions.call(this, this.mvc.model.actions);
            //когда все события собраны и все акшены смержены - создаем события
            this._addEvents();
            this.triggerHandler.add(this.mvc.model.triggers, this);
        },

        _addEvents: function () {
            if (!this.mvc.model.events) {
                return;
            }

            _.forOwn(this.mvc.model.events, function (event, eventType) {
                this.events[eventType] = new Event(event);
            }, this);

        },

        onAfterModelInit: function () {
            this.renderBorder();
        },

        onOneTimeConnectionsProcess: function () {

        },

        addConnectionsForParameter: function (parameterName) {
            return ParameterConnections.process(parameterName, this.mvc.model.getProperty(parameterName), this.mvc);
        },

        //
        // rendering
        //

        onRender: function () {

            this.mvc.view.$().attr('data-control-role', this.mvc.model.getRole());
            this.renderId();
            this.renderBorder();
            this.renderOverflow();
            this.renderPosition();
        },

        renderId: function () {
            this.mvc.view.$().attr('data-control-id', this.mvc.model.getId());
        },

        renderBorder: function () {
            var borderWidth = this.mvc.model.get(Enums.ParameterRoles.BORDER_THIKNESS);

            if (borderWidth) {
                Appearence.border.applyBoxSizing(this.mvc.view.$(), 'border-box');
            }
        },

        renderOverflow: function () {
            this.mvc.view.$().css('overflow', 'hidden');
        },

        renderPosition: function () {
            this.mvc.view.$().css('position', 'absolute');
        },

        onParentActualVisibleChanged: function (newValue) {
            if (!window.initialized)
                return;
            this.fireActualVisibleChanged(newValue);
        },

        onParentActualEnabledChanged: function (newValue) {
            if (!window.initialized)
                return;
            this.applyIsEnabled();
            if (this.mvc.model.getIsEnabled()) {
                this.fireActualEnabledChanged(newValue);
            }
        },

        onParentAbsoluteXChanged: function () {
            if (!window.initialized)
                return;
            this.fireAbsoluteXChanged();
        },

        onParentAbsoluteYChanged: function () {
            if (!window.initialized)
                return;
            this.fireAbsoluteYChanged();
        },

        onParentActualWidthChanged: function () {
            if (!window.initialized)
                return;
            if (Utilites.isPercents(this.mvc.view.$()[0].style.width)) {
                this.fireActualWidthChanged();

                // TODO Check this
                //check for anchor
                //if (this.mvc.view.$().css('left') === 'auto') {
                //    this.fireAbsoluteXChanged();
                //}
            }
        },

        onParentActualHeightChanged: function () {
            if (!window.initialized)
                return;
            if (Utilites.isPercents(this.mvc.view.$()[0].style.height)) {
                this.fireActualHeightChanged();

                // TODO Check this
                ////check for anchor
                //if (this.mvc.view.$().css('top') === 'auto') {
                //    this.fireAbsoluteYChanged();
                //}
            }
        },

        clientId: function () {
            var p = this.mvc.parent();

            var clientId = this.mvc.model.getId();

            if (p && p.controller && typeof p.controller.clientId == 'function') {
                clientId = p.controller.clientId() + '_' + clientId;
            }

            return clientId;
        },

        getVariablesSource: function (id) {
            return this.variablesSource.getVMById(id);
        },
        getControlLeftPx: function () {
            var value = this.mvc.view.$().css('left');
            if (value === 'auto') {
                this.mvc.view.$()[0].offsetLeft;
            }
            else {
                return parseInt(value, 10);
            }
        },

        setControlLeft: function (value) {
            var oldValue = this.mvc.view.$().css('left');

            if (value !== oldValue) {
                this.mvc.view.$().css('left', value);
                this.fireAbsoluteXChanged();
            }
        },

        getControlTopPx: function () {
            return parseInt(this.mvc.view.$().css('top'), 10);
        },

        setControlTop: function (value) {
            var oldValue = this.mvc.view.$().css('top');

            if (value !== oldValue) {
                this.mvc.view.$().css('top', value);
                this.fireAbsoluteYChanged();
            }
        },

        setControlRight: function (value) {
            var oldValue = this.mvc.view.$().css('right');

            if (value !== oldValue) {
                this.mvc.view.$().css('right', value);
                this.fireAbsoluteXChanged();
            }
        },

        getControlBottomPx: function () {
            return parseInt(this.mvc.view.$().css('bottom'), 10);
        },

        setControlBottom: function (value) {
            var oldValue = this.mvc.view.$().css('bottom');

            if (value !== oldValue) {
                this.mvc.view.$().css('bottom', value);
                this.fireAbsoluteYChanged();
            }
        },

        setControlWidth: function (value) {
            var oldValue = this.mvc.view.$().css('width');

            if (value !== oldValue) {
                this.mvc.view.$().css('width', value);
                this.fireActualWidthChanged();
            }
        },

        setControlHeight: function (value) {
            var oldValue = this.mvc.view.$().css('height');

            if (value !== oldValue) {
                this.mvc.view.$().css('height', value);
                this.fireActualHeightChanged();
            }
        },

        getControlWidthPx: function () {
            return this.mvc.view.$().outerWidth();
        },

        getControlInnerWidth: function () {
            return this.mvc.view.$().innerWidth();
        },

        getControlHeightPx: function () {
            return this.mvc.view.$().outerHeight();
        },

        getControlInnerHeight: function () {
            return this.mvc.view.$().innerHeight();
        },


        // TODO Check anchors, logc has been changed
        setLeftAnchor: function (anchor) {

            var oldLeft = this.mvc.view.$().css('left');
            var oldRight = this.mvc.view.$().css('right');

            if (oldLeft !== anchor || oldRight !== 'auto') {
                this.mvc.view.$().css({ left: anchor, right: 'auto' });
                this.fireAbsoluteXChanged();
            }
        },

        setRightAnchor: function (anchor) {
            var oldLeft = this.mvc.view.$().css('left');
            var oldRight = this.mvc.view.$().css('right');

            if (oldLeft !== 'auto' || oldRight !== anchor) {

                this.mvc.view.$().css({ left: 'auto', right: anchor });
                this.fireAbsoluteXChanged();
            }
        },

        setTopAnchor: function (anchor) {
            var oldTop = this.mvc.view.$().css('top');
            var oldBottom = this.mvc.view.$().css('bottom');

            if (oldTop !== anchor || oldBottom !== 'auto') {
                this.mvc.view.$().css({ top: anchor, bottom: 'auto' });
                this.fireAbsoluteYChanged();
            }
        },

        setBottomAnchor: function (anchor) {
            var oldTop = this.mvc.view.$().css('top');
            var oldBottom = this.mvc.view.$().css('bottom');

            if (oldTop !== 'auto' || oldBottom !== anchor) {
                this.mvc.view.$().css({ top: 'auto', bottom: anchor });
                this.fireAbsoluteYChanged();
            }
        },

        actualVisible: function () {
            var visible = this.mvc.model.getIsVisible();

            if (!visible) {
                return false;
            }

            return this.actualParentVisible();
        },

        actualParentVisible: function () {
            var p = this.mvc.parent();
            if (!p) {
                //control is not on page
                return false;
            }

            var parentController = p.controller;
            if (!parentController || !parentController.actualVisible) {
                //may be parent is not regular control, so assume it is visible
                return true;
            }

            return parentController.actualVisible();
        },

        actualEnabled: function () {
            var enabled = this.mvc.model.getIsEnabled();

            if (enabled === false) {
                return false;
            }

            return this.actualParentEnabled();
        },

        actualParentEnabled: function () {
            var p = this.mvc.parent();
            if (!p) {
                //control is not on page
                return true;
            }

            var parentController = p.controller;
            if (!parentController || !parentController.actualEnabled) {
                //may be parent is not regular control, so assume it is enabled
                return true;
            }

            return parentController.actualEnabled();
        },

        applyShadow: function () {

            var shadow = {
                size: this.mvc.model.get(Enums.ParameterRoles.SHADOW_SIZE),
                color: this.getMixedShadowColor()
            };

            Appearence.shadow.applyBoxShadow(this.mvc.view.$(), shadow);
        },


        getMixedShadowColor: function () {
            var shadowColor = this.mvc.model.getShadowColor();
            var bgBrush = this.mvc.model.getBackgroundColor();

            if (this._isARGB(shadowColor) && this._isARGB(bgBrush)) {
                var shadowObj = Appearence.color.ARGBtoObj(shadowColor);
                var bgBrushObj = Appearence.color.ARGBtoObj(bgBrush);
                if (shadowObj && bgBrushObj) {
                    if (shadowObj.a > bgBrushObj.a)
                        shadowObj.a = bgBrushObj.a;
                    return Appearence.color.objToARGB(shadowObj);
                }
            }

            return Appearence.color.toCssColor(shadowColor);

        },

        _isARGB: function (color) {
            var brushType = Appearence.background.getBrushType(color);
            switch (brushType) {
                case Enums.BrushType.RADIAL_GRADIENT:
                case Enums.BrushType.LINEAR_GRADIENT:
                    return false;
                default:
                    return true;
            }
        },

        applyWidth: function () {
            //abs is for line support
            var width = Math.abs(this.mvc.model.getWidth()),
                widthUnits = this.mvc.model.getWidthUnits();

            var displayValue;
            if (this.mvc.parent()
                && this.mvc.parent().controller) {
                if (widthUnits == Enums.SizeType.Relative) {
                    var parentWidth = Math.abs(this.mvc.parent().model.getWidth()) || 1;
                    displayValue = (100 * width / parentWidth);
                }
                else {
                    displayValue = width;
                }
            } else {
                displayValue = width;
                widthUnits = Enums.SizeType.Pixel;
            }
            this.width = displayValue;
            this.setControlWidth(displayValue + Utilites.getMeasureSuffix(widthUnits));
        },

        applyHeight: function () {

            //abs is for line support
            var height = Math.abs(this.mvc.model.getHeight());
            var heightUnits = this.mvc.model.getHeightUnits();

            var displayValue;
            if (this.mvc.parent()
                && this.mvc.parent().controller) {
                if (heightUnits == Enums.SizeType.Relative) {//means that parent is not document
                    var parentHeight = Math.abs(this.mvc.parent().model.getHeight()) || 1;
                    displayValue = (100 * height / parentHeight);
                }
                else {
                    displayValue = height;
                }
            } else {
                displayValue = height;
                heightUnits = Enums.SizeType.Pixel;
            }
            this.setControlHeight(displayValue + Utilites.getMeasureSuffix(heightUnits));
        },

        applyBorder: function () {

            this.setupBorderTo(this.mvc.view.$());
        },

        applyIsEnabled: function () {
            if (!this.actualEnabled()) {
                this.mvc.view.$().attr('disabled', 'true');
                this.mvc.view.$().addClass('disabled');
                this.grayScaleControl();
                this.mvc.view.$().css('pointer-events', 'none');

            } else {
                this.mvc.view.$().removeAttr('disabled');
                this.mvc.view.$().removeClass('disabled');
                this.controlColorChanged(this.mvc.model.getBackgroundColor(), this.mvc.model.getBorderColor());
                this.mvc.view.$().css('pointer-events', '');
            }
        },

        applyFlash: function () {
            var isFlashEnabled = this.mvc.model.get(Enums.ParameterRoles.FLASH);
            if (isFlashEnabled) {
                this.mvc.view.$().addClass(this.flashClass);
            }
            else {
                this.mvc.view.$().removeClass(this.flashClass);
            }
        },

        /*
        * @method setupBorder used to setup border for element (exactly same as for control)
        * @param el {jQuery} target element(s)
        */
        setupBorderTo: function (el) {
            var border = {
                penType: this.mvc.model.get(Enums.ParameterRoles.BORDER_STYLE),
                size: this.mvc.model.get(Enums.ParameterRoles.BORDER_THIKNESS),
                color: this.mvc.model.get(Enums.ParameterRoles.BORDER_COLOR)
            };

            Appearence.border.applyBorder(el, border);
        },

        find: function (path) {
            var children = this.mvc._container.children;
            for (var i in children) {
                if (children[i].model.getId() === path) {
                    return children[i];
                }
            }
        },

        /*
        * we do not need to fire property changed for these properties while fireAll
        * because 
        */
        skipFireAllProperty: function (property) {
            switch (property) {
                case Enums.ParameterRoles.X_UNITS:
                case Enums.ParameterRoles.Y_UNITS:
                case Enums.ParameterRoles.WIDTH_UNITS:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                case Enums.ParameterRoles.SHADOW_COLOR:
                case Enums.ParameterRoles.BORDER_COLOR:
                case Enums.ParameterRoles.BORDER_STYLE:
                    return true;
                default:
                    return false;
            }
        },

        //
        //property changed handlers
        //
        modelPropertyChanged: function (event) {
            if (event.property) {
                if ((event.newValue) && ((event.newValue.value == null) || (event.newValue.value == NaN) || (event.newValue.value == undefined))) {
                    Error.info(String.format('{0} {1} параметру {2} пришло значение {3} от параметра {4}', this.mvc.model.getRole(), this.mvc.model.getId(), event.property, String(event.newValue.value), event.newValue.itemId));
                }
                this.modelPropertyChangedInternal(event);
            }
            else {
                var parameters = this.mvc.model.getParameters();
                var i = parameters.length;

                while (i--) {

                    if (this.skipFireAllProperty(parameters[i].role)) {
                        continue;
                    }

                    this.modelPropertyChangedInternal({
                        target: this.mvc.model,
                        property: parameters[i].role
                    });
                }
            }
        },

        modelPropertyChangedInternal: function (event) {
            // обновляем связи типа, если изменились значения свойств экземпляра.
            // Связи экземпляра обновит тип, в котором этот экземпляр находится.
            // Поэтому путь к свойству строится без учета родителей
            var path = '/' + event.property;
            this.mvc.model.getType()._connectionsHandler.onchange(path, this.mvc, event.oldValue);
            // TODO сделать из этого Object, чтобы не перебирать switch'ем. Все-таки время доступа к свойству в объекте О(1)
            var value = this.mvc.model.get(event.property);
            if (this.modelChanged[event.property]) {
                this.modelChanged[event.property](value);
            }
        },

        onWidthChanged: function (value) {
            this.applyWidth();
        },

        onWidthUnitsChanged: function (value) {
            this.applyWidth();
        },

        onHeightChanged: function (value) {
            this.applyHeight();
        },

        onHeightUnitsChanged: function (value) {
            this.applyHeight();
        },

        onZChanged: function (value) {
            this.mvc.view.$().css('z-index', value);
        },

        onAngleChanged: function (value) {
            var scale = {
                scaleX: this.mvc.model.get(Enums.ParameterRoles.SCALE_X),
                scaleY: this.mvc.model.get(Enums.ParameterRoles.SCALE_Y),
                angle: value
            };

            Appearence.position.scale(this.mvc.view.$(), scale);
        },

        onScaleXChanged: function (value) {

            var scale = {
                scaleX: value,
                scaleY: this.mvc.model.get(Enums.ParameterRoles.SCALE_Y),
                angle: this.mvc.model.get(Enums.ParameterRoles.ANGLE)
            };

            Appearence.position.scale(this.mvc.view.$(), scale);
        },

        onScaleYChanged: function (value) {

            var scale = {
                scaleX: this.mvc.model.get(Enums.ParameterRoles.SCALE_X),
                scaleY: value,
                angle: this.mvc.model.get(Enums.ParameterRoles.ANGLE)
            };

            Appearence.position.scale(this.mvc.view.$(), scale);
        },

        onTabIndexChanged: function (value) {
            //from ParameterProcessor: TabIndex cannot begin with zero, so add one
            if (value === 0) {
                value++;
            }
            this.mvc.view.$().attr('tabindex', value);
        },

        onColspanChanged: function (value) {
        },

        onRowspanChanged: function (value) {
        },

        onToolTipChanged: function (value) {
            this.mvc.view.$().tooltip("option", "content", value);
        },

        onIsVisibleChanged: function (value) {
            if (value.toLowerCase) {
                value = value.toLowerCase();
            }

            var cssValue = value.toString() == 'false' ? 'hidden' : '';
            this.mvc.view.$().css('visibility', cssValue);
            this.fireActualVisibleChanged(value);
        },

        onIsEnabledChanged: function (value) {
            this.applyIsEnabled();
            this.fireActualEnabledChanged(this.actualEnabled());
        },

        onFlashChanged: function (value) {
            this.applyFlash();
        },

        onOpacityChanged: function (value) {
            if (value == 0) {
                this.mvc.view.$().css('pointer-events', 'none')
            }
            else {
                this.mvc.view.$().css('pointer-events', '');
            }
            this.mvc.view.$().css('opacity', value / 100);
        },

        onShadowSizeChanged: function (value) {
            this.applyShadow();
        },

        onShadowColorChanged: function (value) {
            this.applyShadow();
        },

        onBackgroundColorChanged: function (value) {
            this.onShadowColorChanged();
            this.onShadowSizeChanged();
            Appearence.background.apply(this.mvc.view.$(), value);
            this.applyIsEnabled();
        },

        onBackgroundTileChanged: function (value) {
            Appearence.background.applyBackgroundTile(this.mvc.view.$(), value);
        },

        onSkinChanged: function (value) {
            //динамизации образа нет 12965
        },

        onResourceChanged: function (value) {
            //get resource url
            var resourceUrl;

            if (!value) {
                resourceUrl = null;
            }
            else {
                resourceUrl = this.mvc.model.resourceManager.resolve(value);
            }

            //asuming that value is url to resource
            Appearence.background.applyBgImage(this.mvc.view.$(), resourceUrl);
        },

        onSvgBorderChanged: function (value){ //будем пока что считать что у svg нет границ
            return;
        },

        onCornerRadiusChanged: function (value) {
            // Нам реально надо в половину меньше, чем пришло из редактора.
            // Связано с особенностями отображения в WPF и HTML.
            value = value / 2;
            Appearence.border.applyCornerRadius(this.mvc.view.$(), { radius: value, units: '%' });
        },

        onThemeChanged: function (value) {
            //TODO:
        },

        //
        // Events
        //

        attachEvents: function() {
            var self = this;
            this.mvc.model.subscribePropertyChanging(this.onPropertyChanging, this);
            this.mvc.model.subscribeOnInit(this.onModelInit, this);
            this.mvc.model.subscribeOnAfterInit(this.onAfterModelInit, this);
            this.isMouseDownFired = false;
            this.mvc.view.$().mousedown(function(event) { self.onMouseDown(event); });
            this.mvc.view.$().on("mouseup touchend touchcancel",
                function(event) {
                    self.onMouseUp(event);
                });
            this.mvc.view.$().mouseover(function(event) { self.onMouseMove(event); });
            this.mvc.view.$().mouseenter(function(event) { self.onMouseEnter(event); });
            this.mvc.view.$().mouseleave(function(event) { self.onMouseLeave(event); });
            this.mvc.view.$().click(function(event) { self.onClick(event); });
            this.mvc.view.$().dblclick(function(event) { self.onDoubleClick(event); });
            this.mvc.view.$().focusin(function(event) { self.onFocus(event); });
            this.mvc.view.$().focusout(function(event) { self.onFocusLost(event); });

            this.mvc.view.$().keypress(function(event) { return self.onKeyPress(event); });
            this.mvc.view.$().keydown(function(event) { return self.onKeyDown(event); });
            this.mvc.view.$().keyup(function(event) { return self.onKeyUp(event); });

            this.mvc.view.$().contextmenu(function(event) { return self.onContextMenu(event); });
            this.mvc.view.$().on('dragstart', function(event) { return self.onDragStart(event); });

        },

        onMouseDown: function (event) {
            this.handleEvent(Enums.EventType.mouseDown, event);
        },

        onMouseUp: function (event) {
            this.handleEvent(Enums.EventType.mouseUp, event);
        },

        onMouseMove: function (event) {
            this.handleEvent(Enums.EventType.mouseMove, event);
        },

        onMouseEnter: function (event) {
            this.handleEvent(Enums.EventType.mouseEnter, event);
        },

        onMouseLeave: function (event) {
            this.handleEvent(Enums.EventType.mouseLeave, event);
        },

        onClick: function (event) {
            this.handleEvent(Enums.EventType.click, event);
        },

        onDoubleClick: function (event) {
            this.handleEvent(Enums.EventType.doubleClick, event);
        },

        onFocus: function (event) {
            this.handleEvent(Enums.EventType.focus, event);
        },

        onFocusLost: function (event) {
            this.handleEvent(Enums.EventType.lostFocus, event);
        },

        onKeyPress: function (event) {
            this.handleEvent(Enums.EventType.keyPress, event);
            return true;
        },

        onKeyDown: function (event) {
            this.handleEvent(Enums.EventType.keyDown, event);
            return true;
        },

        onKeyUp: function (event) {
            this.handleEvent(Enums.EventType.keyUp, event);
            return true;
        },

        onContextMenu: function (event) {
            event.stopPropagation();
            event.preventDefault();
            this.handleEvent(Enums.EventType.contextMenu, event);
            return true;
        },

        onDragStart: function (event) {
            event.stopPropagation();
            event.preventDefault();
            return true;
        },

        handleEvent: function (eventType, event, force) {
            if (!(this.actualEnabled() || force === true)) {
                return when.resolve();
            }

            if (this.events[eventType] === undefined) {
                return when.resolve();
            }
            if (!force && this.events[eventType].getIsTriggerOnlyEvent() == false) {
                //если событие вызвало действие пользователя, 
                //а не внешний вызов (CallEventAction)
                //и оно не только, т.е есть параметры
                this.events[eventType].updateParameters(event, this);
            }
            this.askEventPermissions(eventType);

            //TODO event.stopPropagation();
            return when.resolve();
        },
        askEventPermissions: function(eventType) {
             var eventActions = this.events[eventType].getActions();
            if (eventActions.length) {               
                this.askPermissions(this.doActions.bind(this,eventActions), this.doNothing, false);
            }
        },
        askConfirmation: function() {      
            this.askPermissions(this.allowAct.bind(this), this.denyAct.bind(this), false)
        },

        askAddPermissions: function (function_allow, function_deny, silent) {
            this.askPermissions(function_allow, function_deny, silent);
        },

        askPermissions: function(function_allow, function_deny, silent) {
            var result = {};
            result.allow_action = function_allow;
            result.deny_action = function_deny;
            result.allow = false;
            result.deny = false;
            result.writelog = false;
            result.confirmation = false;
            result.silent = silent; //true
            this.check(result);
        },
        allowAct: function(value) {
            return when.resolve();
        },
        denyAct: function() {
            return when.resolve();
        },
        doActions: function(eventActions){
            for (var i = 0; i < eventActions.length; i++) {
                this.actions[eventActions[i]].execute(this);
            }
            return when.resolve();
        },
        doNothing: function(){
            return when.resolve();
        },
        check: function (result) {
            if (PermissionChecker.needCheck()) {
                PermissionChecker.check("Control", this, result)
                if (result.allow) {
                    if (result.confirmation && (!result.silent)) {
                        dialog = $("<div>" + L10n.get("actions.Control") + "</div>").dialog({
                            autoOpen: true,
                            modal: true,
                            title: L10n.get("messages.confirmAction"),
                            resizable: false,
                            dialogClass: "confirm",
                            collapseEnabled: false,
                            buttons: {
                                "confirm": {
                                    text: L10n.get("controls.okText"),
                                    class: 'confirmbtn btn',
                                    click: function () {
                                        result.allow_action();
                                        $(this).dialog("close");
                                    }

                                },
                                "cancel": {
                                    text: L10n.get("controls.cancelText"),
                                    class: 'confirmbtn btn',
                                    click: function () {
                                        result.deny_action();
                                        $(this).dialog("close");
                                    }
                                }
                            }
                        });
                    } else
                        result.allow_action();
                } else
                    result.deny_action();
            }
            else {
                result.allow_action(); //если пользователей нет, то считается что всё можно
            }
        },
        onPropertyChanging: function (event) {
            this.validateProperty(event.property, event.newProperty);
        },

        validateProperty: function (propertyName, newProperty) {

        },

        callMethod: function (methodName, params) {
            if (this[methodName]) {
                return this[methodName].apply(this, params);
            }

            return when.resolve();
        },

        //
        // custom events
        //
        subscribeActualVisibleChanged: function (handler, context) {
            this.eventTarget.addListener(ACTUAL_VISIBLE_CHANGED, handler, context);
        },

        unscubscribeActualVisibleChanged: function (handler) {
            this.eventTarget.removeListener(ACTUAL_VISIBLE_CHANGED, handler);
        },
        /*
        * @method fireActualVisibleChanged fires event "actual visibility of control changed"
        * (even if parent visiblility was changed)
        */
        fireActualVisibleChanged: function (newValue) {
            this.eventTarget.fire({ type: ACTUAL_VISIBLE_CHANGED, target: this, value: newValue });
        },
        //----------------------------------------------------------------------
        subscribeActualEnabledChanged: function (handler, context) {
            this.eventTarget.addListener(ACTUAL_ENABLED_CHANGED, handler, context);
        },

        unscubscribeActualEnabledChanged: function (handler) {
            this.eventTarget.removeListener(ACTUAL_ENABLED_CHANGED, handler);
        },

        /*
        * @method fireActualEnabledChanged fires event "actual enabled status of control changed"
        * (even if parent enabled was changed)
        */
        fireActualEnabledChanged: function (newValue) {
            this.eventTarget.fire({ type: ACTUAL_ENABLED_CHANGED, target: this, value: newValue });
        },
        //----------------------------------------------------------------------
        subscribeAbsoluteXChanged: function (handler, context) {
            this.eventTarget.addListener(ABSOLUTE_X_CHANGED, handler, context);
        },

        unscubscribeAbsoluteXChanged: function (handler) {
            this.eventTarget.removeListener(ABSOLUTE_X_CHANGED, handler);
        },

        /*
		* @method fireAbsoluteYChanged fires event "absolute Y position of control changed"
		* (position relatively to top left corner of the page)
		*/
        fireAbsoluteYChanged: function () {
            this.eventTarget.fire({ type: ABSOLUTE_Y_CHANGED, target: this });
        },
        //----------------------------------------------------------------------
        subscribeAbsoluteYChanged: function (handler, context) {
            this.eventTarget.addListener(ABSOLUTE_Y_CHANGED, handler, context);
        },

        unscubscribeAbsoluteYChanged: function (handler) {
            this.eventTarget.removeListener(ABSOLUTE_Y_CHANGED, handler);
        },

        /*
       * @method fireAbsoluteXChanged fires event "absolute X position of control changed"
       * (position relatively to top left corner of the page)
       */
        fireAbsoluteXChanged: function () {
            this.eventTarget.fire({ type: ABSOLUTE_X_CHANGED, target: this });
        },

        //----------------------------------------------------------------------
        subscribeActualWidthChanged: function (handler, context) {
            this.eventTarget.addListener(ACTUAL_WIDTH_CHANGED, handler, context);
        },

        unscubscribeActualWidthChanged: function (handler) {
            this.eventTarget.removeListener(ACTUAL_WIDTH_CHANGED, handler);
        },
        /*
        * @method fireActualWidthChanged fires when with if control was changed
        * (even if control width is specified in % and parent control width changed
        * @param newValue {number} size in pixels
        */
        fireActualWidthChanged: function (newValue) {
            if (!window.initialized && this.widthType == Enums.SizeType.Pixel) {
                return;
            }
            this.eventTarget.fire({ type: ACTUAL_WIDTH_CHANGED, target: this, value: newValue });
        },
        //----------------------------------------------------------------------
        subscribeActualHeightChanged: function (handler, context) {
            this.eventTarget.addListener(ACTUAL_HEIGHT_CHANGED, handler, context);
        },

        unscubscribeActualHeightChanged: function (handler) {
            this.eventTarget.removeListener(ACTUAL_HEIGHT_CHANGED, handler);
        },

        /*
        * @method fireActualHeightChanged fires when height if control was changed
        * (even if control height is specified in % and parent control height changed
        * @param newValue {number} size in pixels
        */
        fireActualHeightChanged: function (newValue) {
            this.eventTarget.fire({ type: ACTUAL_HEIGHT_CHANGED, target: this, value: newValue });
        },
        /**
         *  Added to let agility know to not do proxy for controller.
         */
        _noProxy: function () { }
    });

    function mergeActions(actions) {
        if (!actions) {
            return;
        }

        this.actions = this.actions || {};
        var i;
        for (i = 0; i < actions.length; i++) {
            var action = actions[i]; //id уникальны
            //действия экземпляра могут только дополнять пришедшие из типа
            if (this.actions[action.id] == undefined) {
                this.actions[action.id] = ActionFactory.createByType(action);
            }
        }
    }

    return ControlController;
});
