﻿
define(['common/Enums', 'common/Utilites', 'common/Error'], function (Enums, Utilites, Error) {

    var Appearence = {};
    var color_map = {};
    Appearence.addFlashStyle = function (interval) {
        var sheet = document.createElement('style');
        var animationString = String.format('FlashAnimation {0}ms infinite;', interval);
        sheet.innerHTML = '.flash { ' +
            '-webkit-animation: ' + animationString +
            '-moz-animation: ' + animationString +
            '-o-animation:  ' + animationString +
            'animation:  ' + animationString +
            '}';

        document.body.appendChild(sheet);
    }

    Appearence.applyVisibility = function (target, value) {
        if (!value) {
            target.css('display', 'none');
            return;
        }

        target.css('display', 'block');
    }

    Appearence.applyReadOnly = function (target, value) {
        if (!value) {
            target.attr("readonly", false);
            return;
        }

        target.attr("readonly", true);
    }

    Appearence.applyDisabled = function (target, value) {
        if (!value) {
            target.prop("disabled", false);
            return;
        }

        target.prop("disabled", true);
    }


    //***************************************************************
    // text utils
    //***************************************************************

    Appearence.text = {};

    /*
    * @method getTextHAlign returns value for CSS property text-align
    * @param value {TextHorizontalAlignment}
    */
    Appearence.text.getTextHAlign = function (value) {
        switch (value) {
            case Enums.TextHorizontalAlignment.Left:
                return 'left'
            case Enums.TextHorizontalAlignment.Right:
                return 'right'
            case Enums.TextHorizontalAlignment.Center:
                return 'center'
            case Enums.TextHorizontalAlignment.Justyfy:
                return 'justify'
            case Enums.TextHorizontalAlignment.Auto:
                return 'auto'
            case Enums.TextHorizontalAlignment.Inherit:
                return 'inherit';
        }
    };

    /*
    * @method getTextVAlign returns value for CSS property vertical-align
    * @param value {TextVerticalAlignment}
    */
    Appearence.text.getTextVAlign = function (value) {
        switch (value) {
            case Enums.TextVerticalAlignment.Baseline:
                return 'baseline';
            case Enums.TextVerticalAlignment.Bottom:
                return 'bottom'
            case Enums.TextVerticalAlignment.Center:
                return 'middle';
            case Enums.TextVerticalAlignment.Sub:
                return 'sub';
            case Enums.TextVerticalAlignment.Super:
                return 'super';
            case Enums.TextVerticalAlignment.Bottom:
                return 'text-bottom';
            case Enums.TextVerticalAlignment.TextTop:
                return 'text-top';
            case Enums.TextVerticalAlignment.Top:
                return 'top';
            case Enums.TextVerticalAlignment.Inherit:
                return 'inherit';
        }
    };
    Appearence.getDisplay = function (value) {
        if (value) {
            return '';
        } else {
            return 'none';
        }
    },

    Appearence.text.getTextBold = function (value) {
        if (value) {
            return 'bold';
        } else {
            return 'normal';
        }
    }
    Appearence.text.applyTextBold = function (target, value) {
        target.css('font-weight', Appearence.text.getTextBold(value));
    }
    Appearence.text.getTextItalic = function (value) {
        if (value) {
            return 'italic';
        } else {
            return 'normal';
        }
    }
    Appearence.text.applyTextItalic = function (target, value) {
        target.css('font-style', Appearence.text.getTextItalic(value));
    }
    Appearence.text.getTextUnderline=function(value){
        if (value) {
            return 'underline';
        } else {
            return '';
        }
    }
    Appearence.text.applyTextUnderline = function (target, value) {
        target.css('text-decoration', Appearence.text.getTextUnderline(value));
    }
    Appearence.text.applyTextColor = function (target, value) {
        target.css('color', value);
    }
    //***************************************************************
    // positioning utils
    //***************************************************************

    Appearence.position = {};

    /*
    * @mathod scale applies scale to target
    * @param target {jQuery}
    * @param scale {object} with mandatory fields scaleX, scaleY, angle
    */
    Appearence.position.scale = function (target, newScale) {

        if (newScale.scaleX === undefined || newScale.scaleX === null) {
            Error.onerror('Invalid scale X value.');
        }

        if (newScale.scaleY === undefined || newScale.scaleY === null) {
            Error.onerror('Invalid scale Y value.');
        }

        if (newScale.angle === undefined || newScale.angle === null) {
            Error.onerror('Invalid rotate value.');
        }
        var style;
        if (newScale.scaleX == 1 && newScale.scaleY == 1 && newScale.angle == 0) {//если трансформации нет, то лучше вообще ее не применять
            style = '';
        } else {
            style = 'rotate(' + newScale.angle + 'deg) scale(' + newScale.scaleX + ', ' + newScale.scaleY + ')';
        }
        if (Utilites.isWebkit()) {
            //9748 - repair chrome subpixel rendering bug
            target.css({ '-webkit-transform': 'translate3d(0, 0, 0) ' + style });
        } else {
            target.css({ 'transform': style });
        }
    };
    Appearence.getTransform = function (newScale) {
        if (newScale.scaleX === undefined || newScale.scaleX === null) {
            Error.onerror('Invalid scale X value.');
        }

        if (newScale.scaleY === undefined || newScale.scaleY === null) {
            Error.onerror('Invalid scale Y value.');
        }

        if (newScale.angle === undefined || newScale.angle === null) {
            Error.onerror('Invalid rotate value.');
        }
        if (newScale.scaleX == 1 && newScale.scaleY == 1 && newScale.angle == 0) //если трансформации нет, то лучше вообще ее не применять
            return '';
        return 'scale(' + newScale.scaleX + ', ' + newScale.scaleY + ') rotate(' + newScale.angle + 'deg)';
    },
    Appearence.parseRotate = function (rotation) {
        var parse = rotation.match(/rotate\(([0-9]*\.?[0-9]*)\s*([0-9]*\.?[0-9]*)?\s*([0-9]*\.?[0-9]*)?\)/);
        return { angle: Math.round(parseInt(parse[1])), x: Math.round(parseInt(parse[2])), y: Math.round(parseInt(parse[3])) };
    },
    Appearence.formatRotate = function (rotation) {
        return ('rotate(' + Math.floor(rotation.angle) + ' ' + Math.floor(rotation.x) + ' ' + Math.floor(rotation.y) + ')');
    },
    //***************************************************************
    // Color utils
    //***************************************************************

    Appearence.color = {
        grayscale: { kRed: 0.3, kGreen: 0.58, kBlue: 0.12 },
        disabled: {a: 0.4, b: 25}
    };

    /*
    @method objToCss converts object representation to CSS compatible color
    @param obj {object} with fields a,r,g,b
    @returns {string} color in format rgba(r,g,b,a) or #RRGGBB if color is opaque
    */
    Appearence.color.objToCSS = function (obj) {
        if (obj.a === 255) {
            return String.format('#{0:x2}{1:x2}{2:x2}',
                obj.r, obj.g, obj.b);
        } else {
            return String.format('rgba({0},{1},{2},{3})',
                obj.r, obj.g, obj.b, (obj.a / 256).toFixed(3));
        }
    };

    /*
    @method objToARGB converts object representation to #AARRGGBB
    @param obj {object} with fields a,r,g,b
    @returns {string} color in format #AARRGGBB
    */
    Appearence.color.objToARGB = function (obj) {
        return String.format('#{0:x2}{1:x2}{2:x2}{3:x2}', obj.a, obj.r, obj.g, obj.b);
    };

    /*
    @method objToRGB converts object representation to #RRGGBB
    @param obj {object} with fields a,r,g,b
    @returns {string} color in format #RRGGBB (a channel is ignored)
    */
    Appearence.color.objToRGB = function (obj) {
        return String.format('#{0:x2}{1:x2}{2:x2}', obj.r, obj.g, obj.b);
    };

    Appearence.color.addToObj = function (obj, k) {
        obj.r = Math.max(0, Math.min(255, obj.r + k));
        obj.b = Math.max(0, Math.min(255, obj.b + k));
        obj.g = Math.max(0, Math.min(255, obj.g + k));
        return obj;
    },
    Appearence.color.mulObj = function (obj, k) {
        obj.r = Math.min(255, Math.floor(obj.r * k));
        obj.b = Math.min(255, Math.floor(obj.b * k));
        obj.g = Math.min(255, Math.floor(obj.g * k));
        return obj;
    },
    Appearence.color.invertObj = function (obj) {
        obj.r = 255 - obj.r;
        obj.b = 255 - obj.b;
        obj.g = 255 - obj.g;
        return obj;
    },
    Appearence.color.parse = function (value) {
        var val = {};
        var brushType = Appearence.background.getBrushType(value);
        switch (brushType) {
            case Enums.BrushType.RADIAL_GRADIENT:
                val = Appearence.background.radialGradient.parse(value);
                break;
            case Enums.BrushType.LINEAR_GRADIENT:
                val = Appearence.background.linearGradient.parse(value);
                break;
            default:
                var color = Appearence.background.solid.parse(value);
                val = { colors: [] };
                val.colors[0] = value;
        }
        for (var i = 0; i < val.colors.length; i++) {
            val.colors[i] = Appearence.color.ARGBtoObj(val.colors[i]);
        }
        val.brushType = brushType;
        return val;
    }
    /*
    @method isColor checks if string represents color
    @param color {string}
    @returns {boolean} true, is string specifies color (predefined or in hex format)
    */
    Appearence.color.isColor = function (color) {
        if (color === '') {
            return true;
        }
        if (color.charAt(0) !== '#') {
            return Enums.PredefinedColors.hasOwnProperty(color.toLowerCase());
        } else {
            return (/#[A-Za-z0-9]{6}/.test(color) || /#[A-Za-z0-9]{8}/.test(color));
        }
    };

    /*
    @method isHex chacks whether value is #AARRGGBB or #RRGGBB color
    */
    Appearence.color.isHex = function (value) {
        if (value.charAt(0) !== '#') {
            return false;
        }
        else {
            return true;
        }
    };
    Appearence.color.isRGBHex = function (value) {
        if (Appearence.color.isHex(value))
            if (value.length < 8)
                return true;
        return false;
    },
    Appearence.color.isARGB = function (value) {
        if (/rgb[a]?[(](\s*[0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*[,]?\s*(?:[-+]?[0-9]*\.?[0-9]*)?[)]/.test(value)) //значение и так в формате rgb/rgba
            return true;
        else {
            return false;
        }
    };
    Appearence.color.color2Matrix = function (value) {
        var color = Appearence.color.ARGBtoObj(value);
        if (color) {
            value = color.r / 255 + " 0 0 0 0 0 " + color.g / 255 + " 0 0 0 0 0 " + color.b / 255 + " 0 0 0 0 0 " + color.a / 255 + " 0";
        }
        else {
            Error.warn(String.format("Недопустимое значение цвета {0} для перевода в матрицу. По умолчанию будет использоваться единичная матрица", value));
            value = "1 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 1 0";
        }
        return value;
    }
    /**
 * Converts an RGB color value to HSL. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
 * Assumes r, g, and b are contained in the set [0, 255] and
 * returns h, s, and l in the set [0, 1].
 *
 * @param   Number  r       The red color value
 * @param   Number  g       The green color value
 * @param   Number  b       The blue color value
 * @return  Array           The HSL representation
 */
    Appearence.color.rgbToHsl = function (obj) {
        var r = obj.r /= 255, g = obj.g /= 255, b = obj.b /= 255;
        var max = Math.max(r, g, b), min = Math.min(r, g, b);
        var h, s, l = (max + min) / 2;

        if (max == min) {
            h = s = 0; // achromatic
        } else {
            var d = max - min;
            s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
            switch (max) {
                case r: h = (g - b) / d + (g < b ? 6 : 0); break;
                case g: h = (b - r) / d + 2; break;
                case b: h = (r - g) / d + 4; break;
            }
            h /= 6;
        }

        return [h, s, l];
    }

    /*
    @method toCssColor converts ARGB color to css compatible value
    @param value {string} color in format #AARRGGBB
    @returns {string} color in format rgba(r,g,b,a)
    */
    Appearence.color.toCssColor = function (value) {
        if (Appearence.color.isARGB(value)) {
            return value;
        }
        if (Appearence.color.isRGBHex(value)) {
            var colorValue = value.substr(1).toLowerCase();
            while (colorValue.length < 6) { //если в hex не хватало знаков
                colorValue = "0" + colorValue;
            }
            return '#' + colorValue;
        }
        if (color_map[value]) { //если такой цвет уже существует в кеше
            return color_map[value];
        }
        else {
            color_map[value] = Appearence.color.toCssColorSlow(value);
            return color_map[value];
        }
    };

    Appearence.color.toCssColorSlow = function (value) {
        value = Appearence.color.toARGB(value);
        var colorValue = value.substr(1).toLowerCase();
        var alpha,
            red,
            green,
            blue;
        while (colorValue.length < 6) { //если в hex не хватало знаков
            colorValue = "0" + colorValue;
        }
        if (colorValue.length == 6) {
            alpha = 'ff';
            red = colorValue.substr(0, 2);
            green = colorValue.substr(2, 2);
            blue = colorValue.substr(4, 2);
        }
        else {
            alpha = colorValue.substr(0, 2);
            red = colorValue.substr(2, 2);
            green = colorValue.substr(4, 2);
            blue = colorValue.substr(6, 2);
        }
        if (alpha === 'ff') {
            return '#' + red + green + blue;
        }
        alpha = parseInt(alpha, 16);
        red = parseInt(red, 16);
        green = parseInt(green, 16);
        blue = parseInt(blue, 16);
        alpha = alpha / 256;
        return String.format('rgba({0},{1},{2},{3})', red, green, blue, alpha.toFixed(3));
    };
    Appearence.color.numberToARGB = function (value) {
        var hex = value.toString(16);
        if (hex.length > 8)
            hex = hex.substr(0, 8);
        while (hex.length!=8) {
            hex = "0" + hex;
        }
        return "#" + hex;
    };

    Appearence.color.toARGB = function (value) {
        var value = value.replace(/Solid\s+/i, '');
        if (!value) {
            Error.warn('Invalid color value ' + value);
            return value;
        }

        if (Enums.PredefinedColors.hasOwnProperty(value.toLowerCase())) {
            value = Enums.PredefinedColors[value.toLowerCase()];
        }

        if ((!Appearence.color.isHex(value)) || (!Appearence.color.isColor(value))) {
            Error.warn('Invalid color value ' + value);
            return value;
        }
        return value;
    }

    Appearence.color.ARGBToNumber = function (value) {
        var color = Appearence.color.toARGB(value);
        return parseInt(color.slice(1, color.length), 16);
    };
    /*
    @method ARGBtoObj parses specified color falue
    @param value {string} in format #AARRGGBB
    @returns {object} with fields a,r,g,b
    */
    Appearence.color.ARGBtoObj = function (value) {
        if (!value) {
            return null;
        }
        if (Appearence.color.isARGB(value)) {
            var color_array = value.match(/rgb[a]?[(](\s*[0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*[,]?\s*([-+]?[0-9]*\.?[0-9]*)?[)]/);
            return {
                a: parseInt(color_array[4]*255),
                r: parseInt(color_array[1]),
                g: parseInt(color_array[2]),
                b: parseInt(color_array[3])
            }
        }
        var v = value;
        //Solid #FFFFFFF
        var value = value.replace(/Solid\s+/i, '');

        if (value.charAt(0) !== '#') {
            value = Enums.PredefinedColors[value.trim().toLowerCase()];
            if (!value) {
                Error.warn('Invalid color value ' + v);
                return undefined;
            }
        }

        var colorValue = value.substr(1).toLowerCase();
        var alpha,
            red,
            green,
            blue;

        if (colorValue.length < 8) {
            alpha = 'ff';
            red = colorValue.substr(0, 2);
            green = colorValue.substr(2, 2);
            blue = colorValue.substr(4, 2);
        }
        else {
            alpha = colorValue.substr(0, 2);
            red = colorValue.substr(2, 2);
            green = colorValue.substr(4, 2);
            blue = colorValue.substr(6, 2);
        }

        return {
            a: parseInt(alpha, 16),
            r: parseInt(red, 16),
            g: parseInt(green, 16),
            b: parseInt(blue, 16)
        };
    };
    Appearence.color.makeGrayscale = function (color) {
        var val, newcolor;
        val = Appearence.color.parse(color);
        for (var i = 0; i < val.colors.length; i++) {
            var new_color = (val.colors[i].r * this.grayscale.kRed + val.colors[i].g * this.grayscale.kGreen + val.colors[i].b * this.grayscale.kBlue) / 3;
            val.colors[i] = Appearence.color.objToARGB({
                a: val.colors[i].a,
                r: parseInt(new_color, 16),
                g: parseInt(new_color, 16),
                b: parseInt(new_color, 16)
            });
        }
        switch (val.brushType) {
            case Enums.BrushType.RADIAL_GRADIENT:
                return Appearence.background.radialGradient.toString(val);
            case Enums.BrushType.LINEAR_GRADIENT:
                return Appearence.background.linearGradient.toString(val);
            default:
                return val.colors[0];
        }
    };
    Appearence.color.makeDisabled = function (color) {
        return Appearence.background.calculateColor(this.makeGrayscale(color), this.disabled.a, this.disabled.b);
    };
    /*
    @method makeOpaque
    @param o {object} with fields a,r,g,b
    @returns {object} with fields a,r,g,b; a==256
    */
    Appearence.color.makeOpaque = function (o) {
        var k = (255 - o.a) * 255 / 256;
        var nR = Math.floor(k + o.a * o.r / 256);
        var nG = Math.floor(k + o.a * o.g / 256);
        var nB = Math.floor(k + o.a * o.b / 256);

        if (nR > 255) nR = 255;
        if (nG > 255) nG = 255;
        if (nB > 255) nB = 255;

        return {
            a: 255,
            r: nR,
            g: nG,
            b: nB
        };
    };

    /*
     @method ARGBToRGB converts specifiend color to opaque color
     @param {string} color in format #AARRGGBB
     @returns {string} color in format #FFRRGGBB
    */
    Appearence.color.ARGBToRGB = function (color) {

        if (!Appearence.color.isHex(color)) {
            return color;
        }

        var source = Appearence.color.ARGBtoObj(color);
        var dest = Appearence.color.makeOpaque(source);

        return Appearence.color.objToARGB(dest);
    };

    /*
     @method ARGBToOpaqueCSS converts specifiend color to opaque color
     @param {string} color in format #AARRGGBB
     @returnds {string} color in format rgba(r,g,b,a)
    */
    Appearence.color.ARGBToOpaqueCSS = function (color) {

        if (!Appearence.color.isColor(color)) { //INSAT: цвета необязательно в hex-формате, они могут называться словами
            return color;
        }

        var source = Appearence.color.ARGBtoObj(color);
        var dest = Appearence.color.makeOpaque(source);

        return Appearence.color.objToCSS(dest);
    };

    //***************************************************************
    // Shadow utils
    //***************************************************************

    Appearence.shadow = {};

    /*
    * @method applyBoxShadow applies shadow to target
    * @param target {jQuery}
    * @param shadow {object} object with size and color fields
    */
    Appearence.shadow.applyBoxShadow = function (target, shadow) {
        if (shadow.size === undefined) {
            Error.onerror('Box shadow size is not specified.');
        }

        if (shadow.color === undefined) {
            Error.onerror('Box shadow color is not specified.');
        }

        var sharpness = (Utilites.isWebkit() ? 1 : 0) & (shadow.size == 0 ? 0 : 1);
        var color = Appearence.color.toCssColor(shadow.color);

        var css = String.format('{0}px {0}px {1}px {2}', shadow.size, sharpness, color);

        target.css('box-shadow', css);
    };

    /*
    * @method applyTextShadow applies shadow to target
    * @param target {jQuery}
    * @param shadow {object} object with size and color fields
    */
    Appearence.shadow.applyTextShadow = function (target, shadow) {
        if (shadow.size === undefined) {
            Error.onerror('Text shadow size is not specified.');
        }
        if (shadow.size == 0) { //если тени нет, то ее не нужно выставлять
            return;
        }
        if (shadow.color === undefined) {
            Error.onerror('Text shadow color is not specified.');
        }
       
        var color = Appearence.color.toCssColor(shadow.color);

        var css = String.format('{0}px {0}px 1px {1}', shadow.size / 2, color);

        target.css('text-shadow', css);
    };

    //***************************************************************
    // Border utils
    //***************************************************************

    Appearence.border = {};

    /*
    * @method applyBorder applys border to target
    * @param target {jQuery}
    * @param value {object} object with fields penType, size, color, penAlignment
    */
    Appearence.border.applyBorder = function (target, border) {
        if (border.penType === undefined) {
            Error.onerror('Pen type is not specified for border.');
        }

        if (border.size === undefined) {
            Error.onerror('Size is not specified for border.');
        }

        if (border.color === undefined) {
            Error.onerror('Color is not specified for border.');
        }

        var borderStyle = Appearence.border.getBorderStyle(border.penType);
        var borderColor = Appearence.color.toCssColor(border.color);
        var borderValue = String.format('{0} {1}px {2}', borderStyle, border.size, borderColor);

        target.css('border', borderValue);
        //target.css('border-right', borderValue);
        //target.css('border-top', borderValue);
        //target.css('border-bottom', borderValue);
    };

    Appearence.border.getBorderStyle = function (penType) {
        switch (penType) {
            case Enums.BorderStyleType.Solid:
                return 'solid';
            case Enums.BorderStyleType.Dot:
                return 'dotted';
            case Enums.BorderStyleType.Dash:
                return 'dashed';
            default:
                return 'none';
        }
    };

    /*
    * @method applyCornerRadius
    * @param target {jQuery}
    * @param value {number} or {object} with fields radius and units
    */
    Appearence.border.applyCornerRadius = function (target, value) {

        if (!value) {
            target.css('border-radius', '');
            return;
        }

        var radius,
            units;
        if (typeof value === 'object') {
            radius = value.radius;
            units = value.units || 'px';
        }
        else {
            radius = value;
            units = 'px';
        }

        target.css('border-radius', radius + units);
    };

    /*
    * @method applyBoxSizing
    * @param target {jQuery}
    * @param value {string}
    */
    Appearence.border.applyBoxSizing = function (target, value) {
        target.css('box-sizing', value);
        target.css('-moz-box-sizing', value);
    };

    //***************************************************************
    // Background utils
    //***************************************************************

    Appearence.background = {};

    Appearence.background.SOLID = 'solid';

    /*
    * @method getBrushType
    * @param value {string} value of BackgroundColor parameter
    * @result {BrushType}
    */
    Appearence.background.getBrushType = function (value) {
        if (!value) {
            return Enums.BrushType.NONE;
        }

        var spaceIndex = value.indexOf(' ');

        var bgType;

        if (spaceIndex < 0) {
            //this is solid color
            return Enums.BrushType.SOLID;
        }
        else {
            bgType = value.slice(0, spaceIndex);
        }

        if (bgType) {
            bgType = bgType.toLowerCase();
        }

        if (bgType === Appearence.background.SOLID) {
            return Enums.BrushType.SOLID;
        } else if (bgType === Appearence.background.radialGradient.NAME) {
            return Enums.BrushType.RADIAL_GRADIENT;
        } else if (Utilites.isEnumValue(Enums.LinearGradientBrushDirection, bgType)) {
            return Enums.BrushType.LINEAR_GRADIENT;
        } else {
            return Enums.BrushType.NONE;
        }
    };

    /*
    * @method getBrushString converts gradient brush object and returns string
    * @param value {object{colors[string]}} result of function parseRadialGradient or parseLinearGradient
    */
    Appearence.background.getBrushString = function (value) {
        var result = '',
            step = 1.0 / (value.colors.length - 1),
            currentStop = 0,
            i,
            color;
        for (i = 0; i < value.colors.length; i++) {

            color = Appearence.color.toCssColor(value.colors[i]);

            result += String.format("{0} {1}%", color, Math.floor(currentStop * 100));

            if (i < value.colors.length - 1) {
                result += ',';
            }

            currentStop += step;
        }

        return result;
    };

    Appearence.background.apply = function (target, value) {
        if (!value || !_.isString(value)) {
            Error.info(value + "не является цветом");
            return;
        }
        var brushType = Appearence.background.getBrushType(value);

        switch (brushType) {
            case Enums.BrushType.RADIAL_GRADIENT:
                Appearence.background.radialGradient.apply(target, value);
                break;
            case Enums.BrushType.LINEAR_GRADIENT:
                Appearence.background.linearGradient.apply(target, value);
                break;
            default:
                Appearence.background.solid.apply(target, value);
                break;
        }
    };
    Appearence.background.calculateColor = function (value, k, b) { //новый цвет по формуле k*x+b
        var val, newcolor, newval;
        val = Appearence.color.parse(value);
        for (var i = 0; i < val.colors.length; i++) {
            newcolor = val.colors[i];
            newcolor = Appearence.color.mulObj(newcolor, k);
            newcolor = Appearence.color.addToObj(newcolor, b + Math.round((1-k)*255));
            val.colors[i] = Appearence.color.objToARGB(newcolor);
        }
        switch (val.brushType) {
            case Enums.BrushType.RADIAL_GRADIENT:
                return Appearence.background.radialGradient.toString(val);
            case Enums.BrushType.LINEAR_GRADIENT:
                return Appearence.background.linearGradient.toString(val);
            default:
                return val.colors[0];
        }
    },
    /*
    * @method getSingleColor returns bg color 
    * if bg is gradient then it returns first color
    * @param {string} value of BackgroundColor parameter of controls
    */
    Appearence.background.getSingleColor = function (value) {

        var brushType = Appearence.background.getBrushType(value);
        var color;

        switch (brushType) {
            case Enums.BrushType.RADIAL_GRADIENT:
                parsedBg = Appearence.background.radialGradient.parse(value);
                if (parsedBg && parsedBg.colors && parsedBg.colors.length > 0) {
                    color = parsedBg.colors[0];
                }
                break;
            case Enums.BrushType.LINEAR_GRADIENT:
                parsedBg = Appearence.background.linearGradient.parse(value);
                if (parsedBg && parsedBg.colors && parsedBg.colors.length > 0) {
                    color = parsedBg.colors[0];
                }
                break;
            default:
                color = Appearence.background.solid.parse(value);
        }

        if (color) {
            color = Appearence.color.toCssColor(color);
            return color;
        }

        return 'transparent';
    };

    // define BackgroundImage css hook
    // it will help us to set simultaneously background image AND any gradient using $.css
    (function ($) {
        if (!$.cssHooks) {
            throw (new Error("jQuery 1.4.3+ is needed for this plugin to work"));
        }

        var bgImage = 'backgroundImage',
			image = /url\(.*\)/,
			gradient = /(radial|linear)-gradient\([\w\d\,\s]*\)/,
			regex;


        $.cssHooks.backgroundImage = {
            get: function (elem, computed, extra) {
                return elem.style[bgImage];
            },
            set: function (elem, value) {
                var existing = elem.style[bgImage];
                // we are setting image
                if (image.test(value)) {
                    regex = image;
                } else if (gradient.test(value)) {
                    regex = gradient;
                }

                if (regex !== undefined) {
                    // existing and new values are from same group
                    if (regex.test(existing) && regex.test(value)) {
                        value = existing.replace(regex, value);
                    } else {
                        value = existing === '' || value === '' ? value : function () {
                            var array = [existing, value],
								result;
                            // regex === image means that existing value is not an URL
                            // to override gradient we need to render image first
                            regex === image && !regex.test(existing) ?
								result = array.reverse().join(', ') :
								result = array.join(', ');
                            return result;
                        }();
                    }
                }

                elem.style[bgImage] = value;
            }
        };
    })(jQuery);

    //
    //will contain Appearence for managing radial gradients
    //
    Appearence.background.radialGradient = {};

    Appearence.background.radialGradient.NAME = "radialgradient";

    /*
    * @method parse parses gradient string to object
    * @param {string} радиальный градиент в формате "RadialGradient x y R1 R2 #ARGB #ARGB" где x,y - положение центра, а R1 R2 - радиус
    */
    Appearence.background.radialGradient.parse = function (value) {
        var params = value.split(/\s+/);

        if (params.length < 8) {
            Error.onerror("Invalid radial gradient format.");
        }

        var radialGradient = {
            fx: parseFloat(params[1]),
            fy: parseFloat(params[2]),
            cx: parseFloat(params[3]),
            cy: parseFloat(params[4]),
            rx: parseFloat(params[5]),
            ry: parseFloat(params[6]),
            colors: params.slice(7)
        };

        return radialGradient;
    };

    Appearence.background.radialGradient.toString = function (value) {
        return "RadialGradient " + value.fx + " " + value.fy + " " + value.cx + " " + value.cy + " " + value.rx + " " + value.ry + " " + value.colors.join(" ");
    };
    /*
    * @method apply applies specified radial gradient brush settings to jQuery object
    * @param target {jQuery object} object(s) to which we want to apply background
    * @param value {string} радиальный градиент в формате "RadialGradient x y R1 R2 #ARGB #ARGB" где x,y - положение центра, а R1 R2 - радиус
    * @returns {object} result of function parse
    */
    Appearence.background.radialGradient.apply = function (target, value) {

        if (!value) {
            target.css('background-image', '');
            return;
        }

        var gradient = typeof value === 'object' ? value : Appearence.background.radialGradient.parse(value);
        var brushString = Appearence.background.getBrushString(gradient);
        var graientValueFormat, browserPrefix = '';
        if (Utilites.isSafari()) {
            //think different (c)
            graientValueFormat = '({2}% {3}%, {0}% {1}%, {4})';
            browserPrefix = '-webkit-'
        } else {
            graientValueFormat = '(ellipse {0}% {1}% at {2}% {3}%, {4})';            
        }
        var radiusScale = Math.ceil(100 - (100/gradient.colors.length)),
            graientValue = String.format(graientValueFormat, gradient.rx * radiusScale, gradient.ry * radiusScale, gradient.cx * 100, gradient.cy * 100, brushString);
        target.css({ backgroundImage: browserPrefix + 'radial-gradient' + graientValue });
    };

    //will contain Appearence for managing radial gradients
    Appearence.background.linearGradient = {};

    Appearence.background.linearGradient.directionToString = function (value) {
        switch (value.toLowerCase()) {
            case Enums.LinearGradientBrushDirection.TOP:
                return 'top';
            case Enums.LinearGradientBrushDirection.LEFT:
                return 'left';
            case Enums.LinearGradientBrushDirection.BOTTOM:
                return 'bottom';
            case Enums.LinearGradientBrushDirection.RIGHT:
                return 'right';
            case Enums.LinearGradientBrushDirection.TOP_LEFT:
                return 'top left';
            case Enums.LinearGradientBrushDirection.TOP_RIGHT:
                return 'top right';
            case Enums.LinearGradientBrushDirection.BOTTOM_LEFT:
                return 'bottom left';
            case Enums.LinearGradientBrushDirection.BOTTOM_RIGHT:
                return 'bottom right';
            default:
                return 'left';

        }
    };

    /*
    * @method parse parses linear gradient string to object
    * @param {string} радиальный градиент в формате "RadialGradient x y R1 R2 #ARGB #ARGB" где x,y - положение центра, а R1 R2 - радиус
    */
    Appearence.background.linearGradient.parse = function (value) {
        var params = value.split(/\s+/);

        if (params.length < 2) {
            Error.onerror("Invalid linear gradient format.");
        }

        var linearGradient = {
            direction: params[0],
            colors: params.slice(1)
        };

        return linearGradient;
    };

    Appearence.background.linearGradient.toString = function (value) {
        return value.direction + " " + value.colors.join(" ");
    };
    /*
    * @method apply applies specified linear gradient brush settings to jQuery object
    * @param target {jQuery object} object(s) to which we want to apply background
    * @param value {string} градиентный цвет в формате: "LeftGradient #ARGB #ARGB #ARGB #ARGB #ARGB..."
    *  LeftGradient, TopGradient, BOTTOMGRADIENT, RIGHTGRADIENT, TOPLEFTGRADIENT, TOPRIGHTGRADIENT, BOTTOMLEFTGRADIENT, BOTTOMRIGHTGRADIENT
    *             {object} result of function parse
    */
    Appearence.background.linearGradient.apply = function (target, value) {

        if (!value) {
            target.css('background-image', '');
            return;
        }

        //gradient should override solid color
        // target.css('background-color', '')

        var gradient = typeof value === 'object' ? value : Appearence.background.linearGradient.parse(value);

        var direction = Appearence.background.linearGradient.directionToString(gradient.direction);
        var brushString = Appearence.background.getBrushString(gradient);

        var gradientValue = String.format('({0},{1})', direction, brushString);

        target.css({ backgroundImage: window.PrefixFree.prefix + 'linear-gradient' + gradientValue });
    };

    //will contain Appearence for managing solid brush
    Appearence.background.solid = {};

    /*
    @method parse extracts solid color
    */
    Appearence.background.solid.parse = function (value) {
        var spaceIndex = value.indexOf(' ');
        if (spaceIndex < 0) {
            return value;
        }

        return value.substr(spaceIndex + 1);
    };

    /*
    * @method apply applies specified solid brush settings to jQuery object
    * @param target {jQuery object} object(s) to which we want to apply background
    * @param value {string} color   
    */
    Appearence.background.solid.apply = function (target, value) {
        if (value === undefined) {
            return;
        }

        var color = Appearence.background.solid.parse(value);

        color = Appearence.color.toCssColor(color);
        if (color === '') {
            // make transparent bg, if we setting empty value
            // override default bg for tags in browser
            color = 'rgba(0,0,0,0)';
        }
        // Если на фоне градиент, убираем его
        if (target.css('background-image') && target.css('background-image').indexOf('url') < 0) {
            target.css({ backgroundImage: '' });
        }

        target.css('background-color', color);
    };

    /*
    * @method sets bg image for target
    * @param target {jQuery} 
    * @param url {string} image url
    */
    Appearence.background.applyBgImage = function (target, url) {
        if (!url) {
            target.css({ backgroundImage: '' });
        }
        else {
            target.css({ backgroundImage: 'url(\'' + url + '\')' });
        }
    };

    /*
    * method configures background settings for target
    * @param target {jQuery}
    @ @param tile {string} image usage
    */
    Appearence.background.applyBackgroundTile = function (target, tile) {

        switch (tile) {
            case Enums.TileType.No:
                target.css({
                    'background-size': '',
                    'background-repeat': 'no-repeat'
                });
                break;
            case Enums.TileType.Tile:
                target.css({
                    'background-size': '',
                    'background-repeat': 'repeat'
                });
                break;
            case Enums.TileType.Fill:
                target.css({
                    'background-size': '100% 100%',
                    'background-repeat': 'no-repeat'
                });
                break;
            case Enums.TileType.Center:
                target.css({
                    'background-size': '',
                    'background-position': 'center center',
                    'background-repeat': 'no-repeat',
                });
                break;
            default:
                target.css({
                    'background-size': '',
                    'background-position': '',
                    'background-repeat': ''
                });
        }
    };

    //***************************************************************
    // keyboard utils
    //***************************************************************

    Appearence.keyboard = {};

    /*
    * @method returns layout value for jquery keyboard by keyboardType
    * @param keyboardType {Enums.KeyboardType}
    */
    Appearence.keyboard.getLayoutByType = function (keyboardType) {

        switch (keyboardType) {
            case Enums.KeyboardType.INTERNATIONAL:
                return 'international';
            case Enums.KeyboardType.English:
                return 'qwerty';
            case Enums.KeyboardType.Russian:
                return 'russian-qwerty';
            case Enums.KeyboardType.NumPad:
                return 'num';
            case Enums.KeyboardType.HEX:
                return 'custom';
            default:
                return 'qwerty';
        }
    };

    /*
    * @method getJQueryKeyboardVisibility
    * @param keyboardVisibility {Enums.KeyboadVisibility}
    * @returns value for openOn property of JQueryKeyboard
    */
    Appearence.keyboard.getJQueryVisibility = function (keyboardVisibility) {
        switch (keyboardVisibility) {
            case Enums.KeyboardVisibilityType.None:
                return 'none';
            case Enums.KeyboardVisibilityType.Always:
                return 'always';
            case Enums.KeyboardVisibilityType.OnFocus:
                return 'focus';
        }
    };

    /*
    * @method getJQueryKeyboardBinding
    * @param attachType {Enums.KeyboardBinding}
    * @returns {string} value for 'binding' property of jquery keyboard
    */
    Appearence.keyboard.getJQueryBinding = function (attachType) {
        switch (attachType) {
            case Enums.AttachType.Control:
                return 'control';
            case Enums.AttachType.Parent:
                return 'parent';
            default: return '';
        }
    };

    /*
    * @method getJQueryKeyboardPosition
    * @param keyboardPosition {Enums.KeyboardPosition}
    * @returns {string} value for position property of jquery keyboard
    */
    Appearence.keyboard.getJQueryPosition = function (keyboardPosition, berth) {
        /*
        UP: 'up',
            DOWN: 'down',
            LEFT: 'left',
            RIGHT: 'right'
        */
        switch (keyboardPosition) {
            case Enums.LayoutType.Up:
                return {
                    my: 'center bottom',
                    at: 'center top',
                    at2: 'center top',
                    of: berth,
                    collision: 'flipfit flipfit'
                };
            case Enums.LayoutType.Down:
                return {
                    my: 'center top',
                    at: 'center bottom',
                    at2: 'center bottom',
                    of: berth,
                    collision: 'flipfit flipfit'
                };
            case Enums.LayoutType.Left:
                return {
                    my: 'right center',
                    at: 'left center',
                    at2: 'left center',
                    of: berth,
                    collision: 'flipfit flipfit'
                };
            case Enums.LayoutType.Right:
                return {
                    my: 'left center',
                    at: 'right center',
                    at2: 'right center',
                    of: berth,
                    collision: 'flipfit flipfit'
                }
            default:
                Error.onerror('Unknown keyboard position ' + keyboardPosition);
        }
    };

    /*
    * @method getHexLayout
    * Returns settings for HEX layout for jquery keyboard
    */
    Appearence.keyboard.getHexLayout = function () {

        return "{" +
        "    'default' : [" +
                'C D E F' +
                '8 9 A B' +
                '4 5 6 7' +
                '0 1 2 3' +
                '{bksp} {a} {c}' +
        ']' +
        '}';
    };

    //***************************************************************
    // svg utils
    //***************************************************************

    Appearence.svg = {};

    /*
    @method getSVGBrushUrl builds identifier for target's brush
    @param target {agility}
    @returns 
    */
    Appearence.svg.getSVGBrush = function (target) {
        var bgBrush = target.model.getBackgroundColor();
        var bgImage = target.model.getResource();
        var imageFile;

        var brushType = Appearence.background.getBrushType(bgBrush);

        var rgb,
            colorsID,
            gradient,
            solid,
            id;

        switch (brushType) {
            case Enums.BrushType.SOLID:
                solid = Appearence.background.solid.parse(bgBrush);
                colorsID = Appearence.color.ARGBToOpaqueCSS(solid);
                break;
            case Enums.BrushType.LINEAR_GRADIENT:
                gradient = Appearence.background.linearGradient.parse(bgBrush);
                colorsID = gradient.colors.join('');
            case Enums.BrushType.RADIAL_GRADIENT:
                gradient = Appearence.background.radialGradient.parse(bgBrush);
                colorsID = gradient.colors.join('');
                break;
            default:
                return 'transparent';
        }

        colorsID = colorsID.replace('#', '_');

        if (bgImage) {
            imageFile = target.model.getResource(bgImage);
            id = 'pattern' + target.controller.clientId() + imageFile;
        }
        else {
            id = 'pattern' + target.controller.clientId() + colorsID;
        }

        return String.format('url(#{0})', id);
    }

    /*
    @method getDashes
    */
    Appearence.svg.getDashes = function (borderStyle, borderWidth) {
        switch (borderStyle) {
            case Enums.BorderStyleType.Dash:
                return String.format('{0}, {0}', 5 * borderWidth);
            case Enums.BorderStyleType.Dot:
                return String.format('{0}, {1}', borderWidth, 2 * borderWidth);
            default:
                return '';
        }
    };

    /*
    * @method extractBrushColors
    * @param brush {array} parsed brush
    * @returns {string} list of comma CSS friendly delimited opaque colors i
    */
    Appearence.svg.extractBrushColors = function (brush) {
        return brush.colors.join(',');

    };

    /*
    * @method getLinearBrushCoords
    * @returns {array}
    */
    Appearence.svg.getLinearBrushCoords = function (brush, colors) {
        switch (brush.direction.toLowerCase()) {
            case Enums.LinearGradientBrushDirection.TOP:
                return ['0%,0%', '0%,100%'];
            case Enums.LinearGradientBrushDirection.LEFT:
                return ['0%,0%', '100%,0%'];
            case Enums.LinearGradientBrushDirection.BOTTOM:
                return ['0%,100%', '0%,0%'];
            case Enums.LinearGradientBrushDirection.RIGHT:
                return ['100%,0%', '0%,0%'];
            case Enums.LinearGradientBrushDirection.TOP_LEFT:
                return ['0%,0%', '100%,100%'];
            case Enums.LinearGradientBrushDirection.TOP_RIGHT:
                return ['100%,0%', '0%,100%'];
            case Enums.LinearGradientBrushDirection.BOTTOM_LEFT:
                return ['0%,100%', '100%,0%'];
            case Enums.LinearGradientBrushDirection.BOTTOM_RIGHT:
                return ['100%,100%', '0%,0%'];
            default:
                return ['0%,0%', '0%,100%'];
        }
    };

    /*
    * @method toGradientStops makes SVG stop items from array
    * @param colors {Array} - colors in format #AARRGGBB or #RRGGBB or named
    * returns {Array} array of [step, color_without_alpha_channel, opacity]
    */
    Appearence.svg.toGradientStops = function (colors) {
        var isNamedColor,
            color,
            colorObj,
            i,
            result = [],
            opacity,
            svgColor,
            currentStop = 0,
            step = 1 / colors.length;

        for (i = 0; i < colors.length; i++) {
            color = colors[i];
            isNamedColor = !Appearence.color.isHex(color);

            if (isNamedColor) {
                svgColor = color;
                opacity = 1;
            }
            else {
                colorObj = Appearence.color.ARGBtoObj(color);
                svgColor = Appearence.color.objToRGB(colorObj);
                opacity = colorObj.a / 255;
            }

            result.push([currentStop * 100 + '%', svgColor, opacity]);

            currentStop += step;
        }

        return result;
    };

    /*
    * @method toGradientStops makes SVG stop items from array
    * @param colors {Array} - colors in format #AARRGGBB or #RRGGBB or named
    * returns {Array} array of [step, color_without_alpha_channel, opacity]
    */
    Appearence.svg.toLinearGradientStops = function (colors) {
        var isNamedColor,
            color,
            colorObj,
            i,
            result = [],
            opacity,
            svgColor,
            currentStop = 0,
            len = colors.length > 1 ? colors.length - 1 : 1,
            step = 1 / len;

        for (i = 0; i < colors.length; i++) {
            color = colors[i];
            isNamedColor = !Appearence.color.isHex(color);

            if (isNamedColor) {
                svgColor = color;
                opacity = 1;
            }
            else {
                colorObj = Appearence.color.ARGBtoObj(color);
                svgColor = Appearence.color.objToRGB(colorObj);
                opacity = colorObj.a / 255;
            }

            result.push([currentStop * 100 + '%', svgColor, opacity]);

            currentStop += step;
        }

        return result;
    };

    return Appearence;

});
