﻿// <copyright file="GeometryController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class GeometryController controller class for Geometry control
*/

define(['common/Enums', 'common/Appearence', 'common/Error', 'base/CanvasContainerController', 'common/Utilites',
     'libs/svg/jquery_svg_drawing'],
function (Enums, Appearence, Error, CanvasContainerController, Utilites, svgApi) {

    var GeometryController = CanvasContainerController.extend({

        //says whether geometry was drawn
        isAddedToDom: false,

        init: function () {
            this._super();

            this.ClassName = 'GeometryController';
        },

        onAddedToDOM: function () {
            this.isWebkit = Utilites.isWebkit();
            this._super();

            this.isAddedToDom = true;

            var clientId = this.clientId();

            this.renderShape(clientId, true);
           
            this.renderBackground(clientId);
        },

        getBrushPatternId: function () {
            return this.clientId() + '_bg';
        },

        renderShape: function (clientId, redrawGeometry) {
            var clipPathId = undefined;
            var parentWidth, parentHeight;
            if (this.mvc._parent.controller.ClassName === "GeometryController") {
                var parent = this.mvc._parent;
                clipPathId = parent.controller.clientId();
                parentWidth = parent.model.get(Enums.ParameterRoles.WIDTH);
                parentHeight = parent.model.get(Enums.ParameterRoles.HEIGHT);
            }

            this.mvc.view.$().attr('id', clientId);

            var dashes = Appearence.svg.getDashes(this.mvc.model.getBorderStyle(),
                                                this.mvc.model.getBorderThickness());
            var strokewidth;
            if (this.mvc.model.getBorderStyle() == Enums.BorderStyleType.None) {
                strokewidth = "0px";
            } else {
                strokewidth = this.mvc.model.getBorderThickness() + 'px';
            }
            var options = {
                data: this.mvc.model.get('BorderGeometryOrigin'),
                width: this.mvc.view.$().css('width'),
                height: this.mvc.view.$().css('height'),
                scaleX: 1,
                scaleY: 1,
                clipPathId: clipPathId,
                style: {
                    id: clientId,
                    strokeLineJoin: 'miter',
                    fill: String.format('url(#{0})', this.getBrushPatternId()),
                    strokeDashArray: dashes,
                    opacity: 1,
                    stroke: Appearence.color.toCssColor(this.mvc.model.getBorderColor()),
                    strokeWidth: strokewidth,
                    fillRule: 'evenodd',
                    vectorEffect: 'non-scaling-stroke'
                }
            };

            svgApi.drawPath(clientId, options, redrawGeometry);
            svgApi.removeGeometryShadow(clientId);
            this.renderShadow(clientId);
        },

        renderShadow: function (clientId) {
            if ((!this.mvc.model.getBackgroundColor()) || (!this.mvc.model.getShadowColor()) || (!this.mvc.model.getBorderColor())) {
                Error.warn(String.format('Невозможно нарисовать тень у примитива {6} {0}={1} {2}={3} {4}={5}', Enums.ParameterRoles.BACKGROUND_COLOR, this.mvc.model.getBackgroundColor(),
                                                                                                Enums.ParameterRoles.SHADOW_COLOR, this.mvc.model.getShadowColor(),
                                                                                                Enums.ParameterRoles.BORDER_COLOR, this.mvc.model.getBorderColor(),
                                                                                                this.mvc.model.getId()));
                return;
            }
            var shadow_obj = {}, border_obj = {}, fill_obj = {};
            shadow_obj = Appearence.color.ARGBtoObj(this.mvc.model.getShadowColor());
            if (Appearence.color.isColor(this.mvc.model.getBorderColor()) || Appearence.color.isARGB(this.mvc.model.getBorderColor())) {
                border_obj = Appearence.color.ARGBtoObj(this.mvc.model.getBorderColor());
            } else {
                border_obj.a = 255;
            }
            if (Appearence.color.isColor(this.mvc.model.getBackgroundColor()) || Appearence.color.isARGB(this.mvc.model.getBackgroundColor())) {
                fill_obj = Appearence.color.ARGBtoObj(this.mvc.model.getBackgroundColor());
            } else {
                fill_obj.a = 255;
            }
            var offset=this.mvc.model.getShadowSize();
            if (offset == 0 || shadow_obj.a == 0 || (border_obj.a == 0 && fill_obj.a == 0)) { //если тень прозрачная, то можно ее не рисовать
                svgApi.removeGeometryShadow(clientId);
            } else {
                var options = {
                    offset: offset,
                    strokeWidth: this.mvc.model.getBorderThickness() + 'px'
                };
                if (fill_obj.a === 0) {
                    options.color = Appearence.color.objToARGB(fill_obj);
                    options.fillOpacity = 0;
                }
                else {
                    shadow_obj.a = fill_obj.a;
                    options.color = Appearence.color.toCssColor(Appearence.color.objToARGB(shadow_obj));
                }
                options.strokeColor = Appearence.color.toCssColor(Appearence.color.objToARGB(shadow_obj));
                svgApi.drawShadowForGeometry(clientId, options);
            }
        },

        renderBackground: function (clientId) {
            this.renderColorBg(clientId);

            var bgImage = this.mvc.model.getResource();

            if (bgImage) {
                this.renderImageBg(clientId);
            }
        },

        renderColorBg: function (clientId) {
            var bgBrush = this.mvc.model.getBackgroundColor();

            if (!bgBrush) {
                svgApi.removeGeometryColorBg(svgApi.getSVG(clientId));
                return;
            }

            var brushType = Appearence.background.getBrushType(bgBrush);

            switch (brushType) {
                case Enums.BrushType.SOLID:
                    this.renderSolidBg(clientId);
                    break;
                case Enums.BrushType.LINEAR_GRADIENT:
                    this.renderLinearGradientBg(clientId);
                    break;
                case Enums.BrushType.RADIAL_GRADIENT:
                    this.renderRadialGradientBg(clientId);
                    break;
            }
        },

        renderSolidBg: function (clientId) {

            var brushColor = Appearence.background.solid.parse(
                this.mvc.model.getBackgroundColor());

            var options = {
                svg: svgApi.getSVG(clientId),
                brushId: this.getBrushPatternId(),
                color: Appearence.color.toCssColor(brushColor),
            };

            svgApi.registerSolidBrush(options);
        },

        renderLinearGradientBg: function (clientId) {

            var brushObj = Appearence.background.linearGradient.parse(
                                        this.mvc.model.getBackgroundColor());
            var brushCoords = Appearence.svg.getLinearBrushCoords(brushObj);

            var options = {
                svg: svgApi.getSVG(clientId),
                brushId: this.getBrushPatternId(),
                colors: brushObj.colors,
                brushCoords: { first: brushCoords[0], last: brushCoords[1] }
            };

            svgApi.registerLinearGradientBrush(options);
        },

        renderRadialGradientBg: function (clientId) {
            var brushObj = Appearence.background.radialGradient.parse(
                this.mvc.model.getBackgroundColor());

            var options = {
                svg: svgApi.getSVG(clientId),
                brushId: this.getBrushPatternId(),
                colors: brushObj.colors,
                brushCoords: { cx: brushObj.cx, cy: brushObj.cy, rx: 1 }
            };

            svgApi.registerRadialGradientBrush(options);
        },

        renderImageBg: function (clientId) {

            var resourceId = this.mvc.model.getResource();
            var resourceUrl;

            if (!resourceId) {
                resourceUrl = null;
                svgApi.removeGeometryImageBg(svgApi.getSVG(clientId));
                return;
            }
            else {
                resourceUrl = this.mvc.model.resourceManager.resolve(resourceId);
            }

            var options = {
                svg: svgApi.getSVG(clientId),
                brushId: this.getBrushPatternId(),
                imagePath: resourceUrl,
                imageTiling: this.mvc.model.getBackgroundTile()
            };

            svgApi.registerImagePattern(options);
        },

        renderOverflow: function () {
            this.mvc.view.$().css('overflow', 'visible');
        },

        applyBorder: function () {
            if (this.isAddedToDom) {
                this.renderShape(this.clientId(), false);
            }
        },

        applyShadow: function () {
            if (this.isAddedToDom) {
                this.renderShadow(this.clientId());
            }
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            var value = this.mvc.model.get(event.property);

            switch (event.property) {
                case Enums.ParameterRoles.BORDER_GEOMETRY:
                    this.onBorderGeometryChanged(value);
                    break;
                case Enums.ParameterRoles.BACKGROUND_COLOR:
                    this.applyShadow();
                    break;
            }
        },

        onBorderGeometryChanged: function (value) {
            if (this.isAddedToDom) {
                var clientId = this.clientId();
                this.renderShape(clientId, true);
                this.renderBackground(clientId);
            }
        },

        onBackgroundColorChanged: function (value) {
            if (this.isAddedToDom) {
                this.renderColorBg(this.clientId());
            }
        },

        onBackgroundTileChanged: function (value) {
            if (this.isAddedToDom) {
                this.renderImageBg(this.clientId());
            }
        },

        onResourceChanged: function (value) {
            if (this.isAddedToDom) {
                this.renderImageBg(this.clientId());
            }
        },

        applyWidth: function () {
            this._super();
            this._forceRedraw();
        },

        applyHeight: function () {
            this._super();
            this._forceRedraw();
        },

        _forceRedraw: function () {
            //хром саботирует обновление геометрии при изменении размеров
            if (this.isWebkit) {
                Utilites.forceRedraw(this.mvc.view.$()[0]);
            }
        }
    });

    return GeometryController;
});
