﻿// <copyright file="MessageHolder.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

define(['when', 'libs/sort', 'base/EventTarget', 'common/Error',
    'common/Enums'],
    function (when, Sort, EventTarget, Error, Enums) {

        var MessageHolder = Class.extend({
            keyFieldName: 'EventId',
            defaultSortField: 'ActiveTime',
            activeTimeField: 'ActiveTime',
            defaultSortFieldType: Enums.parameterType.date,
            MESSAGES_UPDATED: 'MessagesUpdated',
            MESSAGE_SELECTED: 'MessageSelected',
            TIME_UPDATED: 'TimeUpdated',
            init: function (journalModel, requestManager) {
                this.keyFieldName = 'EventId';
                this.eventTarget = new EventTarget();
                this.model = journalModel;
                this.requestManager = requestManager;
                this.requestManager.subscribeMessagesChanged(this._onMessagesChanged, this);
                this.requestManager.subscribeArchiveMessagesReceived(this._onArchiveMessagesReceived, this);
                this.requestManager.subscribeTimeUpdated(this._onTimeUpdated, this);
                this.model.subscribePropertyChanged(this.onModelPropertyChanged, this);

                this.messageContainer = {
                    messages: [],
                    messageId: {} //по keyFieldName, ссылка на messages
                }

                //пропущенные при выключенной автопрокрутке сообщения
                this.skippedMessageContainer = {
                    messages: [],
                    messageId: {}
                }
                this.rightBorderTimepoint = null;
                this.selectedMessage = null;
            },

            _onMessagesChanged: function (event) {
                var changes = {};
                if (this.model.getAutoScroll()) {
                    //обновляем/добавляем/удаляем
                    changes = this._processNUEnabledAutoscroll(event.newOrUpdated, false);
                    if (!this.useArchive) {
                        event.newOrUpdated.forEach(function(message){
                            changes.deletedMessages=changes.deletedMessages.concat(this._processDeleted(this._getMessagesWithSameItemId(message)));
                        }, this);
                    //    this.deleteMessages(changes.deletedMessages);
                    }
                    changes.deletedMessages = changes.deletedMessages.concat(this._processDeleted(event.deleted));
                } else {
                    //только обновляем    
                    var result = this._processNUDisabledAutoscroll(event.newOrUpdated, false);
                    changes.updatedMessages = result.updatedMessages;
                    changes.skippedMessages = result.skippedNewMessages;
                    //this.messageContainer.messages.concat(this.skippedMessageContainer.messages);
                }

                changes.isArchive = false;
                this.fireMessagesUpdated(changes);
            },

            _onTimeUpdated: function (event) {
                if (this.model.getAutoScroll() || (!this.model.getRightBorder())) { //на тот случай если границы еще не инициализировались
                    this.model.setRightBorder(event.serverTime, false);
                    this.model.setLeftBorder(event.serverTime - this.model.getInterval(), false);
                    if (!this.currentTime) {
                        this.deleteAll();
                        //перед стартом запрашиваем сообщения с последней страницы
                        this.loadArchiveByInterval(this.model.getLeftBorder(), this.model.getRightBorder());
                    }
                }
                this.currentTime = event.serverTime;
            },

            loadArchiveByInterval: function (startBorder, endBorder) {
                if (!this.useArchive) {
                    return when.resolve();
                }
                return this.requestManager.getArchiveEvents(startBorder, endBorder, 0);
            },

            loadArchive: function (inverse, limitBorder) {
                if (!this.useArchive) {
                    return when.resolve();
                }
                var startBorder, endBorder;
                //inverse - листание назад
                if (inverse) {
                    startBorder = limitBorder || 0;
                    //-1 - что бы не получить сообщение,
                    //которое уже на странице от предыдущего запроса
                    endBorder = this.model.getLeftBorder() - 1;
                } else {
                    //листание вперед
                    //+1 - что бы не получить сообщение,
                    //которое уже на странице от предыдущего запроса
                    startBorder = this.model.getRightBorder() + 1;
                    endBorder = 0;
                }

                var count = this.model.getPageMessageCount();

                if (count === 0) {
                    return when.resolve();
                }

                //запрашиваем полную страницу
                return this.requestManager.getArchiveEvents(startBorder, endBorder, count, inverse)
                    .then(function (messages) {
                        if (messages && messages.length > 0) {
                            var rightTime, leftTime;
                            if (inverse) {
                                rightTime = messages[0][this.activeTimeField].getTime();
                                leftTime = messages[messages.length - 1][this.activeTimeField].getTime()

                                this._updateBorders(leftTime, rightTime);
                            } else {
                                rightTime = messages[messages.length - 1][this.activeTimeField].getTime();
                                leftTime = messages[0][this.activeTimeField].getTime();
                                this._updateBorders(leftTime, rightTime);                               
                            }

                        }

                        return messages;

                        //return when.resolve();
                    }.bind(this));
            },

            _updateBorders: function (leftTime, rightTime) {
                //сначала идут последние, потом первые
                //самое новое - правая граница страницы                
                if (rightTime > this.model.getRightBorder()) {
                    this.model.setRightBorder(rightTime);
                }
                //самое старое - левая граница страницы                
                if (leftTime < this.model.getLeftBorder()) {
                    this.model.setLeftBorder(leftTime);
                }
            },

            getArchiveLeftBorder: function () {
                return this.requestManager.getArchiveLeftBorder().then(function (row) {
                    return row ? row[this.activeTimeField].getTime() : 0;
                }.bind(this));
            },

            _onArchiveMessagesReceived: function (event) {
                var changes = this._processNUEnabledAutoscroll(event.newOrUpdated, true);
                changes.isArchive = true;
                this.fireMessagesUpdated(changes);
            },

            _processNUEnabledAutoscroll: function (messages, isFromArchive) {
                var result = {
                    updatedMessages: [],
                    newMessages: [],
                    deletedMessages: []
                };

                messages.forEach(function (message) {
                    if (!this._messageExists(message)) {
                        //режим автоскролла - добавляем новые
                        result.newMessages.push(this._addMessage(message, false));
                    } else {
                        //существующее приоритетней архивного
                        if (!isFromArchive) {
                            result.updatedMessages.push(this._updateMessage(message, false));
                        }
                    }

                }, this);

                return result;
            },

            _processNUDisabledAutoscroll: function (messages, isFromArchive) {
                var result = {
                    updatedMessages: [],
                    skippedNewMessages: []
                };

                messages.forEach(function (message) {

                    if (this._messageExists(message)) {
                        //режим без автоскролла - только обновляем
                        //существующее приоритетней архивного
                        if (!isFromArchive) {
                            result.updatedMessages.push(this._updateMessage(message, false));
                        }
                    } else {
                        if (this.skippedMessageContainer.messageId[message[this.keyFieldName]] === undefined) {
                            this.skippedMessageContainer.messageId[message[this.keyFieldName]] = message;

                            result.skippedNewMessages.push(message);
                        }
                    }
                }, this);

                return result;
            },
            deleteAll: function(){
                var deletedMessages = this.messageContainer.messages;
                this.messageContainer.messages = [];
                for (index = 0; index < deletedMessages.length; index++) {
                    delete this.messageContainer.messageId[deletedMessages[index][this.keyFieldName]];
                }
                this.fireMessagesUpdated({ deletedMessages: deletedMessages });
            },

            _processDeleted: function (messageIds) {
                var deletedMessages = [];
                messageIds.forEach(function (messageId) {
                    deletedMessages.push(this._removeMessageByKeyFieldValue(messageId));
                }, this);

                return deletedMessages;
            },

            _messageExists: function (message) {
                return this.messageContainer.messageId[message[this.keyFieldName]] !== undefined;
            },

            _getMessagesWithSameItemId: function (message) {
                var result = [];
                for (eventId in this.messageContainer.messageId) {
                    if (eventId.substring(0, 8) == message.EventId.substring(0, 8) && (eventId != message.EventId))
                        result.push(eventId);
                }
                return result;
            },
            _addMessage: function (message) {
                this.messageContainer.messageId[message[this.keyFieldName]] = message;
                this.messageContainer.messages.push(message);
                return message[this.keyFieldName];
            },

            _updateMessage: function (message) {
                //keep reference
                var currentMessage = this.messageContainer.messageId[message[this.keyFieldName]];
                for (prop in currentMessage) {
                    if ((message[prop] == null) && (prop != 'AckedTime')) {
                        Error.info("Поле " + prop + " сообщения " + message.EventId + " равно null, вместо него останется старое значение " + currentMessage[prop]);
                    } else {
                        currentMessage[prop] = message[prop];
                    }
                }

                return currentMessage;
            },

            _getMessageCount: function () {
                return this.messageContainer.messages.length;
            },

            hasMessages: function () {
                return this.messageContainer.messages.length > 0;
            },

            _getLastMessageTime: function () {
                var msg = _.min(this.messageContainer.messages, function (message) {
                    return message[this.activeTimeField].getTime();
                }, this);

                return msg[this.activeTimeField].getTime();
            },

            _removeMessageByKeyFieldValue: function (messageId) {
                var deleted = null;

                if (this.messageContainer.messageId[messageId]) {
                    var ctx = this;
                    var arrMessageIndex = _.findIndex(this.messageContainer.messages, function (msg) {
                        return (msg[ctx.keyFieldName] == messageId);
                    });

                    if (arrMessageIndex === -1) {
                        console.warn('Message not found in array ' + messageId);
                    } else {
                        this.messageContainer.messages.splice(arrMessageIndex, 1);
                        deleted = this.messageContainer.messageId[messageId];
                        delete this.messageContainer.messageId[messageId];
                    }

                } else {
                    console.warn('Cannot delete message ' + messageId);
                }

                return deleted;
            },

            _onRightBorderChanged: function () {
                //удаляем сообщение вне интервала пр          
                //this._deleteOldMessaged();
            },

            deleteOldMessaged: function (forth) {
                var deletedMessages = [];
                var index;
            //    if (this._getMessageCount() <= this.model.getPageMessageCount()) { //сообщений в контейнере меньше страницы
            //        return;
            //    }
                var num_to_del;
                if (forth) {
                    this.model.setLeftBorder(this.model.getRightBorder() - this.model.getInterval());
                    for (index = this.model.getPageMessageCount(); index < this._getMessageCount() ; index++) {
                        if (this.messageContainer.messages[index][this.activeTimeField].getTime() < this.model.getLeftBorder()) {
                            num_to_del = index;
                            break;
                        }
                    }
                    if (!num_to_del)
                        return;
                    deletedMessages = this.messageContainer.messages.slice(num_to_del);
                    this.messageContainer.messages = this.messageContainer.messages.slice(0, num_to_del);
                    if (num_to_del && num_to_del == (this.model.getPageMessageCount())) {
                        this.model.setLeftBorder(this.messageContainer.messages[this._getMessageCount() - 1][this.activeTimeField].getTime());
                    }
                }
                else {
                    this.model.setRightBorder(this.model.getLeftBorder() + this.model.getInterval());
                    for (index = this._getMessageCount() - this.model.getPageMessageCount()-1; index >= 0 ; index--) {
                        if (this.messageContainer.messages[index][this.activeTimeField].getTime() > this.model.getRightBorder()) {
                            num_to_del = index;
                            break;
                        }
                    }
                    if (!num_to_del)
                        return;
                    deletedMessages = this.messageContainer.messages.slice(0, num_to_del+1); //slice[a1,a2)
                    this.messageContainer.messages = this.messageContainer.messages.slice(num_to_del+1);
                    if (num_to_del && num_to_del == (this._getMessageCount() - this.model.getPageMessageCount()-1)) {
                        this.model.setRightBorder(this.messageContainer.messages[0][this.activeTimeField].getTime());
                    }
                }
                if (deletedMessages.length > 0) {
                    for (index = 0; index < deletedMessages.length; index++) {
                        delete this.messageContainer.messageId[deletedMessages[index][this.keyFieldName]];
                    }
                    this.fireMessagesUpdated({ deletedMessages: deletedMessages });
                }
            },
            deleteMessages: function (deletedMessages) {
                for (index = 0; index < deletedMessages.length; index++) {
                    delete this.messageContainer.messageId[deletedMessages[index][this.keyFieldName]];
                }
            },

            onModelPropertyChanged: function (event) {
                switch (event.property) {
                    case Enums.ParameterRoles.RIGHT_BORDER:
                        this._onRightBorderChanged();
                        break;
                    case Enums.ParameterRoles.AUTOSCROLL:
                        this._onAutoscrollChanged();
                        break;
                }
            },

            _onAutoscrollChanged: function () {
                if (this.model.getAutoScroll()) {
                    this.skippedMessageContainer = {
                        messages: [],
                        messageId: {}
                    }
                }
            },
            clear: function () {
                this.messageContainer.messages = [];
                this.messageContainer.messageId = {};
                this.fireMessagesUpdated({ refresh: true });
            },

            sort: function (field, sortOrder) {

                var fieldT = this._getFieldType(field);
                var sortFunc = undefined;
                var compareFunc = undefined;

                var ascCompare = function (a, b) {
                    return a < b ? -1 : a > b ? 1 : 0;
                };
                var descCompare = function (a, b) {
                    return a > b ? -1 : a < b ? 1 : 0;
                };

                switch (sortOrder) {
                    case Enums.SortType.Desc:
                        compareFunc = descCompare;
                        break;

                    case Enums.SortType.Asc:
                        compareFunc = ascCompare;
                        break;

                    case Enums.SortType.None:
                    case Enums.SortType.No:
                        fieldT = this.defaultSortFieldType;
                        field = this.defaultSortField;
                        compareFunc = ascCompare;
                        break;
                }

                switch (fieldT) {
                    case Enums.parameterType.number:
                        sortFunc = function (a, b) {
                            return compareFunc(parseFloat(a[field]), parseFloat(b[field]));
                        }
                        break;
                    case Enums.parameterType.date:
                        sortFunc = function (a, b) {
                            a = new Date(a[field]);
                            b = new Date(b[field]);
                            return compareFunc(a, b);
                        }
                        break;
                    case Enums.parameterType.boolean:
                        sortFunc = function (a, b) {
                            return (a === b) ? 0 : a ? -1 : 1;
                        }
                        break;
                    default:
                        sortFunc = function (a, b) {
                            return compareFunc(a[field], b[field]);
                        }
                        break;
                }

                this.messageContainer.messages = Sort(this.messageContainer.messages, sortFunc);

                return this.messageContainer.messages;
            },
            getFirstMessageTime: function () {
                if (this.messageContainer.messages.length)
                    return this.messageContainer.messages[0][this.activeTimeField].getTime();
                Error.warn(String.format('Ошибка в getFirstMessageTime: Пустой контейнер сообщений'));
                return 0;
            },
            getLastMessageTime: function () {
                if (this.messageContainer.messages.length)
                    return this.messageContainer.messages[this.messageContainer.messages.length - 1][this.activeTimeField].getTime();
                Error.warn(String.format('Ошибка в getLastMessageTime: Пустой контейнер сообщений'));
                return 0;
            },
            getMessages: function (start, end) {
                var startIndex = 0,
                    endIndex = this.messageContainer.messages.length;

                if (start >= 0 && start < this.messageContainer.messages.length)
                    startIndex = start;

                if (end && end < this.messageContainer.messages.length && end > startIndex)
                    return this.messageContainer.messages.slice(startIndex, end);

                return this.messageContainer.messages.slice(startIndex);
            },

            selectMessage: function (eventId, suppressWarning) {
                this.selectedMessage = this.messageContainer.messageId[eventId];

                if (this.selectedMessage === undefined && !suppressWarning) {
                    Error.warn(String.format('Selected message with eventId:{0} not found', eventId));
                }

                //console.log(eventId);
                this.fireMessageSelected(this.selectedMessage);
            },

            confirmMessage: function (eventId, comment) {
                this.requestManager.acknowledgeEvents([eventId], comment);
            },

            confirmAllMessages: function (comment) {
                var eventIds = _.map(this.messageContainer.messages, function (message) {
                    return message[this.keyFieldName];
                }, this);

                return this.requestManager.acknowledgeEvents(eventIds, comment);
            },

            subscribeMessagesUpdated: function (handler, context) {
                this.eventTarget.addListener(this.MESSAGES_UPDATED, handler, context);
            },

            fireMessagesUpdated: function (changes) {
                this.eventTarget.fire({
                    type: this.MESSAGES_UPDATED,
                    target: this,
                    deletedMessages: changes.deletedMessages,
                    updatedMessages: changes.updatedMessages,
                    newMessages: changes.newMessages,
                    refresh: changes.refresh ? false : true,
                    isArchive: changes.isArchive ? changes.isArchive : false,
                    skippedMessages: changes.skippedMessages
                });
            },

            unsubscribeMessagesUpdated: function (handler) {
                this.eventTarget.removeListener(this.MESSAGES_UPDATED, handler);
            },

            subscribeMessageSelected: function (handler, context) {
                this.eventTarget.addListener(this.MESSAGE_SELECTED, handler, context);
            },

            fireMessageSelected: function (message) {
                this.eventTarget.fire({
                    type: this.MESSAGE_SELECTED,
                    target: this,
                    message: message
                });
            },

            unsubscribeMessageSelected: function (handler) {
                this.eventTarget.removeListener(this.MESSAGE_SELECTED, handler);
            },

            _getFieldType: function (field) {
                return this.requestManager.getFieldType(field);
            }
        });

        return MessageHolder;
    });
