﻿// <copyright file="LineController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class LineController controller class for Line control
*/

define(['common/Enums', 'common/Appearence', 'base/ControlController',
     'libs/svg/jquery_svg_drawing'],
function (Enums, Appearence, ControlController, svgApi) {

    var LineController = ControlController.extend({

        isAddedToDom: false,

        init: function () {
            this._super();

            this.ClassName = 'LineController';
        },

        onAddedToDOM: function () {
            this._super();

            this.isAddedToDom = true;

            var clientId = this.clientId();
            this.mvc.view.$().attr('id', clientId);
            this.renderShapeInternal(clientId);
            this.renderShadow(clientId);
        },

        getBrushPatternId: function () {
            return this.clientId() + '_bg';
        },

        renderShapeInternal: function (clientId) {
            var svg = svgApi.getSVG(clientId);

            var lineCoords = this.calcLineCoords();

            var lineStyle = {
                vectorEffect: 'non-scaling-stroke'
            };

            svg.line(lineCoords.x1, lineCoords.y1, lineCoords.x2, lineCoords.y2, lineStyle);

            this.updateSvgSize();
            this.applyBorder();
        },

        calcSvgSize: function () {
            var controlWidth = this.mvc.model.getWidth();
            var controlHeight = this.mvc.model.getHeight();
            var svgSize = { width: '100%', height: '100%' };

            if (controlWidth === 0) {
                svgSize.width = this.mvc.model.getBorderThickness() +
                                this.mvc.model.getShadowSize();
            }

            if (controlHeight === 0) {
                svgSize.height = this.mvc.model.getBorderThickness() +
                                this.mvc.model.getShadowSize();
            }

            return svgSize;
        },

        calcLineCoords: function () {
            var controlWidth = this.mvc.model.getWidth(),
                controlHeight = this.mvc.model.getHeight(),
                bt = this.mvc.model.getBorderThickness()

            var lineCoords = { x1: 0, x2: '100%', y1: 0, y2: '100%' };

            if (controlWidth <= 0 && controlHeight >= 0) {
                lineCoords.x1 = 0;
                lineCoords.y1 = '100%';
                controlWidth == 0 ? lineCoords.x2 = 0: lineCoords.x2 = '100%';
                lineCoords.y2 = 0;
            }
            else if (controlHeight <= 0 && controlWidth >= 0) {
                lineCoords.x1 = 0;
                controlHeight == 0 ? lineCoords.y1 = 0 : lineCoords.y1 = '100%';
                controlWidth == 0 ? lineCoords.x2 = 0 : lineCoords.x2 = '100%';
                lineCoords.y2 = 0;
            }
            else if (controlHeight <= 0 && controlWidth <= 0) {
                lineCoords.x1 = 0;
                lineCoords.y1 = 0;
                lineCoords.x2 = '100%';
                lineCoords.y2 = '100%';
            }

            else if (controlHeight >= 0 && controlWidth >= 0) {
                lineCoords.x1 = 0;
                lineCoords.y1 = 0;
                lineCoords.x2 = '100%';
                lineCoords.y2 = '100%';
            }

            return lineCoords;
        },

        updateSvgSize: function () {
            var svg = svgApi.getSVG(this.clientId());
            var svgSize = this.calcSvgSize();
            svg.change(svg.root(), { width: svgSize.width, height: svgSize.height });
        },

        updateLinePosition: function () {
            var svg = svgApi.getSVG(this.clientId());

            var lineCoords = this.calcLineCoords();

            $('line', svg.root()).attr('x1', lineCoords.x1)
                                 .attr('y1', lineCoords.y1)
                                 .attr('x2', lineCoords.x2)
                                 .attr('y2', lineCoords.y2);
        },
        renderBorder: function () {

        },

        renderShadow: function (clientId) {

            var options = {
                color: this.mvc.model.getShadowColor(),
                offset: this.mvc.model.getShadowSize()
            };

            var svg = svgApi.getSVG(clientId);

            this.removeShadow(clientId, svg);

            if (options.offset !== 0) {
                this.addShadow(clientId, options, svg);
            }
        },

        addShadow: function (clientId, options, svg) {
            var filterId = clientId + '_shadow';

            var colorObj = Appearence.color.ARGBtoObj(options.color);
           
            //do not draw shadow without colour
            if (colorObj) {
                var filter = svg.filter(filterId, '0', '0', '100%', '100%',
                                        { filterUnits: 'userSpaceOnUse', primitiveUnits: 'userSpaceOnUse' });
                svg.filters.componentTransfer(filter, '', [['discrete', [colorObj.r]],
                                                           ['discrete', [colorObj.g]],
                                                           ['discrete', [colorObj.b]]]);
                svg.filters.gaussianBlur(filter, '', '', 2);
                svg.filters.offset(filter, 'offsetBlur', '', options.offset, options.offset);
                svg.filters.composite(filter, '', 'over', 'SourceGraphic', 'offsetBlur');

                $('line', svg.root()).attr('filter', 'url(#' + filterId + ')');
            }
        },

        removeShadow: function (clientId, svg) {
            $('line', svg.root()).attr('filter', '');
            $('filter', svg.root()).remove();
        },

        renderOverflow: function () {
            this.mvc.view.$().css('overflow', 'visible');
        },

        applyIsEnabled: function () {
            if (!this.actualEnabled()) {
                this.mvc.view.$().attr('disabled', 'true');
                this.mvc.view.$().addClass('disabled');

            } else {
                this.mvc.view.$().removeAttr('disabled');
                this.mvc.view.$().removeClass('disabled');
            }
        },

        applyBorder: function () {
            if (!this.isAddedToDom) {
                return;
            }

            var svg = svgApi.getSVG(this.clientId());

            var borderWidth = this.mvc.model.getBorderThickness();
            var lineSpacing = 0;

            var lineStyle = {
                stroke: Appearence.color.toCssColor(this.mvc.model.getBorderColor()),
                strokeWidth: this.mvc.model.getBorderThickness() + 'px'
            };

            switch (this.mvc.model.getBorderStyle()) {
                case Enums.BorderStyleType.Dash:
                    lineSpacing = 5 * borderWidth;
                    lineStyle.strokeDasharray = [lineSpacing, lineSpacing].join(' ');
                    break;
                case Enums.BorderStyleType.Dot:
                    lineSpacing = 1.25 * borderWidth;
                    lineStyle.strokeDasharray = [lineSpacing, lineSpacing].join(' ');
                    lineStyle.strokeLineCap = 'round';
                    break;
                case Enums.BorderStyleType.None:
                    lineStyle.strokeWidth = '0px';
                    break;
                default:
                    lineStyle.strokeDasharray = '';
            }

            this.updateSvgSize();
        //    this.updateLinePosition();
            $('line', svg.root()).css(lineStyle);
        },

        applyShadow: function () {
            if (this.isAddedToDom) {
                this.renderShadow(this.clientId())
            }
        },

        onWidthChanged: function (value) {

            this._super(value);

            if (!this.isAddedToDom) {
                return;
            }

            this.updateSvgSize();
            this.updateLinePosition();
        },

        onHeightChanged: function (value) {

            this._super(value);

            if (!this.isAddedToDom) {
                return;
            }

            this.updateSvgSize();
            this.updateLinePosition();
        },

        onBackgroundColorChanged: function (value) {

        },

        onBackgroundTileChanged: function (value) {

        },

        onResourceChanged: function (value) {

        },

    });

    return LineController;
});
