﻿// <copyright file="SliderController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class SliderController controller class for Slider control
*/

define(['common/Enums', 'common/Appearence', 'base/ControlController'],
function (Enums, Appearence, RangeControlController) {
    //here we will define controller for Slider

    var SliderController = RangeControlController.extend({

        init: function () {
            this._super();

            this.ClassName = 'SliderController';
            this.initialized = false;
        },
   
        onModelInit: function (event) {            
            this._super();

            var self = this;

            this.mvc.view.$().slider({
                orientation: sliderOrientation(this.mvc.model.getOrientation()),
                min: this.mvc.model.getMinValue(),
                max: this.mvc.model.getMaxValue(),
                value: this.mvc.model.getValue(),
                angleRotation: this.mvc.model.get(Enums.ParameterRoles.ANGLE),
                disabled: !this.mvc.model.getIsEnabled(),                
                animate: 'fast',
                change: function (event, ui) { self.onChange(event, ui); }
            });

			// Drop the default background
			this.mvc.view.$('.ui-slider-handle')
                .css('background', '');
			this.applyForeColor(this.mvc.model.getTextColor());
			this.initialized = true;
        },

        attachEvents: function () {
            this._super();
            var self = this;
            this.mvc.view.$().mousewheel(function (e, delta) {
                self.onWheel(e, delta);
            }.bind(this));
        },

        modelPropertyChangedInternal: function (event) {
            this._super(event);
            switch (event.property) {
                case Enums.ParameterRoles.WIDTH:
                    if (this.mvc.model.getOrientation() === Enums.OrientationType.Vertical) {
                        var margin = (-50 * (this.mvc.view.$('.ui-slider-handle').height() / this.mvc.view.$('.ui-slider-handle').width()));
                        this.mvc.view.$('.ui-slider-handle').css({
                            'margin-bottom': margin + '%'
                        });
                    }
                    break;
                case Enums.ParameterRoles.MIN_VALUE:
                case Enums.ParameterRoles.MAX_VALUE:
                    this.onMinMaxValueChanged();
                    break;
                case Enums.ParameterRoles.VALUE:
                    this.onValueChanged();
                    break;
                case Enums.ParameterRoles.TEXT_COLOR: //это не цвет текста, это цвет ползунка
                    this.onForeColorChanged();
                    break;
            }
        },
        onAddedToDOM: function (event) {
            this._super(event);
            if (this.initialized === true) {
                var size = this.mvc.model.getSliderRelativeSize();
                if (this.mvc.model.getOrientation() === Enums.OrientationType.Horizontal) {
                    this.mvc.view.$('.ui-slider-handle').css({
                        'width': size + '%',
                        'margin-left': -size / 2 + '%'
                    });
                } else {
                    //if directional is vertical 50% need us for hide half slider in max/min positions
                    this.mvc.view.$('.ui-slider-handle').css({
                        'height': size + '%'
                    });
                    var margin = (-50 * (this.mvc.view.$('.ui-slider-handle').height() / this.mvc.view.$('.ui-slider-handle').width()));
                    this.mvc.view.$('.ui-slider-handle').css({
                        'margin-bottom': margin + '%'
                    });
                }
            }
        },

		applyForeColor: function (value) {
			if (Appearence.color.isColor(value) || Appearence.color.isARGB(value))
                this.mvc.view.$('.ui-slider-handle')
                .css('background', Appearence.color.toCssColor(value));
        },
		onForeColorChanged: function(value){
		    this.applyForeColor(this.mvc.model.getTextColor());
		},
		onValueChanged: function () {
		    this.settingValue = true;
		    this.mvc.view.$().slider("option", "value", this.mvc.model.get(Enums.ParameterRoles.VALUE));
            this.settingValue = false;
		},

        onAngleChanged:function(value){
            this._super(value);
            this.mvc.view.$().slider("option", "angleRotation",
                this.mvc.model.get(Enums.ParameterRoles.ANGLE));
        },

        onMinMaxValueChanged: function () {
            if (this.mvc.model.get(Enums.ParameterRoles.MIN_VALUE) < this.mvc.model.get(Enums.ParameterRoles.MAX_VALUE)) {
                this.mvc.view.$().slider("option", "min", this.mvc.model.get(Enums.ParameterRoles.MIN_VALUE));
                this.mvc.view.$().slider("option", "max", this.mvc.model.get(Enums.ParameterRoles.MAX_VALUE));
            } else {
                console.warn(String.format("У ползунка {0} нижняя граница больше правой: {1}-{2}", this.mvc.model.getId(), this.mvc.model.get(Enums.ParameterRoles.MIN_VALUE), this.mvc.model.get(Enums.ParameterRoles.MAX_VALUE)))
            }
        },

        onOrientationChanged: function (value, event) {
            this.mvc.view.$().slider("option", "orientation", sliderOrientation(value));
            this.onAddedToDOM(event);
        },

        onTextColorChanged: function (value) {
            this.applyForeColor(value);
        },

        onIsEnabledChanged: function (value) {
            this._super(value);

            if (value) {
                this.mvc.view.$().slider('enable');
            }
            else {
                this.mvc.view.$().slider('disable');
            }
        },

        //
        // Events
        //      
        onWheel: function (e, delta) {
            var value = this.mvc.model.getValue() + delta;
            if ((value > this.mvc.model.getMaxValue()) || value < this.mvc.model.getMinValue()) {
                return;
            }
            else {
                this.mvc.model.setValue(value);
            }
        },

        onChange: function (event, ui) {

            if (this.settingValue) {
                return;
            }
            this.mvc.model.setValue(ui.value);
        },
    });

    /*
    * @method sliderOrientation returns slider orientation by enum
    */
    function sliderOrientation(orientation) {
        return orientation === Enums.OrientationType.Horizontal ? 'horizontal' : 'vertical';
    }

    return SliderController;
});
