﻿// <copyright file="TabControlController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


/*
* @class TabControlController controller class for TabControl control
*/
define(['common/Enums', 'common/Appearence', 'base/CanvasContainerController'],
function (Enums, Appearence, CanvasContainerController) {

    var TabControlController = CanvasContainerController.extend({

        //contains coefficient for calculation actual font size
        //fontSizeRatio == fontSize/width 
        //where width = control.outerWidth 
        //      fontSize = (fontSize from model)
        fontSizeRatio: null,
        tabsInitialized: false,

        init: function () {
            this._super();

            this.ClassName = 'TabControlController';
            this.tabTitleHeight = 0;
            this.tabTitleWidth = 0;
        },     

        onAddedToDOM: function () {
            this._super();

            //call it here because element will have width after it will be added to DOM
            this.calculateFontSizeRatio();
            this.applyFontSize();

            //add tabs
            this.addTabs();

            //init tab control
            this.mvc.view.$().tabs({
                heightStyle: "fill",
                orientation: this.getTabsOrientation(),
                textOrientation: this.getTextOrientation(),
                activate: function (event, ui) {
                    this.onParentActualWidthChanged(event);
                }.bind(this)
            });
            this.enable(this.mvc.model.getIsEnabled());

            this.tabsInitialized = true;
            this.repositionTabs();
        },

        onChildrenAdded: function (event) {
            this._super(event);
        },

        textElement: function () {
            return this.mvc.view.$('> ul');
        },
        applyTextShadow: function () {

            var shadowColor = this.mvc.model.getTextShadowColor();
            var shadowSize = this.mvc.model.getTextShadowSize();

            Appearence.shadow.applyTextShadow(this.textElement(), { color: shadowColor, size: shadowSize });
        },

        applyFontSize: function () {
            var actualSize = this.fontSizeRatio * this.mvc.view.$().width();

            this.textElement().css('font-size', actualSize + 'px');
            //this.textElement().css('font-size', this.mvc.model.getFontSize() + 'px');

            this.onActualFontSizeChanged(actualSize);
        },

        calculateFontSizeRatio: function () {
            var width = this.mvc.model.get(Enums.ParameterRoles.WIDTH);

            if (width == 0) {
                this.fontSizeRatio = 0;
            }
            else {
                this.fontSizeRatio = this.mvc.model.getFontSize() / width;
            }
        },

        addTabs: function () {
            var i,
                child,
                tabId,
                a,
                li,
                ul = this.mvc.view.$('ul');

            for (i = 0; i < this.orderedControls.length; i++) {
                child = this.orderedControls[i];
                tabId = child.controller.clientId();
                child.view.$().attr('id', child.controller.clientId());
                //child.controller.setTabId(tabId);

                a = $(document.createElement('a'))
                    .attr('href', '#' + tabId)
                    .text(child.model.get(Enums.ParameterRoles.TEXT));
                li = $(document.createElement('li')).append(a);
                ul.append(li);
            }
        },

        adjustPanelsHeight: function () {
            if (this.tabsInitialized && this.mvc.view.$().data("ui-tabs")) {
                this.mvc.view.$().tabs("option", "heightStyle", "fill");
            }
        },

        enable: function (value) {
            if (value) {
                this.mvc.view.$().tabs("enable");
            }
            else {
                this.mvc.view.$().tabs("disable");
            }
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            var value = this.mvc.model.get(event.property);

            switch (event.property) {
                case Enums.ParameterRoles.TEXT_COLOR:
                    this.onTextColorChanged(value);
                    break;
                case Enums.ParameterRoles.FONT_NAME:
                    this.onFontNameChanged(value);
                    break;
                case Enums.ParameterRoles.FONT_SIZE:
                    this.onFontSizeChanged(value);
                    break;
                case Enums.ParameterRoles.FONT_ITALIC:
                    this.onFontItalicChanged(value)
                    break;
                case Enums.ParameterRoles.FONT_BOLD:
                    this.onFontBoldChanged(value)
                    break;
                case Enums.ParameterRoles.FONT_UNDERLINED:
                    this.onFontUnderlinedChanged(value);
                    break;
                case Enums.ParameterRoles.TEXT_SHADOW_COLOR:
                    this.onTextShadowColorChanged(value);
                    break;
                case Enums.ParameterRoles.TEXT_SHADOW_SIZE:
                    this.onTextShadowSizeChanged(value);
                    break;
            }
        },

        onTextColorChanged: function (value) {
            this.textElement().css('color', value);
        },

        onFontNameChanged: function (value) {
            this.textElement().css('font-family', value);
        },

        onFontSizeChanged: function (value) {

            this.calculateFontSizeRatio();

            this.applyFontSize();
        },

        onFontItalicChanged: function (value) {
            if (value) {
                this.textElement().css('font-style', 'italic');
            } else {
                this.textElement().css('font-style', '');
            }
        },

        onFontBoldChanged: function (value) {
            if (value) {
                this.textElement().css('font-weight', 'bold');
            } else {
                this.textElement().css('font-weight', '');
            }
        },

        onFontUnderlinedChanged: function (value) {
            if (value) {
                this.textElement().css('text-decoration', 'underline');
            } else {
                this.textElement().css('text-decoration', '');
            }
        },

        onTextShadowColorChanged: function (value) {
            this.applyTextShadow();
        },

        onTextShadowSizeChanged: function (value) {
            this.applyTextShadow();
        },

        onIsEnabledChanged: function (value) {

            if (this.tabsInitialized) {
                this.enable(value);
            }
            
            this._super(value);
        },

        onBorderThiknessChanged: function (value) {
            this._super(value);
            this.adjustPanelsHeight();
        },

        onActualFontSizeChanged: function (newValue) {
            this.adjustPanelsHeight();
        },

        fireActualWidthChanged: function (newValue) {
            this.applyFontSize();

            this._super(newValue);
        },

        fireActualHeightChanged: function (newValue) {
            this._super(newValue);

            this.adjustPanelsHeight();
        },

        onParentActualWidthChanged: function (event) {            
            this._super(event);
            this.repositionTabs();
        },

        onParentActualHeightChanged: function (event) {
            this._super(event);
            this.repositionTabs();
        },

        onActualVisibleChanged: function (newValue) {
            this._super(newValue);
            this.repositionTabs();
        },

        getTabsOrientation: function () {
            switch (this.mvc.model.getLayout()) {
                case Enums.LayoutType.Up:
                    return 'up';
                case Enums.LayoutType.Down:
                    return 'down';
                case Enums.LayoutType.Left:
                    return 'left';
                case Enums.LayoutType.Right:
                    return 'right';
                default:
                    throw 'Invalid layout value ' + this.mvc.model.getLayout();
            }
        },

        getTextOrientation: function () {
            switch (this.mvc.model.getOrientation()) {
                case Enums.OrientationType.Horizontal:
                    return 'horizontal';
                case Enums.OrientationType.Vertical:
                    return 'vertical';
            }
        },

        repositionTabs: function(){
            var header =  this.textElement();
            switch (this.mvc.model.getLayout()) {
                case Enums.LayoutType.Up:
                    {
                        this.tabTitleHeight = header.height() + 10;
                        this.tabTitleWidth = 0;
                    }
                    break;
                case Enums.LayoutType.Left: {
                    this.tabTitleHeight = 0;
                    this.tabTitleWidth = header.width();
                }
                    break;
            }

            var i;
            for (i = 0; i < this.orderedControls.length; i++) {
                child = this.orderedControls[i];               
                this.applyChildY(child);
                this.applyChildX(child);
            }
        },         

        getChildY: function (childControl) {                       
            return this.tabTitleHeight;
        },

        getChildX: function (childControl) {          
            return this.tabTitleWidth;
        },

        onBeforeDestroy: function () {
            this._super();
            this.mvc.view.$().tabs("destroy");
        },
    });

    return TabControlController;
});
