﻿// <copyright file="TrendEventWrapper.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// <author>Bogdan Nosachenko b.nosachenko@infostroy-software.com</author>

/*
* @class TrendEventWrapper controller class for events on canvas
*/
define(['when', 'common/Enums', 'controls/Trend/modules/Alghoritms'],
    function (when, Enums, Alghoritms) {

        var TrendEventWrapper = Class.extend({

            init: function (controller, canvas) {
                this.controller = controller;
                this.canvas = canvas;
                this.meterScale = 0;
                this._lastClickTime = 0;
                this.startX = 0;
                this.startCoordsFingers = null;
                this.endCoordsFingers = null;
                this.isCaught = false;
                this.isUpdated = false;
                this.defaultTrendlineRange = this._saveDefaultTrendlineRange();
            },

            nextHalfPage: function () {
                if (!this.controller.mvc.model.getIsAutoScroll()) {
                    var nextmark = this.controller.mark + (this.controller.mvc.model.getInterval() / 2),
                        previous = this.controller.mark - this.controller.mvc.model.getInterval() + this.controller.mvc.model.getInterval() / 2;
                    this.controller._setMark(nextmark);
                    return this.controller.getArchiveData(previous, nextmark);
                }

                return when.resolve();
            },

            previousHalfPage: function () {
                if (!this.controller.mvc.model.getIsAutoScroll()) {
                    var previousMark = this.controller.mark - (this.controller.mvc.model.getInterval() / 2),
                        oldMark = this.controller.mark;
                    //we need to move left point on 2x interval
                    this.controller._setMark(previousMark);
                    return this.controller.getArchiveData(oldMark - this.controller.mvc.model.getInterval() -
                        (this.controller.mvc.model.getInterval() / 2), previousMark);
                }

                return when.resolve();
            },

            onMoveHalfPage: function (coords, eventType) {
                if (eventType === Enums.PanType.panEnd) {
                    return;
                }
                this.controller.isUpdated = true;
                this.controller.mvc.model.stopAutoscroll();
                if (eventType === Enums.SwipeDirection.swipeLeft) {
                    this.nextHalfPage();
                } else if (eventType === Enums.SwipeDirection.swipeRight) {
                    this.previousHalfPage();
                } 
            },

            onPinchStart: function (coords, eventType, event) {
                if (event.changedTouches.length > 1) {
                    this.startCoordsFingers = $.extend(true, [], event.changedTouches);
                    this.isCaught = true;
                }
            },

            onPinchMove: function (coords, eventType, event) {
                if (event.changedTouches.length > 1) {
                    this.endCoordsFingers = event.changedTouches;
                }
            },

            onPinchIn: function () {
                this.meterScale--;
                return true;
            },

            onPinchOut: function () {
                this.meterScale++;
                return true;
            },

            changeXIntervalScale: function (lambda) {
                var c = this.controller.mvc.model.getInterval();
                this.controller.mvc.model.setInterval(c + lambda, Enums.ParametersSource.gesture);
            },

            onPinchEnd: function (coords, eventType, event) {
                if (this.isCaught) {
                    this.controller.mvc.model.stopAutoscroll();
                    var firstDistanceX = (this.startCoordsFingers[1].clientX - this.startCoordsFingers[0].clientX); 
                        secondDistanceX = (this.endCoordsFingers[1].clientX - this.endCoordsFingers[0].clientX); 
                        trendlineScale = Math.abs(this._resultX(secondDistanceX, firstDistanceX) / this.controller.mvc.model.getInterval()),
                        range = this.controller.backLayer.getYRange(),
                        sign = this.meterScale < 0 ? 1 : -1;

                    if (!this.isUpdated) {
                        this._saveDefaultYRange();
                    }

                    this._changeYRange(range, sign * trendlineScale);
                    this._changeTrendlineRange(sign * trendlineScale);
                    this.changeXIntervalScale(sign * Math.abs(this._resultX(secondDistanceX,firstDistanceX)));
                    // this flag need for avoid repeat call of this method in TrendController(for touch devices)
                    this.controller.isUpdated = true;
                    // this one need for saving primary setting
                    this.isUpdated = true; 
                    this.isCaught = false;
                    this.startCoordsFingers = null;
                }

                this.meterScale = 0;
            },

            onCanvasClick: function (coords, eventType) {
                this._lastClickTime++;
                if (this._lastClickTime > 1) {
                    clearTimeout(this._clickTimer);
                    this._lastClickTime = 0;
                    this.controller.mvc.model.enableAutoscroll();
                   /* if (this.isUpdated || this.controller.isSavedYRange) {
                        this._resetDefaultSetting();
                        this.isUpdated = false;
                    }*/
                } else {
                    this._clickTimer = setTimeout(function () {
                        if (this.hideCircle) {
                            return;
                        }
                        this.controller.verticalCursor.onCursorAdded(coords, eventType);
                        this._lastClickTime = 0;
                    }.bind(this), 500);
                }
            },

            onCanvasStartPan: function (coords) {
                this.startX = coords.x;
            },
       
            panDirectionOnInterval: function (value) {
                if (value > 0) {
                    var nextmark = this.controller.mark + value,
                        previousMark = this.controller.mark - this.controller.mvc.model.getInterval() + value;

                    this.controller._setMark(nextmark);
                    return this.controller.getArchiveData(previousMark, nextmark);
                }
                else if (value < 0) {
                    var previousMark = this.controller.mark + value,
                        oldMark = this.controller.mark;

                    this.controller._setMark(previousMark);
                    return this.controller.getArchiveData(oldMark -
                        this.controller.mvc.model.getInterval() + value, this.controller.mark);
                }
            },

            onCanvasEndPan: function (coords, eventType) {
                var scale = this._resultX(this.startX, coords.x);
                this.controller.isUpdated = true;
                this.controller.mvc.model.stopAutoscroll();
                if (!this.controller.mvc.model.getIsAutoScroll()) {
                    this.panDirectionOnInterval(scale);  
                }
            },

            onRightMouseDown: function (coords, eventType, event) {
                if (event.which === 3) {
                    this.startX = coords.x;
                }
            },

            onRightMouseUp: function (coords, eventType, event) {
                if (event.which !== 3) {
                    return;
                }

                if (!this.isUpdated) {
                    this._saveDefaultYRange();
                }

                this.controller.mvc.model.stopAutoscroll();

                var resultX = this._resultX(this.startX, coords.x),
                    trendlineScale = resultX / this.controller.mvc.model.getInterval(),
                    range = this.controller.backLayer.getYRange();

                if (trendlineScale === 0) {
                    return;
                }

                this._changeYRange(range, trendlineScale);
                this._changeXRange(resultX, Enums.EventType.mouseUp);
                this._changeTrendlineRange(trendlineScale);
                this.isUpdated = true;
                this.controller.isUpdated = true;
            },

            dispose: function () {
                this.controller = null;
            },

            _resultX: function (startPoint, endPoint) {
                var width = this.controller.backLayer.getGridDimensions().width,
                    x0 = Alghoritms.getX(startPoint, width, this.controller.mark - this.controller.mvc.model.getInterval(), this.controller.mark),
                    x1 = Alghoritms.getX(endPoint, width, this.controller.mark - this.controller.mvc.model.getInterval(), this.controller.mark);
                return Math.floor(x1 - x0);
            },

            _saveDefaultYRange: function () {
                if (this.controller.isSavedYRange) {
                    // if range has been changed by control, we don't need rewrite this value
                    return; 
                }
                this.defaultYRange = this.controller.backLayer.getYRange();
            },

            _saveDefaultTrendlineRange: function () {
                var i = 0,
                    l = this.controller.graphs.length,
                    trendlineRange = {
                        min: [],
                        max: []
                    }

                for (; i < l; i++) {
                    var serie = this.controller.graphs[i],
                        minY = serie.model.getMin(),
                        maxY = serie.model.getMax();

                    trendlineRange.max.push(maxY);
                    trendlineRange.min.push(minY);
                }
                return trendlineRange;
            },

            _resetDefaultSetting: function () {
                this.controller.mvc.model.resetInterval(this.controller.mvc.model.getInterval());
                //if we didn't change range
                if (!this.defaultYRange) {
                    return;
                }

                this._changeYRange(this.defaultYRange, 0);

                var i = 0,
                    l = this.controller.graphs.length,
                    serie;
                for (; i < l; i++) {
                    serie = this.controller.graphs[i];
                    serie.model.setMinY(this.defaultTrendlineRange.min[i]);
                    serie.model.setMaxY(this.defaultTrendlineRange.max[i]);
                }
            },

            _changeXRange: function (value, eventType) {
                var c = this.controller.mvc.model.getInterval();
                this.controller.mvc.model.setInterval(c + Math.ceil(value), eventType);
            },

            _changeYRange: function (range, value) {
                if(isNaN(value)){
                    value = 0;
                }
                range[1] = (range[1] * (1 + value));
                range[0] = (range[0] * (1 + value));
                if (range[0] > 0.01 || range[1] < 0.01) {
                    return;
                }

                this.controller._updateYAxis({
                    min: range[0].toFixed(2),
                    max: range[1].toFixed(2)
                });
            },

            _changeTrendlineRange: function (value) {
                var i = 0,
                   l = this.controller.graphs.length,
                   serie;

                for (; i < l; i++) {
                    serie = this.controller.graphs[i];
                    serie.model.setMinY(serie.model.getMin() * (1 + value));
                    serie.model.setMaxY(serie.model.getMax() * (1 + value));
                }
            },
        });

        return TrendEventWrapper;
    });