// <copyright file="TrendVariablesManager.js" company="">
// , 2014
// </copyright>
// 


/** 
	* @class TrendVariablesManager
	* @classdesc Represents an object, which holds and manages state of "server variable" abstraction.
	
	*/
define(['server/VariablesManager', 'server/adapters/DataAdapter', 'when', 'core/Timer', 'server/ServerStateManager',
    'common/Enums', 'common/Utilites', 'common/Error'],
    function (VariablesManager, DataAdapter, when, Timer, ServerStateManager, Enums, Utilities, Error) {

        var VAR_NOT_FOUND = 'GetArchiveItems: Variable {0} not found in archive. Status code {1}';

        var TrendVariablesManager = VariablesManager.extend({
            init: function (options) {
                this._super(options);
                this.className = 'TrendVariablesManager';
                this.variables.get.archive = {};
                this.variables.get.archiveInfo = {
                    minStartTime: 0,
                    maxEndTime: 0
                };

                this.requestParamsCache = [];
            },

            initialize: function () {
                this.timer = new Timer();
            },           

            _addGetVariableToSubscription: function (variable) {
                this._super(variable);
                variable.timestampsToReturn = Enums.timeStampReturnType.source;
                variable.queueSize = 0;
            },
            _updateVariable: function (variable, record) {
                variable.statusCode = record.statusCode !== undefined ? record.statusCode : 0;
                variable.sourceTime = +record.sourceTime; //string to number
                // INSAT:  -  12273 (   )
                /*if (Utilities.compare(variable.value, record.value) == 0) {
                    return;
                }*/

                var oldvalue = variable.value;
                variable.value = record.value;
                this.firePropertyChanged(this._getVariablePropertyPath(variable), oldvalue, {                   
                    value: variable.value,
                    sourceTime: +variable.sourceTime,
                    statusCode: variable.statusCode
                });                
            },        

            updateArchiveInfo: function () {
                if (!this._hasGetVariables()) {
                    return when.resolve();
                }
                this.requestParamsCache = this._getArchiveParams(this.variables.get.protocol);
                return this._updateArchiveInfo();
            },

            reatartDataUpdate: function () {
                if (!this._hasGetVariables()) {
                    return when.resolve();
                }

                return this.dataAdapter.refresh();
            },

            getServerTime: function () {
                return this.serverStateManager.get(this.stateVars.serverTime);
            },

            initArchive: function () {
                if (!this._hasGetVariables()) {
                    return when.resolve();
                }

                var ctx = this;              
                return this.dataAdapter.getArchiveItems(this.variables.get.protocol).then(function (items) {
                    var i;
                    for (i = 0; i < items.length; i++) {
                        if (items[i].statusCode !== undefined && items[i].statusCode !== 0) {
                        //    ctx._logVarNotFound(ctx.variables.get.protocol[i][ctx.keyFieldName], items[i]);
                            continue;
                        }

                        ctx.variables.get.protocol[i].archiveItemId = items[i].archiveItemId;
                        ctx.variables.get.archive[items[i].archiveItemId] = ctx.variables.get.protocol[i];
                    }
                    this.requestParamsCache = items.slice(0);
                });
            },

            getArchiveInfo: function () {
                return this.variables.get.archiveInfo;
            },

            _getArchiveParams: function (variables) {
                return variables.map(function (variable) {
                     return {
                        archiveItemId: variable.itemId,
                        continuationPoint: 0
                    };
                });
            },

            _removeVariable: function (variable, removedVariable) {
                this._super(variable, removedVariable);
                removedVariable.usesCounts--;
            },

            _canDeleteSubscription: function () {
                var i;
                for (i = 0; i < this.variables.get.protocol.length; i++) {
                    if (this.variables.get.protocol[i].usesCounts !== 0) {
                        return false;
                    }
                }               

                return true;                
            },
           
            _updateArchiveInfo: function () {
                var params = {
                    startTime: 0,
                    endTime: 0,
                    returnBounds: false,
                    returnFirstLastValue: true
                }
                var ctx = this;

                return this.dataAdapter.getArchiveData(params, this.variables.get.protocol).then(function (items) {

                    var min = Number.POSITIVE_INFINITY,
                        max = Number.NEGATIVE_INFINITY;

                    _.map(items, function (item) {
                        if (item.statusCode !== undefined && item.statusCode !== 0) {
                            var i = _.indexOf(items, item);
                            this._logVarNotFound(ctx.variables.get.protocol[i][ctx.keyFieldName], item);
                        } else if (!item.values || item.values.length < 2) {
                            Error.info(String.format('Cannot get archive for {0}. Status code {1}',
                                item.archiveItemId, item.statusCode || 0));
                        } else {
                            //  ,         
                            //    
                            //   
                            if (+item.values[0][0] < min) {
                                min = +item.values[0][0];
                            }
                            if (+item.values[1][0] > max) {
                                max = +item.values[1][0];
                            }
                        }
                    });

                    ctx.variables.get.archiveInfo.maxEndTime = max;
                    ctx.variables.get.archiveInfo.minStartTime = min;

                });
            },

            getArchiveData: function (startTime, endTime, resampleInterval, storage) {
                if (!this._hasGetVariables()) {
                    return when.resolve();
                }

                var params = {
                    startTime: startTime,
                    endTime: endTime,
                    resampleInterval: resampleInterval
                };

                if (this.requestParamsCache.length === 0) {
                    return when.resolve();
                }
                for (i = 0; i < this.requestParamsCache.length; i++) {
                    this.requestParamsCache[i].continuationPoint = 0;
                }
                return this._getArchiveData(params, this.variables.get.protocol, storage);
            },           

            _getArchiveData: function(params, data, storage){
                var ctx = this;                
                params.startTime = params.startTime ? Math.floor(params.startTime) : params.startTime;
                params.endTime = params.endTime ? Math.ceil(params.endTime) : params.endTime;
                var k = 0;
                return this.dataAdapter.getArchiveData(params, data).then(function (items) {
                    var i, j, nextRequestData = [];
                    for (i = 0; i < items.length; i++) {
                        if (k >= storage.length)
                            break;
                        if (items[i].statusCode !== undefined && items[i].statusCode !== 0) {
                            this._logVarNotFound(ctx.variables.get.protocol.getValues[i][ctx.keyFieldName], items[i]);
                            continue;
                        }

                        if (items[i].areMoreValues === true && items[i].continuationPoint !== undefined) {
                            //   
                            data[i].continuationPoint = items[i].continuationPoint;
                            nextRequestData.push(data[i]);
                        }

                        var values = items[i].values;
                        while (!storage[k].hasDataSource() && k < storage.length) {
                            k++;
                        }
                        storage[k].dataStorage.addPacket(values);
                        k++;
                    }
                    
                    data = null;
                    if (nextRequestData.length > 0) {
                        //console.log('Read more');
                        return ctx._getArchiveData(params, nextRequestData, storage);
                    }
                });
            },

            _hasGetVariables: function(){
                return this.variables.get.protocol.length > 0;
            },

            _logVarNotFound: function (id, item) {
                Error.warn(String.format(VAR_NOT_FOUND, id, item.statusCode || 0));
            },
            _getProperty: function (id, isCopy) {
                if (this.variables.get.schema[id] !== undefined) {
                    return this._getPropertyValue(this.variables.get.schema[id], isCopy);
                } else if (this.variables.set.schema[id] !== undefined) {
                    return this._getPropertyValue(this.variables.set.schema[id], isCopy);
                }
                return "";
            },

            unsubscribe: function (vars) {
                if (vars.getValues !== undefined)
                    return this._super(vars);

                var def = when.defer();
                var i;

                //get -  + 
                var monitoredItemsIds = [];
                if (vars.get !== undefined && vars.get.protocol !== undefined) {
                    for (i = vars.get.protocol.length - 1; i >= 0; i--) { //vars    this.variables,      
                        var variable = vars.get.protocol[i];
                        var removedVariable = this.variables.get.schema[this.getKey(variable)];
                        //.     ,   addToSubscription
                        //  ,          
                        //       
                        removedVariable.value = undefined;
                        // undefined -    ,
                        //       CreateMonitoredDataItems
                        if (removedVariable.monitoredItemId !== undefined) {
                            monitoredItemsIds.push(removedVariable.monitoredItemId);
                        }

                        this._removeVariable(variable, removedVariable);
                    }
                }

                if (monitoredItemsIds.length > 0) {
                    //       -  
                    if (this._canDeleteSubscription() === true) {
                        //console.log('Delete subscription');
                        this.initialized = false;
                        //  , . -   
                        this.timer.remove(this.getAllWrap);
                        this._deleteSubscription().then(function () {
                            def.resolve();
                        });
                    } else {
                        this._unsubscribeVars(monitoredItemsIds).then(function () {
                            def.resolve();
                        }, function () {
                            def.reject('Unsubscribe Vars rejected');
                        });
                    }
                }
                else {
                    def.resolve();
                }

                return def.promise;
            },

        });
        return TrendVariablesManager;
    });