// <copyright file="Graph.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>

define(['base/ObservableObject', 'common/Enums', 'common/Appearence',
    'controls/Trend/modules/TrendlineBase', 'controls/Trend/modules/Borderline',
'controls/Trend/modules/Alghoritms', 'controls/Trend/modules/models/TrendlineModel'],

    function (ObservableObject, Enums, Appearence, TrendlineBase, Borderline, Alghoritms, TrendlineModel) {
        var suffix = 'Changed',
        borderRegEx = /^PenLimits(\[\d+\])\./gi;

        var Trendline = TrendlineBase.extend({

            init: function (idx, parent, trendModel) {
                this.model = new TrendlineModel(idx, trendModel);
                this._super(idx, parent, trendModel);
                this.borderLines = [];
                this.hiBorderLines = [];
                this.lowBorderLines = [];

                var borders = this.model.getBorders();
                for (var j = 0, l = borders.length; j < l; j++) {
                    var borderLine = new Borderline(j, parent, borders[j].value);
                    this.borderLines.push(borderLine);
                    if (borderLine.model.get(Enums.ParameterRoles.FILL_DIRECTION) === Enums.FillDirection.Up) {
                        this.hiBorderLines.push(borderLine);
                    } else {
                        this.lowBorderLines.push(borderLine);
                    }
                }

                //для корректных перекрытий, верхние границы рисуем в обратном порядке
                this.hiBorderLines.reverse();
            },

            modelPropertyChanged: function (event) {
                var p = event.property;
                if (borderRegEx.test(p)) {
                    var idx = +p.match(/\d+/)[0];
                    event.property = event.property.replace(borderRegEx, '');
                    if (this.model.isDataStoring || event.property != Enums.ParameterRoles.DATA_SOURCE) {
                        this.borderLines[idx].model.firePropertyChanged(event.property, event.oldValue, event.newValue);
                    }
                    return;
                }

                switch (event.property) {
                    default:
                        if (this['on' + event.property + suffix] !== undefined) {
                            this['on' + event.property + suffix](event);
                        }
                        break;
                }
            },

            onDataSourceChanged: function (event) {
                if (this.model.isDataStoring && event.newValue && event.newValue.value != undefined && event.newValue.sourceTime) {
                    this.dataStorage.add(event.newValue.value, event.newValue.sourceTime, event.newValue.statusCode);
                }
            },
           
            updateWithRange: function (canvas, canvasWidth, range, interval, mark, currTimestamp) {
                this._update(canvas, canvasWidth, range, interval, mark, currTimestamp);
            },

            update: function (canvas, canvasWidth, range, interval, mark, currTimestamp) {
                var range = this.model.getRange();
                this._update(canvas, canvasWidth, range, interval, mark, currTimestamp);
                this._pointsCache = [];
            },

            updateBorderLines: function (borderLines, canvas, canvasWidth, range, interval, mark, currTimestamp) {
                borderLines.forEach(function (borderLine) {
                    borderLine.update(canvas, canvasWidth, range, interval, mark, currTimestamp);
                    if (borderLine.model.getIsVisible() && this.model.isSelected) {
                        this._drawFill(canvas, borderLine);
                    }
                }, this);
            },

            _update: function (canvas, canvasWidth, range, interval, mark, currTimestamp) {
                if (this.model.getIsVisible()) {
                    this.updateLine(canvas, canvasWidth, range, interval, mark, currTimestamp);
                    this.updateBorderLines(this.hiBorderLines, canvas, canvasWidth, range, interval, mark, currTimestamp);
                    this.updateBorderLines(this.lowBorderLines, canvas, canvasWidth, range, interval, mark, currTimestamp);
                }
            },

            _drawFill: function (canvas, borderLine) {
                var borderLinePoints = borderLine.getPoints();
                var fillColor = Appearence.color.toCssColor(borderLine.model.get(Enums.ParameterRoles.BACKGROUND_COLOR));
                if (this._pointsCache.length < 2) {
                    return;
                }

                var intersectPoint = null; //[x, y]
                var lastBorderIntersectIndex = 0;
                var lastLineIntersectIndex = 0;
                var wasIntersected = false;
                var i, j;
                //для каждого отрезка линии, проходим по всем отрезкам границы
                for (i = 0; i < this._pointsCache.length - 1; i++) {
                    for (j = 0; j < borderLinePoints.length - 1; j++) {
                        
                        var lfp = this._pointsCache[i],
                            lsp = this._pointsCache[i + 1],
                            bfp = borderLinePoints[j],
                            bsp = borderLinePoints[j + 1];

                        var intersectCheck = Alghoritms.getIntersection(lfp, lsp, bfp, bsp);

                        if (isFinite(intersectCheck[0]) && isFinite(intersectCheck[1])) {
                            if (intersectPoint == null) {
                                //запоминаем точку первого пересечения
                                intersectPoint = { x: intersectCheck[0], y: intersectCheck[1] };

                                //если линия до первого пересечения была выше границы, а стала ниже 
                                //(или наоборот для нижней границы)
                                //учитывать, то y сверхну вниз
                                var isPassedBorderInRightDirection = this._intercectInRightWay(intersectPoint, lfp, lsp, bfp, bsp, borderLine.getFillDirection());
                                if (wasIntersected === false && !isPassedBorderInRightDirection) {
                                    this._drawStartFillRectangle(canvas, intersectPoint, borderLinePoints.slice(0, j + 1), this._pointsCache.slice(0, i + 1), fillColor);
                                }

                                // касательная
                                // коснулись второй точкой отрезка границы
                                // за счет isPassedBorderInRightDirection достаточно проверить знаки у тангент границы
                                // т.е. не задумываемся о том, верхняя или нижняя это граница
                                if (isPassedBorderInRightDirection) {
                                    // 2 - это число пикселей, на которые мы допускаем отклонение
                                    if (Alghoritms.euclideanDistance(intersectCheck, [bsp.x, bsp.y]) <= 2) {
                                        // проверяем наличие второго сегмента участка границы
                                        if (j < borderLinePoints.length - 1) {
                                            var firstSegmentK = Alghoritms.lineTangent([bfp.x, bfp.y], [bsp.x, bsp.y]),
                                                // у границы может быть всего две (реально одна) точка (когда граница - константа).
                                                // есть смысл взять последнюю (фейковую) точку
                                                // тогда знак произведения тангент будет 0 и мы таки не будем ничего заливать.
                                                nextPoint = borderLinePoints[j + 2] || bsp,
                                                secondSegmentK = Alghoritms.lineTangent([bsp.x, bsp.y], [nextPoint.x, nextPoint.y]);
                                            // нас интересуют только разные знаки у тангент
                                            // т.е. один участок убывает до точки касания, а второй не убывает
                                            if (firstSegmentK * secondSegmentK <= 0) {
                                                intersectPoint = null;
                                            }
                                        }
                                    }
                                }

                                //такая точка нам не нужна -
                                //нам нужно пересечение снизу вверх и наоборот.
                                if (!isPassedBorderInRightDirection) {
                                    intersectPoint = null;
                                }

                                wasIntersected = true;
                            } else {
                                //первая точка уже есть, пора заливать
                                canvas.beginPath();
                                canvas.moveTo(intersectPoint);
                                //рисуем пути по границе до следующей точки пересечения
                                var k;
                                for (k = lastBorderIntersectIndex; k < j + 1; k++) {
                                    canvas.lineTo(borderLinePoints[k]);
                                }
                                //и в точку пересечения
                                canvas.lineTo({ x: intersectCheck[0], y: intersectCheck[1] });

                                //и назад по линии в точку первого пересечения
                                for (k = i; k >= lastLineIntersectIndex; k--) {
                                    canvas.lineTo(this._pointsCache[k]);
                                }

                                //и в точку первого пересечения
                                //   this.canvas.lineTo(intersectPoint);
                                canvas.closePath().setFillStyle(fillColor).fill();

                                intersectPoint = null;
                            }

                            lastBorderIntersectIndex = j + 1;
                            lastLineIntersectIndex = i + 1;
                            break;
                        }
                    }
                }

                var isAboveBorder = !this._isMore(this._pointsCache[this._pointsCache.length - 1].y,
                    borderLinePoints[borderLinePoints.length - 1].y, borderLine.getFillDirection());

                if (intersectPoint == null && wasIntersected == false) {
                    //ни разу не пересекли - может просто выше (или ниже для нижней границы)?
                    //не забываем, что y идет вниз                   
                    if (!isAboveBorder) {
                        this._drawRectangleFill(canvas, borderLinePoints, fillColor);
                        return;
                    }
                }

                if (intersectPoint != null && isAboveBorder) {
                    //правая граница незакрыта
                    this._drawEndFillRectangle(canvas, intersectPoint, borderLinePoints.slice(lastBorderIntersectIndex - 1, borderLinePoints.length),
                        this._pointsCache.slice(lastLineIntersectIndex, this._pointsCache.length), fillColor);
                }

            },

            _drawStartFillRectangle: function (canvas, intersectPoint, borderLinePointsBeforeIntersect, linePointsBeforeIntersect, fillColor) {
                canvas.beginPath();
                //первая точка - результат опускания перпендикуляра из первой точки линии на границу
                //т.к пытаемся опустить перпендикуляр на границу, 
                //то не допускаем обгона _pointsCache[0] крайней левой точки borderLinePoints
                var leftPoint;
                //если линия левее конца границы
                if (linePointsBeforeIntersect[0].x < borderLinePointsBeforeIntersect[0].x) {
                    //для ур-в прямой нужно две точки линии, но их до пересечения может и не быть
                    var secondLinePoint = linePointsBeforeIntersect[1] || intersectPoint;
                    // инвертируем знаки
                    var fp = [linePointsBeforeIntersect[0].x, linePointsBeforeIntersect[0].y],
                        sp = [secondLinePoint.x, secondLinePoint.y];

                    leftPoint = {
                        x: borderLinePointsBeforeIntersect[0].x,
                        y: Alghoritms.getYOnLine([fp, sp], borderLinePointsBeforeIntersect[0].x)
                    };
                } else {
                    leftPoint = {
                        x: linePointsBeforeIntersect[0].x,
                        y: linePointsBeforeIntersect[0].y
                    };
                }

                if (borderLinePointsBeforeIntersect.length === 1) {
                    //одной точки маловато для прямой, докинем точку пересечения
                    borderLinePointsBeforeIntersect.push(intersectPoint);
                }

                //от крайней левой точки, к крайней точки линии
                canvas.moveTo(leftPoint);
                var i;
                for (i = 1; i < linePointsBeforeIntersect.length; i++) {
                    canvas.lineTo(linePointsBeforeIntersect[i]);
                }
                //и в точку пересечения
                canvas.lineTo(intersectPoint);
                //и по границе обратно в точку перпендикуляра
                for (i = borderLinePointsBeforeIntersect.length - 1; i >= 0; i--) {
                    canvas.lineTo(borderLinePointsBeforeIntersect[i]);
                }

                canvas.closePath().setFillStyle(fillColor).fill();


            },
            hasDataSource: function (){
                if (this.model.getProperty(Enums.ParameterRoles.DATA_SOURCE).connections.length) //проверка задан ли параметр объекта
                    return true;
                else
                    return false;
            },

            _drawEndFillRectangle: function (canvas, intersectPoint, borderLinePointsAfterIntersect, linePointsAfterIntersect, fillColor) {
                canvas.beginPath();
                //первая точка - результат опускания перпендикуляра из последней точки
                //линии на границу
                //из последней точки линии опускаем перпендикуляр на границу
                //и недопускаем обгон _pointsCache[0] крайней левой точки borderLinePoints
                var leftPoint = {
                    x: linePointsAfterIntersect[linePointsAfterIntersect.length - 1].x > borderLinePointsAfterIntersect[borderLinePointsAfterIntersect.length - 1].x ?
                        borderLinePointsAfterIntersect[borderLinePointsAfterIntersect.length - 1].x : linePointsAfterIntersect[linePointsAfterIntersect.length - 1].x,
                    y: linePointsAfterIntersect[linePointsAfterIntersect.length - 1].y
                };

                if (borderLinePointsAfterIntersect.length === 1) {
                    //одной точки маловато для прямой, докинем точку пересечения
                    borderLinePointsAfterIntersect.insert(0, intersectPoint);
                }

                var perpendicular = this._findPerpendicularInterstPoint(borderLinePointsAfterIntersect, leftPoint);
                canvas.moveTo(perpendicular.point);
                //от точки перпендикуляра, по линии
                var i;
                for (i = linePointsAfterIntersect.length - 1; i >= 0; i--) {
                    canvas.lineTo(linePointsAfterIntersect[i]);
                }
                //и в точку пересечения
                canvas.lineTo(intersectPoint);
                //и по границе обратно в точку перпендикуляра, точку до пересечения
                //отбрасываем
                for (i = 1; i < borderLinePointsAfterIntersect.length; i++) {
                    canvas.lineTo(borderLinePointsAfterIntersect[i]);
                }

                canvas.closePath().setFillStyle(fillColor).fill();
            },

            _drawRectangleFill: function (canvas, borderLinePoints, fillColor) {
                canvas.beginPath();
                //первая точка - результат опускания перпендикуляра из первой точки
                //линии на границу
                //т.к пытаемся опустить перпендикуляр на границу, 
                //то недопускаем обгона _pointsCache[0] крайней правой точки borderLinePoints
                var rightPoint = {
                    x: this._pointsCache[0].x < borderLinePoints[0].x ? borderLinePoints[0].x : this._pointsCache[0].x,
                    y: this._pointsCache[0].y
                };

                var startPerpendicular = this._findPerpendicularInterstPoint(borderLinePoints, rightPoint);
                canvas.moveTo(startPerpendicular.point);
                //от точки перпендикуляра, к крайней точки линии
                canvas.lineTo(rightPoint);
                var i;
                for (i = 1; i < this._pointsCache.length; i++) {
                    canvas.lineTo(this._pointsCache[i]);
                }
                //из последней точки линии опускаем перпендикуляр на границу
                //и недопускаем обгон _pointsCache[0] крайней левой точки borderLinePoints
                var leftPoint = {
                    x: this._pointsCache[this._pointsCache.length - 1].x < borderLinePoints[borderLinePoints.length - 1].x ?
                        borderLinePoints[borderLinePoints.length - 1].x : this._pointsCache[this._pointsCache.length - 1].x,
                    y: this._pointsCache[this._pointsCache.length - 1].y
                };
                var endPerpendicular = this._findPerpendicularInterstPoint(borderLinePoints, leftPoint);
                canvas.lineTo(endPerpendicular.point);
                //и от нее по границе назад
                for (i = endPerpendicular.index ; i >= startPerpendicular.index + 1; i--) {
                    canvas.lineTo(borderLinePoints[i]);
                }

                canvas.closePath().setFillStyle(fillColor).fill();
            },

            _findPerpendicularInterstPoint: function (linePoints, point) {
                var i;
                if (linePoints.length == 1)
                    return {
                        point: { x: linePoints[0].x, y: linePoints[0].y },
                        index: 0
                    };
                for (i = 0; i < linePoints.length - 1; i++) {
                    if (linePoints[i].x <= point.x && linePoints[i + 1].x >= point.x) {
                        return {
                            point: Alghoritms.getPerpendicularIntersectPoint([linePoints[i], linePoints[i + 1]], point),
                            index: i
                        }
                    }
                }
            },

            _intercectInRightWay: function (intersectPoint, linefp, linesp, borderfp, bordersp, direction) {
                if (this._isHorizontalLine(borderfp, bordersp) && this._isVerticalLine(linefp, linesp)) {
                    return direction === Enums.FillDirection.Up ? bordersp.y < linesp.y : bordersp.y >= linesp.y;
                }
                if (this._isHorizontalLine(linefp, linesp) && this._isVerticalLine(borderfp, bordersp)) {
                    return direction === Enums.FillDirection.Up ? bordersp.y > linesp.y : bordersp.y <= linesp.y;
                }

                //y сверху вниз
                var lineGrow = linesp.y < linefp.y,
                    borderGrow = bordersp.y < borderfp.y;

                if (lineGrow && !borderGrow) {
                    return direction === Enums.FillDirection.Up ? true : false;
                }

                if (!lineGrow && borderGrow) {
                    return direction === Enums.FillDirection.Up ? false : true;
                }

                // если и линия, и граница возрастают или убывают одновременно
                if ((lineGrow && borderGrow)
                    || (!lineGrow && !borderGrow)) {
                    //k = (y - y1) / (x - x1)
                    var lineK = Alghoritms.lineTangent([linefp.x, linefp.y], [linesp.x, linesp.y]);
                    var borderK = Alghoritms.lineTangent([borderfp.x, borderfp.y], [bordersp.x, bordersp.y]);
                    var diff = lineK - borderK < 0;
                    return direction === Enums.FillDirection.Up ? diff : !diff;
                }
            },

            reset: function (interval) {
                this._super(interval);
                this.borderLines.forEach(function (borderLine) {
                    borderLine.reset(interval);
                });
                this.dataStorage.clear();
            },            

            _isVerticalLine: function (linefp, linesp) {
                return linefp.x === linesp.x;
            },

            _isHorizontalLine: function (linefp, linesp) {
                return linefp.y === linesp.y;
            },

            _isMore: function (line, border, direction) {
                return direction === Enums.FillDirection.Up ? line > border : line < border;
            },

            _getPropertyName: function (property) {
                return this.model.getPropertyName(property);
            }

        });


        return Trendline;
    });