﻿// <copyright file="VerticalCursor.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// _completedRequest

define(['common/Enums', 'base/ObservableObject', 'common/Utilites', 'common/Appearence', 'controls/Trend/modules/Alghoritms', 'common/Appearence'],
    function (Enums, ObservableObject, Utilites, Appearence, Alghoritms, Appearence) {

        var events = Enums.TrendEvents;

        var VerticalCursor = ObservableObject.extend({
            options: {    // options used for drawing
                valuePadding: 5,
                circleColor: 'rgb(0, 220, 0)',
                circleBorderColor: 'rgb(0,0,0)',
                borderHeight: 10
            },
            init: function (canvas, model, backLayerCanvas, backLayerYAxis) {
                this._super();
                this.model = model;
                this._verticalCursors = [];
                this.canvas = canvas;
                this.backLayerCanvas = backLayerCanvas;
                this.backLayerYAxis = backLayerYAxis;
                this.isOnCircle = false;
                this._selectedCursorId = null;
                this.maxBufferCanvasSize = 0;
                this.factorStick = 0.2; 
                this.yCircleMaxSize = this.model.get(Enums.ParameterRoles.Y_ARROW_HEIGHT)
                                   + this.model.get(Enums.ParameterRoles.Y_ARROW_MARGIN);
                this.isCursorDeleted = false;
            },

            update: function (width, mark) {
                //сохраняем для использовании в onCursorAdded
                var delta = mark - (this.lastMark || mark);
                this.lastMark = mark;
                this.width = width;
                _.remove(this._verticalCursors, function (cursor) {
                    return cursor.toRemove === true;
                });

                if (this.model.getFreezeCursor() === true) {
                    this._verticalCursors.forEach(function (cursor) {
                        return cursor.timePoint += delta;
                    });
                }
            },

            onCursorAdded: function (coords, eventType) {
                var verticalCursor = {
                    x: coords.x,
                    y: coords.y,
                    timePoint: this._getTimePoint(coords.x),
                    prevX: null
                };

                this._verticalCursors.push(verticalCursor);
                if (this._verticalCursors.length > 1) {
                    var cursors = this._verticalCursors.splice(0, 1);
                    this.cleanBackLayer();
                }
            },

            getCursors: function () {
                return this._verticalCursors;
            },

            drawCursor: function (dimensions, cursor, mark, data, yRange) {
                var strokeColor = Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.CURSOR_COLOR)),
                    height = dimensions.height,
                    width = dimensions.width,
                    bounds = Alghoritms.binarySearch(data, 'timepoint', cursor.timePoint),
                    x = this.model.getFreezeCursor() ? cursor.x :
                        Alghoritms.getScreenX(cursor.timePoint, this.model.getInterval(), mark, width);

                this.maxBufferCanvasSize = this.canvas.getSize().width;

                if (bounds.lowBound === -1 || x < 1 || bounds.highBound >= data.length) {
                    if (!this.isCursorDeleted) {
                        this.cleanBackLayer();
                        this.isCursorDeleted = true;
                    }
                    return;
                } else {
                    this.isCursorDeleted = false;
                }

                if (bounds.lowBound === bounds.highBound) {
                    bounds.lowBound = Math.max(0, bounds.lowBound - 1);
                    bounds.highBound = Math.min(data.length - 1, bounds.highBound + 1);
                }

                var trendlineSegmentStart = this._getLineSegment(data[bounds.lowBound], mark, width, height, yRange),
                    trendlineSegmentEnd = this._getLineSegment(data[bounds.highBound], mark, width, height, yRange);

                var intersectionPoint = Alghoritms.getIntersection(trendlineSegmentStart,
                    trendlineSegmentEnd, { x: x, y: 0 }, { x: x, y: height }),
                    y = Alghoritms.getY(intersectionPoint[1], height, yRange);
                this.xCoordsLable = x + 4 * this.model.get(Enums.ParameterRoles.CURSOR_THICKNESS);
                var text = (y.toFixed(2)).toString();
                this.options.borderWidth = this.canvas.measureText(text) + 2;
                this._drawLabelBorder(this.xCoordsLable, intersectionPoint[1] + 1);
                this.canvas.setFillStyle(Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.CURSOR_COLOR)));
                this.canvas.fillText({
                    x: x + this.options.valuePadding,
                    y: intersectionPoint[1],
                    text: y.toFixed(2)
                });

                this.canvas.drawLine({
                    x1: x,
                    x2: x,
                    y1: 0,
                    y2: height,
                    strokeStyle: strokeColor,
                    lineWidth: this.model.get(Enums.ParameterRoles.CURSOR_THICKNESS)
                });

                this.updateCircle(cursor, x + this.canvas.canvas.offsetLeft,
                this.canvas.canvas.offsetTop - this.backLayerCanvas.canvas.offsetTop);

                if (this.model.getIsAutoScroll()) {
                    this.cleanBackLayer();
                    this.updateCircle(cursor, x + this.canvas.canvas.offsetLeft,
                    this.canvas.canvas.offsetTop - this.backLayerCanvas.canvas.offsetTop);
                }

                return x;
            },
            _drawLabelBorder: function (x, y) {
                var border = {};
                border.width = this.options.borderWidth;
                border.height = this.options.borderHeight;
                border.x = x - 2;
                border.y = y - border.height;
                border.fillStyle = Appearence.color.toCssColor(this.model.get(Enums.ParameterRoles.CHART_BACKGROUND));
                this.canvas.drawRect(border);
                border = null;
            },

            onBackLayerMouseDown: function (coords, eventType, event) {
                var backLayerHeight = this.canvas.canvas.offsetTop - this.backLayerCanvas.canvas.offsetTop;

                if (event.which === 1 && coords.y <= backLayerHeight) {
                    var distance,
                        // "1 - this.factorStick" - factor circle
                        yCircleCenter = backLayerHeight * (1 - this.factorStick) / 2, 
                        circleRadius = yCircleCenter,
                        backLayerXCoord = coords.x - this.canvas.canvas.offsetLeft;

                    for (var i = 0; i < this._verticalCursors.length; ++i) {
                        // equation of a circle
                        distance = Math.pow((backLayerXCoord - this._verticalCursors[i].x), 2) + Math.pow((coords.y - yCircleCenter), 2) - Math.pow(circleRadius, 2); 
                        if (distance <= 0) {
                            this.isOnCircle = true;
                            this._selectedCursorId = i;
                            break;
                        }
                    }
                }
                
            },

            yAxisRedraw: function (coords) {
                var arrowWidth = this.model.get(Enums.ParameterRoles.Y_ARROW_WIDTH),
                    backLayerHeight = this.canvas.canvas.offsetTop - this.backLayerCanvas.canvas.offsetTop;

                if (coords.x - this.canvas.canvas.offsetLeft - arrowWidth < (backLayerHeight) * this.factorStick / 2) {
                    this.backLayerYAxis.draw();
                }
            },

            onBackLayerMouseMove: function (coords, eventType, event) {
                var currentCursor = this._verticalCursors[this._selectedCursorId],
                    backLayerHeight = this.canvas.canvas.offsetTop - this.backLayerCanvas.canvas.offsetTop,
                    backLayerXCoord = coords.x - this.canvas.canvas.offsetLeft,
                    backLayerVerticalCursor;

                if (currentCursor === undefined) {
                    return;
                } else if (event.which === 1 && coords.y <= backLayerHeight && this.isOnCircle === true) {
                    backLayerVerticalCursor = currentCursor.x + this.canvas.canvas.offsetLeft;
                    if (backLayerXCoord > 1 && backLayerXCoord < this.maxBufferCanvasSize) {
                        this.cleanBackLayer();
                        currentCursor.x = backLayerXCoord;
                        currentCursor.timePoint = this._getTimePoint(backLayerXCoord);
                    }
                }
            },

            onBackLayerMouseUp: function (coords) {
                this._selectedCursorId = null;
            },

            updateCircle: function (cursor, x, y) {
                cursor.prevX = x;
                var line = {},
                    circle = {};

                line.x1 = x;
                line.y1 = y;
                line.x2 = x;
                line.y2 = y - this.factorStick * this.yCircleMaxSize;

                // "-1" need for keeping 2px distance between title and circle
                circle.radius = Math.floor((line.y2 - (y - this.yCircleMaxSize)) / 2) - 1; 
                circle.xCenter = x;
                circle.yCenter = line.y2 - circle.radius;
                circle.fillStyle = this.options.circleColor;
                circle.strokeStyle = this.options.circleBorderColor;

                cursor.x = circle.xCenter - this.canvas.canvas.offsetLeft;

                this.backLayerCanvas.beginRender();
                this.backLayerCanvas.drawLine(line);
                this.backLayerCanvas.drawCircle(circle);
                this.backLayerCanvas.endRender();

                line = null;
                circle = null;
            },

            cleanBackLayer: function () {
               
                var rectangle = {},
                y = this.canvas.canvas.offsetTop - this.backLayerCanvas.canvas.offsetTop;

                rectangle.x = this.canvas.canvas.offsetLeft;
                rectangle.y = y - this.yCircleMaxSize;
                // "+ (this.factorStick * y) + 1" needs for delete half part circle on the right border
                rectangle.width = this.canvas.getSize().width + (this.factorStick * y) + 1; 
                rectangle.height = y;
               
                this.backLayerCanvas.clearRect(rectangle);
                rectangle = null;
                this.backLayerYAxis.draw();
                this.backLayerCanvas.endRender();               
            },

            _getLineSegment: function (item, mark, width, height, range) {
                return Alghoritms.getPointScreenCoords(item, this.model.getInterval(), mark, width, height, range);
            },

            _getTimePoint: function (x) {
                var interval = this.model.getInterval();
                return this.lastMark - interval + ((x / this.width) * interval);
            }

        });

        return VerticalCursor;
    });