﻿// <copyright file="WindowController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

define(['common/Enums', 'base/CanvasContainerController', 'server/sources/SchemeVariablesSource',
    'core/Statistic', 'server/ServerLogger', 'core/functionBlocks/TaskManager',
'common/Utilites', 'common/Error'],
function (Enums, CanvasContainerController, SchemeVariablesSource, Statistic, ServerLogger, TaskManager, Utilites, Error) {
    //here we will define controller for button
    var WindowController = CanvasContainerController.extend({
        timeOut: 5000,
        init: function (statisticId) {
            //stores width/height coefficient
            this.statisticId = statisticId;
            this.sideRatio = null;
            this.proportionFixedSide = null;
            this.widthApplied = false;
            this.heightApplied = false;
            this._super();
            this.ClassName = 'WindowController';
            this.variablesSource = SchemeVariablesSource;
            this._createPreloaders();
            this.variablesSourcePropertyChangedInternalBind = this.variablesSourcePropertyChangedInternal.bind(this);
            this.taskManagerPropertyChangedInternalBind = this.taskManagerPropertyChangedInternal.bind(this);
            this.onServerErrorBind = this.onServerError.bind(this);
            this.waitingSubscribes = 0;
            this.preventChangeProportions = false;
        },

        onModelInit: function () {
            this._super();
            if (Utilites.hasServerVariables(this.mvc.model.serverVariables)) {
                this.variablesSource.addToSubscription(this.mvc.model.serverVariables, this.mvc.model.getId());
                this.variablesSource.subscribePropertyChanged(this.variablesSourcePropertyChangedInternalBind, this);
                this.variablesSource.subscribeServerErrorStateChanged(this.onServerErrorBind, this);
            }

            if (this.mvc.model.hasFBConnections()) {
                TaskManager.subscribePropertyChanged(this.taskManagerPropertyChangedInternalBind, this);
            }
        },

        onAfterModelInit: function () {

            //ожидаем, только если нужны get
            if (!Utilites.hasGetServerVariables(this.mvc.model.serverVariables)
                && !this.mvc.model.hasFBConnections()) {
                this.updateTypeConnections();
                this.printIfNeed();
            } else {
                //ожидаем, пока VM наполнится реальными данными
                //обработчик сработает единожды, и удалится
                this.mvc.view.$().css('opacity', 0);
                this.mvc.view.$().append(this.preloader);
                if (Utilites.hasGetServerVariables(this.mvc.model.serverVariables)) {
                    this.waitingSubscribes++;
                    this.variablesSource.subscribeSourceDataReady(this.onSourceDataReady, this, this.mvc.model.getId());
                }

                if (this.mvc.model.hasFBConnections()) {
                    this.waitingSubscribes++;
                    TaskManager.subscribeSubscribeTaskExecutedOnce(this.onSourceDataReady, this);
                }
                setTimeout(this.onTimeOut.bind(this), 5000);
            }

            this.calculateResizeProportions();
        },
        onTimeOut: function(event){
            if (this.waitingSubscribes != 0) {
                var errorText = "";
                errorText = "По прошествии " + this.timeOut + "мс ";
                if (this.variablesSource.variablesManagers.MPLCDataSource.lastSubscribeVars.length){
                    errorText += "не произошла подписка на " + this.variablesSource.variablesManagers.MPLCDataSource.lastSubscribeVars.length + " параметров сервера";
                }
                errorText += "\nне были получены значения параметров с id: \n" + this.variablesSource.getInfoOnWaitingVariables();
                Error.info(errorText);
                this.mvc.view.$().css('opacity', 1);
            }
        },
        onSourceDataReady: function (event) {
            //подписка на TaskExecuted идет как Once
            if (event.type === Enums.eventType.taskExecuted ||
                (event.type === Enums.eventType.sourceDataReady && event.objectId === this.mvc.model.getId())) {
                this.waitingSubscribes--;
            }

            if (this.waitingSubscribes < 0) {
                Error.debug('Waiting subscribes ' + this.waitingSubscribes);
            }

            if (this.waitingSubscribes === 0) {
                this.updateTypeConnections();
                this.preloader.hide();
                this.printIfNeed();
            }
            this.mvc.model.fireConnectedPropertiesChanged();
            this.mvc.view.$().css('opacity', 1);
        },

        onServerError: function (event) {
            if (event.isError === true) {
                this.mvc.view.$().css('opacity', 1);
                this.preloader.hide();
            }
        },

        onDestroy: function () {
            if (Utilites.hasServerVariables(this.mvc.model.serverVariables)) {
                this.variablesSource.unsubscribeSourceDataReady(this.onSourceDataReady, this.mvc.model.getId());
                this.variablesSource.unsubscribe(this.mvc.model.serverVariables, this.mvc.model.getId());
                this.variablesSource.unsubscribePropertyChanged(this.variablesSourcePropertyChangedInternalBind);
                this.variablesSource.unsubscribeServerErrorStateChanged(this.onServerErrorBind, this);                
            }

            this._super();
        },
        applyWidth: function(){
            if (this.mvc.parent() && this.mvc.parent().model.ClassName == 'FrameModel' && this.mvc.parent().model.getSizeContent() === Enums.SizeToContentType.SetSize)
                return;
            else
                this._super();
        },
        applyHeight: function(){
            if (this.mvc.parent() && this.mvc.parent().model.ClassName == 'FrameModel' && this.mvc.parent().model.getSizeContent() === Enums.SizeToContentType.SetSize)
                return;
            else
                this._super();
        },
        onAddedToParent: function () {
            this._super();
            Statistic.stopMeasurement(this.statisticId);            
        },

        onOneTimeConnectionsProcess: function () {
            //связи хранят только окна
            //сначала свои, потом уже дочерних (каскад)
            this.mvc.model.getType()._connectionsHandler.updateOneTimeConnections(this.mvc);

            this._super();          
        },

        onAddedToDOM: function () {
            this._super();
            //do it here because we may get parent sizes only after scheme was added to document
            //we need parent element size to calculate proportions for windows
            //this.calculateResizeProportions();
            this.onParentResized();
        },

        variablesSourcePropertyChangedInternal: function (event) {
            var path = '/' + event.property;
            this.mvc.model.updateTypeConnection(path, this.mvc);
        },

        taskManagerPropertyChangedInternal: function (event) {
            var path = '/' + event.property;
            this.mvc.model.updateTypeConnection(path, this.mvc);
        },

        onWidthUnitsChanged: function (value) {
            this.calculateResizeProportions();
            this.applyWidth();
        },

        onWidthChanged: function (value) {
            this.calculateResizeProportions();
            this.applyWidth();
        },

        onHeightChanged: function (value) {
            this.calculateResizeProportions();
            this.applyHeight();
        },

        onHeightUnitsChanged: function (value) {
            this.calculateResizeProportions();
            this.applyHeight();
        },

        onParentActualWidthChanged: function (event) {

            this._super(event);

            this.onParentResized();
        },

        onParentActualHeightChanged: function (event) {
            this._super(event);

            this.onParentResized();
        },

        onParentResized: function () {
            if (this.sideRatio === null || !window.initialized) {
                return;
            }

            var elementWidth = this.mvc.view.$().width();
            var elementHeight = this.mvc.view.$().height();

            if (this.proportionFixedSide === Enums.ControlSide.WIDTH) {
                elementHeight = Math.ceil(this.sideRatio * elementWidth);
                // this.mvc.view.$().css('width', '100%');
                this.mvc.view.$().css('height', elementHeight);
                this.fireActualHeightChanged(elementHeight);
            }
            else {
                elementWidth = Math.ceil(this.sideRatio * elementHeight);
                //   this.mvc.view.$().css('height', '100%');
                this.mvc.view.$().css('width', elementWidth);
                this.fireActualWidthChanged(elementWidth);
            }
        },

        calculateResizeProportions: function () {
            var proportionType = this.mvc.model.getSizeProportionType();
            if (proportionType === undefined || this.preventChangeProportions === true) {
                return;
            }

            var widthUnits = this.mvc.model.getWidthUnits();
            var heightUnits = this.mvc.model.getHeightUnits();

            var width = this.mvc.model.getWidth();
            var height = this.mvc.model.getHeight();

            if (proportionType !== Enums.ProportionType.No
                && widthUnits === Enums.SizeType.Relative
                && heightUnits === Enums.SizeType.Relative
                && width > 0
                && height > 0) {
                //determinate fixed side
                //we will use this value to calculate real size of opposite side
                this.proportionFixedSide = this.getFixedSide(proportionType, width, height);

                //now calculate ratio
                this.sideRatio = this.calculateSideRatio(this.proportionFixedSide, width, height);

                //we are ready to setup new size in parent.onresize
            }
            else {
                this.proportionFixedSide = null;
                this.sideRatio = null;
            }
        },

        getFixedSide: function (proportionType, widthPixels, heightPixels) {

            switch (proportionType) {
                case Enums.ProportionType.Width:
                    return Enums.ControlSide.WIDTH;
                case Enums.ProportionType.Height:
                    return Enums.ControlSide.HEIGHT;
                case Enums.ProportionType.Min:
                    return widthPixels > heightPixels ? Enums.ControlSide.HEIGHT : Enums.ControlSide.WIDTH;
                case Enums.ProportionType.Max:
                    return widthPixels > heightPixels ? Enums.ControlSide.WIDTH : Enums.ControlSide.HEIGHT;
                default:
                    throw 'Unknown proportion type ' + proportionType;
            }
        },
        printIfNeed: function () {
            if (this.mvc.model.needPrint) {
                window.print();
                this.mvc.model.needPrint = false;
            }
        },
        calculateSideRatio: function (fixedSide, widthPixels, heightPixels) {

            if (fixedSide === Enums.ControlSide.WIDTH) {
                return heightPixels / widthPixels;
            }
            else {
                return widthPixels / heightPixels;
            }
        },
        
        _createPreloaders: function () {
            this.preloader = $('<div></div>')
             .addClass("preloaderBase")
             .addClass("preloader");
        },

        hasGetServerVariables: function () {
            return this.mvc.model.serverVariables
                 && (this.mvc.model.serverVariables.hasOwnProperty('getValues') && this.mvc.model.serverVariables.getValues.length > 0)
        },
    });

    return WindowController;
});
