﻿// <copyright file="Statistic.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

define(['common/Enums'], function (Enums) {
    //for Steve Jobs browser
    if (typeof window.performance === 'undefined') {
        window.performance = {};
    }

    if (!window.performance.now) {
        var nowOffset = Date.now();
        if (performance.timing && performance.timing.navigationStart) {
            nowOffset = performance.timing.navigationStart;
        }

        window.performance.now = function now() {
            return Date.now() - nowOffset;
        };
    }
        
    var Statistic = Class.extend({
        statisticKey: 'insat_statistic',
        maxStatisticsDataCount: 100,
        init: function () {
            this.statistic = {}; //commandName : count: 0, times: []
            this.startedMeasurements = {}; //statisticHandle:startTimestamp
            this.statisticHandle = 0;
        },

        startMeasurement: function (type) {
            var id = (type || '') + this.statisticHandle++;
            this.startedMeasurements[id] = {
                id: id,
                type: type,
                timestamp: performance.now()
            };

            return id;
        },

        stopMeasurement: function (id) {
            var stopTimeStamp = performance.now();
            var measurementUnit = this.startedMeasurements[id];
            if (measurementUnit === undefined) {
                console.warn(String.format('Measurement for {0} was not started.', id));
                return;
            }

            delete this.startedMeasurements[id];
            this._setStatistic(measurementUnit.type, stopTimeStamp - measurementUnit.timestamp);
        },

         cancelMeasurement: function (id) {           
            if (this.startedMeasurements[id] === undefined) {
                console.warn(String.format('Measurement for {0} was not started.', id));
                return;
            }

            delete this.startedMeasurements[id];
        },

        getServerOperationStatistic: function () {
            var serverStatistic = {};
            var statistic = this.getStatistic();
            _.forEach(Enums.queryType, function (queryType) {
                if (statistic[queryType] !== undefined) {                    
                    serverStatistic[queryType] = this._getStatisticByType(queryType);
                }
            }, this);

            return serverStatistic;
        },

        _getStatisticByType: function(type){
            var statistic = this.statistic[type];
            if (statistic === undefined) {
                return {
                    max: 0,
                    average: 0,
                    median: 0,
                    count: 0
                };
            };

            return {
                max: _.max(statistic.times),
                average: this._getAverage(statistic.times),
                median: this._getMedian(statistic.times),
                count: statistic.count
            };
        },

        getStatisticReport: function(){
            return {
                window: this._getStatisticByType(Enums.statisticType.window),
                server: this.getServerOperationStatistic()
            };
        },

        _getAverage: function (statistic) {
            var sum = _.reduce(statistic, function (sum, num) {
                return sum + num;
            });

            return sum / statistic.length;
        },

        _getMedian: function (statistic) {
            var sorted = statistic.slice().sort();

            if (sorted.length % 2 === 1) {
                return sorted[(sorted.length - 1) / 2];
            } else {
                var a = sorted[(sorted.length / 2) - 1];
                var b = sorted[(sorted.length / 2)];
                return (a + b) / 2;
            }
        },

        _setStatistic: function (type, timeDelta) {
            var statistic = this.getStatistic();

            if (statistic[type] === undefined) {
                statistic[type] = {
                    count: 0,
                    times: []
                };
            } else if (statistic[type].times.length > this.maxStatisticsDataCount) {
                statistic[type].times.splice(0, 1);
            }

            statistic[type].count++;
            statistic[type].times.push(timeDelta);

            if (this._isLocalStorageSupported()) {
                localStorage.setItem(this.statisticKey, JSON.stringify(statistic));
            }
        },

        getStatistic: function () {
            if (this._isLocalStorageSupported()) {
                var statistic = localStorage.getItem(this.statisticKey) ;
                return statistic !== null ? JSON.parse(statistic) : {};
            } else {
                return this.statistic;
            }
        },

        _isLocalStorageSupported: function () {
            // return typeof (Storage) !== "undefined";
            return false;
        }
    });

    return new Statistic();
});