﻿/* http://keith-wood.name/svg.html
   SVG for jQuery v1.4.5.
   Written by Keith Wood (kbwood{at}iinet.com.au) August 2007.
   Dual licensed under the GPL (http://dev.jquery.com/browser/trunk/jquery/GPL-LICENSE.txt) and 
   MIT (http://dev.jquery.com/browser/trunk/jquery/MIT-LICENSE.txt) licenses. 
   Please attribute the author if you use it. */

 (function ($) { // Hide scope, no $ conflict

    /* SVG manager.
	Use the singleton instance of this class, $.svg, 
	to interact with the SVG functionality. */
    function SVGManager() {
        this._settings = []; // Settings to be remembered per SVG object
        this._extensions = []; // List of SVG extensions added to SVGWrapper
        // for each entry [0] is extension name, [1] is extension class (function)
        // the function takes one parameter - the SVGWrapper instance
        this.regional = []; // Localisations, indexed by language, '' for default (English)
        this.regional[''] = {
            errorLoadingText: 'Error loading',
            notSupportedText: 'This browser does not support SVG'
        };
        this.local = this.regional['']; // Current localisation
        this._uuid = new Date().getTime();
        this._renesis = detectActiveX('RenesisX.RenesisCtrl');
    }

    /* Determine whether a given ActiveX control is available.
	@param  classId  (string) the ID for the ActiveX control
	@return  (boolean) true if found, false if not */
    function detectActiveX(classId) {
        try {
            return !!(window.ActiveXObject && new ActiveXObject(classId));
        }
        catch (e) {
            return false;
        }
    }

    var PROP_NAME = 'svgwrapper';

    $.extend(SVGManager.prototype, {
        /* Class name added to elements to indicate already configured with SVG. */
        markerClassName: 'hasSVG',

        /* SVG namespace. */
        svgNS: 'http://www.w3.org/2000/svg',
        /* XLink namespace. */
        xlinkNS: 'http://www.w3.org/1999/xlink',

        /* SVG wrapper class. */
        _wrapperClass: SVGWrapper,

        /* Camel-case versions of attribute names containing dashes or are reserved words. */
        _attrNames: {
            class_: 'class', in_: 'in',
            alignmentBaseline: 'alignment-baseline', baselineShift: 'baseline-shift',
            clipPath: 'clip-path', clipRule: 'clip-rule',
            colorInterpolation: 'color-interpolation',
            colorInterpolationFilters: 'color-interpolation-filters',
            colorRendering: 'color-rendering', dominantBaseline: 'dominant-baseline',
            enableBackground: 'enable-background', fillOpacity: 'fill-opacity',
            fillRule: 'fill-rule', floodColor: 'flood-color',
            floodOpacity: 'flood-opacity', fontFamily: 'font-family',
            fontSize: 'font-size', fontSizeAdjust: 'font-size-adjust',
            fontStretch: 'font-stretch', fontStyle: 'font-style',
            fontVariant: 'font-variant', fontWeight: 'font-weight',
            glyphOrientationHorizontal: 'glyph-orientation-horizontal',
            glyphOrientationVertical: 'glyph-orientation-vertical',
            horizAdvX: 'horiz-adv-x', horizOriginX: 'horiz-origin-x',
            imageRendering: 'image-rendering', letterSpacing: 'letter-spacing',
            lightingColor: 'lighting-color', markerEnd: 'marker-end',
            markerMid: 'marker-mid', markerStart: 'marker-start',
            shapeRendering: 'shape-rendering',
            stopColor: 'stop-color', stopOpacity: 'stop-opacity',
            strokeWidth: 'stroke-width',
            strikethroughPosition: 'strikethrough-position',
            strikethroughThickness: 'strikethrough-thickness',
            textAnchor: 'text-anchor', textDecoration: 'text-decoration', textRendering: 'text-rendering',
            underlinePosition: 'underline-position', underlineThickness: 'underline-thickness',
            vertAdvY: 'vert-adv-y', vertOriginY: 'vert-origin-y',
            wordSpacing: 'word-spacing', vectorEffect: 'vector-effect', writingMode: 'writing-mode'
        },

        _styleAttrNames: {
            fill: 'fill', stroke: 'stroke', strokeOpacity: 'stroke-opacity',
            strokeDashArray: 'stroke-dasharray', strokeDashOffset: 'stroke-dashoffset',
            strokeLineCap: 'stroke-linecap', strokeLineJoin: 'stroke-linejoin',
            strokeMiterLimit: 'stroke-miterlimit'
        },

        /* Add the SVG object to its container. */
        _attachSVG: function (container, settings) {
            var svg = (container.namespaceURI == this.svgNS ? container : null);
            var container = (svg ? null : container);
            if ($(container || svg).hasClass(this.markerClassName)) {
                return;
            }
            if (typeof settings == 'string') {
                settings = { loadURL: settings };
            }
            else if (typeof settings == 'function') {
                settings = { onLoad: settings };
            }
            $(container || svg).addClass(this.markerClassName);
            try {
                if (!svg) {
                    svg = document.createElementNS(this.svgNS, 'svg');
                    svg.setAttribute('version', '1.1');
                    if (container.clientWidth > 0) {
                        svg.setAttribute('width', container.clientWidth);
                    }
                    if (container.clientHeight > 0) {
                        svg.setAttribute('height', container.clientHeight);
                    }

                    container.insertBefore(svg, container.firstChild);
                }
                this._afterLoad(container, svg, settings || {});
            }
            catch (e) {
                if ($.browser.msie) {
                    if (!container.id) {
                        container.id = 'svg' + (this._uuid++);
                    }
                    this._settings[container.id] = settings;
                    container.innerHTML = '<embed type="image/svg+xml" width="100%" ' +
					'height="100%" src="' + (settings.initPath || '') + 'blank.svg" ' +
					'pluginspage="http://www.adobe.com/svg/viewer/install/main.html"/>';
                }
                else {
                    container.innerHTML = '<p class="svg_error">' +
					this.local.notSupportedText + '</p>';
                }
            }
        },

        /* SVG callback after loading - register SVG root. */
        _registerSVG: function () {
            for (var i = 0; i < document.embeds.length; i++) { // Check all
                var container = document.embeds[i].parentNode;
                if (!$(container).hasClass($.svg.markerClassName) || // Not SVG
					$.data(container, PROP_NAME)) { // Already done
                    continue;
                }
                var svg = null;
                try {
                    svg = document.embeds[i].getSVGDocument();
                }
                catch (e) {
                    setTimeout($.svg._registerSVG, 250); // Renesis takes longer to load
                    return;
                }
                svg = (svg ? svg.documentElement : null);
                if (svg) {
                    $.svg._afterLoad(container, svg);
                }
            }
        },

        /* Post-processing once loaded. */
        _afterLoad: function (container, svg, settings) {
            var settings = settings || this._settings[container.id];
            this._settings[container ? container.id : ''] = null;
            var wrapper = new this._wrapperClass(svg, container);
            $.data(container || svg, PROP_NAME, wrapper);
            try {
                if (settings.loadURL) { // Load URL
                    wrapper.load(settings.loadURL, settings);
                }
                if (settings.settings) { // Additional settings
                    wrapper.configure(settings.settings);
                }
                if (settings.onLoad && !settings.loadURL) { // Onload callback
                    settings.onLoad.apply(container || svg, [wrapper]);
                }
            }
            catch (e) {
                alert(e);
            }
        },

        /* Return the SVG wrapper created for a given container.
		@param  container  (string) selector for the container or
		(element) the container for the SVG object or
		jQuery collection - first entry is the container
		@return  (SVGWrapper) the corresponding SVG wrapper element, or null if not attached */
        _getSVG: function (container) {
            container = (typeof container == 'string' ? $(container)[0] :
			(container.jquery ? container[0] : container));
            return $.data(container, PROP_NAME);
        },

        /* Remove the SVG functionality from a div.
		@param  container  (element) the container for the SVG object */
        _destroySVG: function (container) {
            var $container = $(container);
            if (!$container.hasClass(this.markerClassName)) {
                return;
            }
            $container.removeClass(this.markerClassName);
            if (container.namespaceURI != this.svgNS) {
                $container.empty();
            }
            $.removeData(container, PROP_NAME);
        },

        /* Extend the SVGWrapper object with an embedded class.
		The constructor function must take a single parameter that is
		a reference to the owning SVG root object. This allows the 
		extension to access the basic SVG functionality.
		@param  name      (string) the name of the SVGWrapper attribute to access the new class
		@param  extClass  (function) the extension class constructor */
        addExtension: function (name, extClass) {
            this._extensions.push([name, extClass]);
        },

        /* Does this node belong to SVG?
		@param  node  (element) the node to be tested
		@return  (boolean) true if an SVG node, false if not */
        isSVGElem: function (node) {
            return (node.nodeType == 1 && node.namespaceURI == $.svg.svgNS);
        }
    });

    /* The main SVG interface, which encapsulates the SVG element.
	Obtain a reference from $().svg('get') */
    function SVGWrapper(svg, container) {
        this._svg = svg; // The SVG root node
        this._container = container; // The containing div
        for (var i = 0; i < $.svg._extensions.length; i++) {
            var extension = $.svg._extensions[i];
            this[extension[0]] = new extension[1](this);
        }
    }

    $.extend(SVGWrapper.prototype, {

        /* Retrieve the width of the SVG object. */
        _width: function () {
            return (this._container ? this._container.clientWidth : this._svg.width);
        },

        /* Retrieve the height of the SVG object. */
        _height: function () {
            return (this._container ? this._container.clientHeight : this._svg.height);
        },

        /* Retrieve the root SVG element.
		@return  the top-level SVG element */
        root: function () {
            return this._svg;
        },

        /* Configure a SVG node.
		@param  node      (element, optional) the node to configure
		@param  settings  (object) additional settings for the root
		@param  clear     (boolean) true to remove existing attributes first,
		false to add to what is already there (optional)
		@return  (SVGWrapper) this root */
        configure: function (node, settings, clear) {
            if (!node.nodeName) {
                clear = settings;
                settings = node;
                node = this._svg;
            }
            if (clear) {
                for (var i = node.attributes.length - 1; i >= 0; i--) {
                    var attr = node.attributes.item(i);
                    if (!(attr.nodeName == 'onload' || attr.nodeName == 'version' ||
						attr.nodeName.substring(0, 5) == 'xmlns')) {
                        node.attributes.removeNamedItem(attr.nodeName);
                    }
                }
            }
            for (var attrName in settings) {
                node.setAttribute($.svg._attrNames[attrName] || attrName, settings[attrName]);
            }
            return this;
        },

        /* Locate a specific element in the SVG document.
		@param  id  (string) the element's identifier
		@return  (element) the element reference, or null if not found */
        getElementById: function (id) {
            return this._svg.ownerDocument.getElementById(id);
        },

        /* Change the attributes for a SVG node.
		@param  element   (SVG element) the node to change
		@param  settings  (object) the new settings
		@return  (SVGWrapper) this root */
        change: function (element, settings) {
            if (element) {
                for (var name in settings) {
                    if (settings[name] == null) {
                        element.removeAttribute($.svg._attrNames[name] || name);
                    }
                    else {
                        if ($.svg._styleAttrNames[name]) {
                            if (name === "width" || name === "height") {
                                element.removeAttribute($.svg._attrNames[name] || name);
                            }

                            element.style[$.svg._styleAttrNames[name] || name] = settings[name];
                        } else {
                            element.setAttribute($.svg._attrNames[name] || name, settings[name]);
                        }
                    }
                }
            }

            return this;
        },

        /* Check for parent being absent and adjust arguments accordingly. */
        _args: function (values, names, optSettings) {
            names.splice(0, 0, 'parent');
            names.splice(names.length, 0, 'settings');
            var args = {};
            var offset = 0;
            if (values[0] != null && values[0].jquery) {
                values[0] = values[0][0];
            }
            if (values[0] != null && !(typeof values[0] == 'object' && values[0].nodeName)) {
                args['parent'] = null;
                offset = 1;
            }
            for (var i = 0; i < values.length; i++) {
                args[names[i + offset]] = values[i];
            }
            if (optSettings) {
                $.each(optSettings, function (i, value) {
                    if (typeof args[value] == 'object') {
                        args.settings = args[value];
                        args[value] = null;
                    }
                });
            }
            return args;
        },

        /* Add a title.
		@param  parent    (element or jQuery) the parent node for the new title (optional)
		@param  text      (string) the text of the title
		@param  settings  (object) additional settings for the title (optional)
		@return  (element) the new title node */
        title: function (parent, text, settings) {
            var args = this._args(arguments, ['text']);
            var node = this._makeNode(args.parent, 'title', args.settings || {});
            node.appendChild(this._svg.ownerDocument.createTextNode(args.text));
            return node;
        },

        /* Add a description.
		@param  parent    (element or jQuery) the parent node for the new description (optional)
		@param  text      (string) the text of the description
		@param  settings  (object) additional settings for the description (optional)
		@return  (element) the new description node */
        describe: function (parent, text, settings) {
            var args = this._args(arguments, ['text']);
            var node = this._makeNode(args.parent, 'desc', args.settings || {});
            node.appendChild(this._svg.ownerDocument.createTextNode(args.text));
            return node;
        },

        /* Add a definitions node.
		@param  parent    (element or jQuery) the parent node for the new definitions (optional)
		@param  id        (string) the ID of this definitions (optional)
		@param  settings  (object) additional settings for the definitions (optional)
		@return  (element) the new definitions node */
        defs: function (parent, id, settings) {
            var args = this._args(arguments, ['id'], ['id']);
            return this._makeNode(args.parent, 'defs', $.extend(
			(args.id ? { id: args.id } : {}), args.settings || {}));
        },

        /* Add a symbol definition.
		@param  parent    (element or jQuery) the parent node for the new symbol (optional)
		@param  id        (string) the ID of this symbol
		@param  x1        (number) the left coordinate for this symbol
		@param  y1        (number) the top coordinate for this symbol
		@param  width     (number) the width of this symbol
		@param  height    (number) the height of this symbol
		@param  settings  (object) additional settings for the symbol (optional)
		@return  (element) the new symbol node */
        symbol: function (parent, id, x1, y1, width, height, settings) {
            var args = this._args(arguments, ['id', 'x1', 'y1', 'width', 'height']);
            return this._makeNode(args.parent, 'symbol', $.extend({
                id: args.id,
                viewBox: args.x1 + ' ' + args.y1 + ' ' + args.width + ' ' + args.height
            },
			args.settings || {}));
        },

        /* Add a marker definition.
		@param  parent    (element or jQuery) the parent node for the new marker (optional)
		@param  id        (string) the ID of this marker
		@param  refX      (number) the x-coordinate for the reference point
		@param  refY      (number) the y-coordinate for the reference point
		@param  mWidth    (number) the marker viewport width
		@param  mHeight   (number) the marker viewport height
		@param  orient    (string or int) 'auto' or angle (degrees) (optional)
		@param  settings  (object) additional settings for the marker (optional)
		@return  (element) the new marker node */
        marker: function (parent, id, refX, refY, mWidth, mHeight, orient, settings) {
            var args = this._args(arguments, ['id', 'refX', 'refY',
			'mWidth', 'mHeight', 'orient'], ['orient']);
            return this._makeNode(args.parent, 'marker', $.extend(
			{
			    id: args.id, refX: args.refX, refY: args.refY, markerWidth: args.mWidth,
			    markerHeight: args.mHeight, orient: args.orient || 'auto'
			}, args.settings || {}));
        },

        /* Add a style node.
		@param  parent    (element or jQuery) the parent node for the new node (optional)
		@param  styles    (string) the CSS styles
		@param  settings  (object) additional settings for the node (optional)
		@return  (element) the new style node */
        style: function (parent, styles, settings) {
            var args = this._args(arguments, ['styles']);
            var node = this._makeNode(args.parent, 'style', $.extend(
			{ type: 'text/css' }, args.settings || {}));
            node.appendChild(this._svg.ownerDocument.createTextNode(args.styles));
            if ($.browser.opera) {
                $('head').append('<style type="text/css">' + args.styles + '</style>');
            }
            return node;
        },

        /* Add a script node.
		@param  parent    (element or jQuery) the parent node for the new node (optional)
		@param  script    (string) the JavaScript code
		@param  type      (string) the MIME type for the code (optional, default 'text/javascript')
		@param  settings  (object) additional settings for the node (optional)
		@return  (element) the new script node */
        script: function (parent, script, type, settings) {
            var args = this._args(arguments, ['script', 'type'], ['type']);
            var node = this._makeNode(args.parent, 'script', $.extend(
			{ type: args.type || 'text/javascript' }, args.settings || {}));
            node.appendChild(this._svg.ownerDocument.createTextNode(args.script));
            if (!$.browser.mozilla) {
                $.globalEval(args.script);
            }
            return node;
        },

        /* Add a linear gradient definition.
		Specify all of x1, y1, x2, y2 or none of them.
		@param  parent    (element or jQuery) the parent node for the new gradient (optional)
		@param  id        (string) the ID for this gradient
		@param  stops     (string[][]) the gradient stops, each entry is
		[0] is offset (0.0-1.0 or 0%-100%), [1] is colour, 
		[2] is opacity (optional)
		@param  x1        (number) the x-coordinate of the gradient start (optional)
		@param  y1        (number) the y-coordinate of the gradient start (optional)
		@param  x2        (number) the x-coordinate of the gradient end (optional)
		@param  y2        (number) the y-coordinate of the gradient end (optional)
		@param  settings  (object) additional settings for the gradient (optional)
		@return  (element) the new gradient node */
        linearGradient: function (parent, id, stops, x1, y1, x2, y2, settings) {
            var args = this._args(arguments,
			['id', 'stops', 'x1', 'y1', 'x2', 'y2'], ['x1']);
            var sets = $.extend({ id: args.id },
			(args.x1 != null ? { x1: args.x1, y1: args.y1, x2: args.x2, y2: args.y2 } : {}));
            return this._gradient(args.parent, 'linearGradient',
			$.extend(sets, args.settings || {}), args.stops);
        },

        /* Add a radial gradient definition.
		Specify all of cx, cy, r, fx, fy or none of them.
		@param  parent    (element or jQuery) the parent node for the new gradient (optional)
		@param  id        (string) the ID for this gradient
		@param  stops     (string[][]) the gradient stops, each entry
		[0] is offset, [1] is colour, [2] is opacity (optional)
		@param  cx        (number) the x-coordinate of the largest circle centre (optional)
		@param  cy        (number) the y-coordinate of the largest circle centre (optional)
		@param  r         (number) the radius of the largest circle (optional)
		@param  fx        (number) the x-coordinate of the gradient focus (optional)
		@param  fy        (number) the y-coordinate of the gradient focus (optional)
		@param  settings  (object) additional settings for the gradient (optional)
		@return  (element) the new gradient node */
        radialGradient: function (parent, id, stops, cx, cy, r, fx, fy, settings) {
            var args = this._args(arguments,
			['id', 'stops', 'cx', 'cy', 'r', 'fx', 'fy'], ['cx']);
            var sets = $.extend({ id: args.id }, (args.cx != null ?
			{ cx: args.cx, cy: args.cy, r: args.r, fx: args.fx, fy: args.fy } : {}));
            return this._gradient(args.parent, 'radialGradient',
			$.extend(sets, args.settings || {}), args.stops);
        },

        /* Add a gradient node. */
        _gradient: function (parent, name, settings, stops) {
            var node = this._makeNode(parent, name, settings);
            for (var i = 0; i < stops.length; i++) {
                var stop = stops[i];
                this._makeNode(node, 'stop', $.extend(
				{ offset: stop[0], stopColor: stop[1] },
				(stop[2] != null ? { stopOpacity: stop[2] } : {})));
            }
            return node;
        },

        /* Add a pattern definition.
		Specify all of vx, vy, xwidth, vheight or none of them.
		@param  parent    (element or jQuery) the parent node for the new pattern (optional)
		@param  id        (string) the ID for this pattern
		@param  x         (number) the x-coordinate for the left edge of the pattern
		@param  y         (number) the y-coordinate for the top edge of the pattern
		@param  width     (number) the width of the pattern
		@param  height    (number) the height of the pattern
		@param  vx        (number) the minimum x-coordinate for view box (optional)
		@param  vy        (number) the minimum y-coordinate for the view box (optional)
		@param  vwidth    (number) the width of the view box (optional)
		@param  vheight   (number) the height of the view box (optional)
		@param  settings  (object) additional settings for the pattern (optional)
		@return  (element) the new pattern node */
        pattern: function (parent, id, x, y, width, height, vx, vy, vwidth, vheight, settings) {
            var args = this._args(arguments, ['id', 'x', 'y', 'width', 'height',
			'vx', 'vy', 'vwidth', 'vheight'], ['vx']);
            var sets = $.extend({
                id: args.id, x: args.x, y: args.y,
                width: args.width, height: args.height
            }, (args.vx != null ?
			{ viewBox: args.vx + ' ' + args.vy + ' ' + args.vwidth + ' ' + args.vheight } : {}));
            return this._makeNode(args.parent, 'pattern', $.extend(sets, args.settings || {}));
        },

        /* Add a clip path definition.
		@param  parent  (element) the parent node for the new element (optional)
		@param  id      (string) the ID for this path
		@param  units   (string) either 'userSpaceOnUse' (default) or 'objectBoundingBox' (optional)
		@return  (element) the new clipPath node */
        clipPath: function (parent, id, units, settings) {
            var args = this._args(arguments, ['id', 'units']);
            args.units = args.units || 'userSpaceOnUse';
            return this._makeNode(args.parent, 'clipPath', $.extend(
			{ id: args.id, clipPathUnits: args.units }, args.settings || {}));
        },

        /* Add a mask definition.
		@param  parent    (element or jQuery) the parent node for the new mask (optional)
		@param  id        (string) the ID for this mask
		@param  x         (number) the x-coordinate for the left edge of the mask
		@param  y         (number) the y-coordinate for the top edge of the mask
		@param  width     (number) the width of the mask
		@param  height    (number) the height of the mask
		@param  settings  (object) additional settings for the mask (optional)
		@return  (element) the new mask node */
        mask: function (parent, id, x, y, width, height, settings) {
            var args = this._args(arguments, ['id', 'x', 'y', 'width', 'height']);
            return this._makeNode(args.parent, 'mask', $.extend(
			{ id: args.id, x: args.x, y: args.y, width: args.width, height: args.height },
			args.settings || {}));
        },

        /* Create a new path object.
		@return  (SVGPath) a new path object */
        createPath: function () {
            return new SVGPath();
        },

        /* Create a new text object.
		@return  (SVGText) a new text object */
        createText: function () {
            return new SVGText();
        },

        /* Add an embedded SVG element.
		Specify all of vx, vy, vwidth, vheight or none of them.
		@param  parent    (element or jQuery) the parent node for the new node (optional)
		@param  x         (number) the x-coordinate for the left edge of the node
		@param  y         (number) the y-coordinate for the top edge of the node
		@param  width     (number) the width of the node
		@param  height    (number) the height of the node
		@param  vx        (number) the minimum x-coordinate for view box (optional)
		@param  vy        (number) the minimum y-coordinate for the view box (optional)
		@param  vwidth    (number) the width of the view box (optional)
		@param  vheight   (number) the height of the view box (optional)
		@param  settings  (object) additional settings for the node (optional)
		@return  (element) the new node */
        svg: function (parent, x, y, width, height, vx, vy, vwidth, vheight, settings) {
            var args = this._args(arguments, ['x', 'y', 'width', 'height',
			'vx', 'vy', 'vwidth', 'vheight'], ['vx']);
            var sets = $.extend({ x: args.x, y: args.y, width: args.width, height: args.height },
			(args.vx != null ? {
			    viewBox: args.vx + ' ' + args.vy + ' ' +
                args.vwidth + ' ' + args.vheight
			} : {}));
            return this._makeNode(args.parent, 'svg', $.extend(sets, args.settings || {}));
        },

        /* Create a group.
		@param  parent    (element or jQuery) the parent node for the new group (optional)
		@param  id        (string) the ID of this group (optional)
		@param  settings  (object) additional settings for the group (optional)
		@return  (element) the new group node */
        group: function (parent, id, settings) {
            var args = this._args(arguments, ['id'], ['id']);
            return this._makeNode(args.parent, 'g', $.extend({ id: args.id }, args.settings || {}));
        },

        /* Add a usage reference.
		Specify all of x, y, width, height or none of them.
		@param  parent    (element or jQuery) the parent node for the new node (optional)
		@param  x         (number) the x-coordinate for the left edge of the node (optional)
		@param  y         (number) the y-coordinate for the top edge of the node (optional)
		@param  width     (number) the width of the node (optional)
		@param  height    (number) the height of the node (optional)
		@param  ref       (string) the ID of the definition node
		@param  settings  (object) additional settings for the node (optional)
		@return  (element) the new node */
        use: function (parent, x, y, width, height, ref, settings) {
            var args = this._args(arguments, ['x', 'y', 'width', 'height', 'ref']);
            if (typeof args.x == 'string') {
                args.ref = args.x;
                args.settings = args.y;
                args.x = args.y = args.width = args.height = null;
            }
            var node = this._makeNode(args.parent, 'use', $.extend(
			{ x: args.x, y: args.y, width: args.width, height: args.height },
			args.settings || {}));
            node.setAttributeNS($.svg.xlinkNS, 'href', args.ref);
            return node;
        },

        /* Add a link, which applies to all child elements.
		@param  parent    (element or jQuery) the parent node for the new link (optional)
		@param  ref       (string) the target URL
		@param  settings  (object) additional settings for the link (optional)
		@return  (element) the new link node */
        link: function (parent, ref, settings) {
            var args = this._args(arguments, ['ref']);
            var node = this._makeNode(args.parent, 'a', args.settings);
            node.setAttributeNS($.svg.xlinkNS, 'href', args.ref);
            return node;
        },

        /* Add an image.
		@param  parent    (element or jQuery) the parent node for the new image (optional)
		@param  x         (number) the x-coordinate for the left edge of the image
		@param  y         (number) the y-coordinate for the top edge of the image
		@param  width     (number) the width of the image
		@param  height    (number) the height of the image
		@param  ref       (string) the path to the image
		@param  settings  (object) additional settings for the image (optional)
		@return  (element) the new image node */
        image: function (parent, x, y, width, height, ref, settings) {
            var args = this._args(arguments, ['x', 'y', 'width', 'height', 'ref']);
            var node = this._makeNode(args.parent, 'image', $.extend(
			{ x: args.x, y: args.y, width: args.width, height: args.height },
			args.settings || {}));
            node.setAttributeNS($.svg.xlinkNS, 'href', args.ref);
            return node;
        },

        /* Draw a path.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  path      (string or SVGPath) the path to draw
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        path: function (parent, path, settings) {
            var args = this._args(arguments, ['path']);
            return this._makeNode(args.parent, 'path', $.extend(
			{ d: (args.path.path ? args.path.path() : args.path) }, args.settings || {}));
        },

        /* Draw a rectangle.
		Specify both of rx and ry or neither.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  x         (number) the x-coordinate for the left edge of the rectangle
		@param  y         (number) the y-coordinate for the top edge of the rectangle
		@param  width     (number) the width of the rectangle
		@param  height    (number) the height of the rectangle
		@param  rx        (number) the x-radius of the ellipse for the rounded corners (optional)
		@param  ry        (number) the y-radius of the ellipse for the rounded corners (optional)
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        rect: function (parent, x, y, width, height, rx, ry, settings) {
            var args = this._args(arguments, ['x', 'y', 'width', 'height', 'rx', 'ry'], ['rx']);
            return this._makeNode(args.parent, 'rect', $.extend(
			{ x: args.x, y: args.y, width: args.width, height: args.height },
			(args.rx ? { rx: args.rx, ry: args.ry } : {}), args.settings || {}));
        },

        /* Draw a circle.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  cx        (number) the x-coordinate for the centre of the circle
		@param  cy        (number) the y-coordinate for the centre of the circle
		@param  r         (number) the radius of the circle
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        circle: function (parent, cx, cy, r, settings) {
            var args = this._args(arguments, ['cx', 'cy', 'r']);
            return this._makeNode(args.parent, 'circle', $.extend(
			{ cx: args.cx, cy: args.cy, r: args.r }, args.settings || {}));
        },

        /* Draw an ellipse.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  cx        (number) the x-coordinate for the centre of the ellipse
		@param  cy        (number) the y-coordinate for the centre of the ellipse
		@param  rx        (number) the x-radius of the ellipse
		@param  ry        (number) the y-radius of the ellipse
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        ellipse: function (parent, cx, cy, rx, ry, settings) {
            var args = this._args(arguments, ['cx', 'cy', 'rx', 'ry']);
            return this._makeNode(args.parent, 'ellipse', $.extend(
			{ cx: args.cx, cy: args.cy, rx: args.rx, ry: args.ry }, args.settings || {}));
        },

        /* Draw a line.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  x1        (number) the x-coordinate for the start of the line
		@param  y1        (number) the y-coordinate for the start of the line
		@param  x2        (number) the x-coordinate for the end of the line
		@param  y2        (number) the y-coordinate for the end of the line
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        line: function (parent, x1, y1, x2, y2, settings) {
            var args = this._args(arguments, ['x1', 'y1', 'x2', 'y2']);
            return this._makeNode(args.parent, 'line', $.extend(
			{ x1: args.x1, y1: args.y1, x2: args.x2, y2: args.y2 }, args.settings || {}));
        },

        /* Draw a polygonal line.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  points    (number[][]) the x-/y-coordinates for the points on the line
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        polyline: function (parent, points, settings) {
            var args = this._args(arguments, ['points']);
            return this._poly(args.parent, 'polyline', args.points, args.settings);
        },

        /* Draw a polygonal shape.
		@param  parent    (element or jQuery) the parent node for the new shape (optional)
		@param  points    (number[][]) the x-/y-coordinates for the points on the shape
		@param  settings  (object) additional settings for the shape (optional)
		@return  (element) the new shape node */
        polygon: function (parent, points, settings) {
            var args = this._args(arguments, ['points']);
            return this._poly(args.parent, 'polygon', args.points, args.settings);
        },

        /* Draw a polygonal line or shape. */
        _poly: function (parent, name, points, settings) {
            var ps = '';
            for (var i = 0; i < points.length; i++) {
                ps += points[i].join() + ' ';
            }
            return this._makeNode(parent, name, $.extend(
			{ points: $.trim(ps) }, settings || {}));
        },

        /* Draw text.
		Specify both of x and y or neither of them.
		@param  parent    (element or jQuery) the parent node for the text (optional)
		@param  x         (number or number[]) the x-coordinate(s) for the text (optional)
		@param  y         (number or number[]) the y-coordinate(s) for the text (optional)
		@param  value     (string) the text content or
		(SVGText) text with spans and references
		@param  settings  (object) additional settings for the text (optional)
		@return  (element) the new text node */
        text: function (parent, x, y, value, settings) {
            var args = this._args(arguments, ['x', 'y', 'value']);
            if (typeof args.x == 'string' && arguments.length < 4) {
                args.value = args.x;
                args.settings = args.y;
                args.x = args.y = null;
            }
            return this._text(args.parent, 'text', args.value, $.extend(
			{
			    x: (args.x && isArray(args.x) ? args.x.join(' ') : args.x),
			    y: (args.y && isArray(args.y) ? args.y.join(' ') : args.y)
			},
			args.settings || {}));
        },

        /* Draw text along a path.
		@param  parent    (element or jQuery) the parent node for the text (optional)
		@param  path      (string) the ID of the path
		@param  value     (string) the text content or
		(SVGText) text with spans and references
		@param  settings  (object) additional settings for the text (optional)
		@return  (element) the new text node */
        textpath: function (parent, path, value, settings) {
            var args = this._args(arguments, ['path', 'value']);
            var node = this._text(args.parent, 'textPath', args.value, args.settings || {});
            node.setAttributeNS($.svg.xlinkNS, 'href', args.path);
            return node;
        },

        /* Draw text. */
        _text: function (parent, name, value, settings) {
            var node = this._makeNode(parent, name, settings);
            if (typeof value == 'string') {
                node.appendChild(node.ownerDocument.createTextNode(value));
            }
            else {
                for (var i = 0; i < value._parts.length; i++) {
                    var part = value._parts[i];
                    if (part[0] == 'tspan') {
                        var child = this._makeNode(node, part[0], part[2]);
                        child.appendChild(node.ownerDocument.createTextNode(part[1]));
                        node.appendChild(child);
                    }
                    else if (part[0] == 'tref') {
                        var child = this._makeNode(node, part[0], part[2]);
                        child.setAttributeNS($.svg.xlinkNS, 'href', part[1]);
                        node.appendChild(child);
                    }
                    else if (part[0] == 'textpath') {
                        var set = $.extend({}, part[2]);
                        set.href = null;
                        var child = this._makeNode(node, part[0], set);
                        child.setAttributeNS($.svg.xlinkNS, 'href', part[2].href);
                        child.appendChild(node.ownerDocument.createTextNode(part[1]));
                        node.appendChild(child);
                    }
                    else { // straight text
                        node.appendChild(node.ownerDocument.createTextNode(part[1]));
                    }
                }
            }
            return node;
        },

        /* Add a custom SVG element.
		@param  parent    (element or jQuery) the parent node for the new element (optional)
		@param  name      (string) the name of the element
		@param  settings  (object) additional settings for the element (optional)
		@return  (element) the new custom node */
        other: function (parent, name, settings) {
            var args = this._args(arguments, ['name']);
            return this._makeNode(args.parent, args.name, args.settings || {});
        },

        /* Create a shape node with the given settings. */
        _makeNode: function (parent, name, settings) {
            parent = parent || this._svg;
            var node = this._svg.ownerDocument.createElementNS($.svg.svgNS, name);
            for (var name in settings) {
                var value = settings[name];
                if (value !== null && value !== '') {
                    if ($.svg._styleAttrNames[name]) {
                        node.style[$.svg._styleAttrNames[name] || name] = value;
                    } else {
                        node.setAttribute($.svg._attrNames[name] || name, value);
                    }
                }
            }
            parent.appendChild(node);
            return node;
        },

        /* Add an existing SVG node to the diagram.
		@param  parent  (element or jQuery) the parent node for the new node (optional)
		@param  node    (element) the new node to add or
		(string) the jQuery selector for the node or
		(jQuery collection) set of nodes to add
		@return  (SVGWrapper) this wrapper */
        add: function (parent, node) {
            var args = this._args((arguments.length == 1 ? [null, parent] : arguments), ['node']);
            var svg = this;
            args.parent = args.parent || this._svg;
            args.node = (args.node.jquery ? args.node : $(args.node));
            try {
                if ($.svg._renesis) {
                    throw 'Force traversal';
                }
                args.parent.appendChild(args.node.cloneNode(true));
            }
            catch (e) {
                args.node.each(function () {
                    var child = svg._cloneAsSVG(this);
                    if (child) {
                        args.parent.appendChild(child);
                    }
                });
            }
            return this;
        },

        /* Clone an existing SVG node and add it to the diagram.
		@param  parent  (element or jQuery) the parent node for the new node (optional)
		@param  node    (element) the new node to add or
		(string) the jQuery selector for the node or
		(jQuery collection) set of nodes to add
		@return  (element[]) collection of new nodes */
        clone: function (parent, node) {
            var svg = this;
            var args = this._args((arguments.length == 1 ? [null, parent] : arguments), ['node']);
            args.parent = args.parent || this._svg;
            args.node = (args.node.jquery ? args.node : $(args.node));
            var newNodes = [];
            args.node.each(function () {
                var child = svg._cloneAsSVG(this);
                if (child) {
                    child.id = '';
                    args.parent.appendChild(child);
                    newNodes.push(child);
                }
            });
            return newNodes;
        },

        /* SVG nodes must belong to the SVG namespace, so clone and ensure this is so.
		@param  node  (element) the SVG node to clone
		@return  (element) the cloned node */
        _cloneAsSVG: function (node) {
            var newNode = null;
            if (node.nodeType == 1) { // element
                newNode = this._svg.ownerDocument.createElementNS(
				$.svg.svgNS, this._checkName(node.nodeName));
                for (var i = 0; i < node.attributes.length; i++) {
                    var attr = node.attributes.item(i);
                    if (attr.nodeName != 'xmlns' && attr.nodeValue) {
                        if (attr.prefix == 'xlink') {
                            newNode.setAttributeNS($.svg.xlinkNS,
							attr.localName || attr.baseName, attr.nodeValue);
                        }
                        else {
                            newNode.setAttribute(this._checkName(attr.nodeName), attr.nodeValue);
                        }
                    }
                }
                for (var i = 0; i < node.childNodes.length; i++) {
                    var child = this._cloneAsSVG(node.childNodes[i]);
                    if (child) {
                        newNode.appendChild(child);
                    }
                }
            }
            else if (node.nodeType == 3) { // text
                if ($.trim(node.nodeValue)) {
                    newNode = this._svg.ownerDocument.createTextNode(node.nodeValue);
                }
            }
            else if (node.nodeType == 4) { // CDATA
                if ($.trim(node.nodeValue)) {
                    try {
                        newNode = this._svg.ownerDocument.createCDATASection(node.nodeValue);
                    }
                    catch (e) {
                        newNode = this._svg.ownerDocument.createTextNode(
						node.nodeValue.replace(/&/g, '&amp;').
						replace(/</g, '&lt;').replace(/>/g, '&gt;'));
                    }
                }
            }
            return newNode;
        },

        /* Node names must be lower case and without SVG namespace prefix. */
        _checkName: function (name) {
            name = (name.substring(0, 1) >= 'A' && name.substring(0, 1) <= 'Z' ?
			name.toLowerCase() : name);
            return (name.substring(0, 4) == 'svg:' ? name.substring(4) : name);
        },

        /* Load an external SVG document.
		@param  url       (string) the location of the SVG document or
		the actual SVG content
		@param  settings  (boolean) see addTo below or
		(function) see onLoad below or
		(object) additional settings for the load with attributes below:
		addTo       (boolean) true to add to what's already there,
		or false to clear the canvas first
		changeSize  (boolean) true to allow the canvas size to change,
		or false to retain the original
		onLoad      (function) callback after the document has loaded,
		'this' is the container, receives SVG object and
		optional error message as a parameter
		parent      (string or element or jQuery) the parent to load
		into, defaults to top-level svg element
		@return  (SVGWrapper) this root */
        load: function (url, settings) {
            settings = (typeof settings == 'boolean' ? { addTo: settings } :
			(typeof settings == 'function' ? { onLoad: settings } :
			(typeof settings == 'string' ? { parent: settings } :
			(typeof settings == 'object' && settings.nodeName ? { parent: settings } :
			(typeof settings == 'object' && settings.jquery ? { parent: settings } :
			settings || {})))));
            if (!settings.parent && !settings.addTo) {
                this.clear(false);
            }
            var size = [this._svg.getAttribute('width'), this._svg.getAttribute('height')];
            var wrapper = this;
            // Report a problem with the load
            var reportError = function (message) {
                message = $.svg.local.errorLoadingText + ': ' + message;
                if (settings.onLoad) {
                    settings.onLoad.apply(wrapper._container || wrapper._svg, [wrapper, message]);
                }
                else {
                    wrapper.text(null, 10, 20, message);
                }
            };
            // Create a DOM from SVG content
            var loadXML4IE = function (data) {
                var xml = new ActiveXObject('Microsoft.XMLDOM');
                xml.validateOnParse = false;
                xml.resolveExternals = false;
                xml.async = false;
                xml.loadXML(data);
                if (xml.parseError.errorCode != 0) {
                    reportError(xml.parseError.reason);
                    return null;
                }
                return xml;
            };
            // Load the SVG DOM
            var loadSVG = function (data) {
                if (!data) {
                    return;
                }
                if (data.documentElement.nodeName != 'svg') {
                    var errors = data.getElementsByTagName('parsererror');
                    var messages = (errors.length ? errors[0].getElementsByTagName('div') : []); // Safari
                    reportError(!errors.length ? '???' :
					(messages.length ? messages[0] : errors[0]).firstChild.nodeValue);
                    return;
                }
                var parent = (settings.parent ? $(settings.parent)[0] : wrapper._svg);
                var attrs = {};
                for (var i = 0; i < data.documentElement.attributes.length; i++) {
                    var attr = data.documentElement.attributes.item(i);
                    if (!(attr.nodeName == 'version' || attr.nodeName.substring(0, 5) == 'xmlns')) {
                        attrs[attr.nodeName] = attr.nodeValue;
                    }
                }
                wrapper.configure(parent, attrs, !settings.parent);
                var nodes = data.documentElement.childNodes;
                for (var i = 0; i < nodes.length; i++) {
                    try {
                        if ($.svg._renesis) {
                            throw 'Force traversal';
                        }
                        parent.appendChild(wrapper._svg.ownerDocument.importNode(nodes[i], true));
                        if (nodes[i].nodeName == 'script') {
                            $.globalEval(nodes[i].textContent);
                        }
                    }
                    catch (e) {
                        wrapper.add(parent, nodes[i]);
                    }
                }
                if (!settings.changeSize) {
                    wrapper.configure(parent, { width: size[0], height: size[1] });
                }
                if (settings.onLoad) {
                    settings.onLoad.apply(wrapper._container || wrapper._svg, [wrapper]);
                }
            };
            if (url.match('<svg')) { // Inline SVG
                loadSVG($.browser.msie ? loadXML4IE(url) :
				new DOMParser().parseFromString(url, 'text/xml'));
            }
            else { // Remote SVG
                $.ajax({
                    url: url, dataType: ($.browser.msie ? 'text' : 'xml'),
                    success: function (xml) {
                        loadSVG($.browser.msie ? loadXML4IE(xml) : xml);
                    }, error: function (http, message, exc) {
                        reportError(message + (exc ? ' ' + exc.message : ''));
                    }
                });
            }
            return this;
        },

        /* Delete a specified node.
		@param  node  (element or jQuery) the drawing node to remove
		@return  (SVGWrapper) this root */
        remove: function (node) {
            node = (node.jquery ? node[0] : node);
            node.parentNode.removeChild(node);
            return this;
        },

        /* Delete everything in the current document.
		@param  attrsToo  (boolean) true to clear any root attributes as well,
		false to leave them (optional)
		@return  (SVGWrapper) this root */
        clear: function (attrsToo) {
            if (attrsToo) {
                this.configure({}, true);
            }
            while (this._svg.firstChild) {
                this._svg.removeChild(this._svg.firstChild);
            }
            return this;
        },

        /* Serialise the current diagram into an SVG text document.
		@param  node  (SVG element) the starting node (optional)
		@return  (string) the SVG as text */
        toSVG: function (node) {
            node = node || this._svg;
            return (typeof XMLSerializer == 'undefined' ? this._toSVG(node) :
			new XMLSerializer().serializeToString(node));
        },

        /* Serialise one node in the SVG hierarchy. */
        _toSVG: function (node) {
            var svgDoc = '';
            if (!node) {
                return svgDoc;
            }
            if (node.nodeType == 3) { // Text
                svgDoc = node.nodeValue;
            }
            else if (node.nodeType == 4) { // CDATA
                svgDoc = '<![CDATA[' + node.nodeValue + ']]>';
            }
            else { // Element
                svgDoc = '<' + node.nodeName;
                if (node.attributes) {
                    for (var i = 0; i < node.attributes.length; i++) {
                        var attr = node.attributes.item(i);
                        if (!($.trim(attr.nodeValue) == '' || attr.nodeValue.match(/^\[object/) ||
							attr.nodeValue.match(/^function/))) {
                            svgDoc += ' ' + (attr.namespaceURI == $.svg.xlinkNS ? 'xlink:' : '') +
							attr.nodeName + '="' + attr.nodeValue + '"';
                        }
                    }
                }
                if (node.firstChild) {
                    svgDoc += '>';
                    var child = node.firstChild;
                    while (child) {
                        svgDoc += this._toSVG(child);
                        child = child.nextSibling;
                    }
                    svgDoc += '</' + node.nodeName + '>';
                }
                else {
                    svgDoc += '/>';
                }
            }
            return svgDoc;
        }
    });

    /* Helper to generate an SVG path.
	Obtain an instance from the SVGWrapper object.
	String calls together to generate the path and use its value:
	var path = root.createPath();
	root.path(null, path.move(100, 100).line(300, 100).line(200, 300).close(), {fill: 'red'});
	or
	root.path(null, path.move(100, 100).line([[300, 100], [200, 300]]).close(), {fill: 'red'}); */
    function SVGPath() {
        this._path = '';
    }

    $.extend(SVGPath.prototype, {
        /* Prepare to create a new path.
		@return  (SVGPath) this path */
        reset: function () {
            this._path = '';
            return this;
        },

        /* Move the pointer to a position.
		@param  x         (number) x-coordinate to move to or
		(number[][]) x-/y-coordinates to move to
		@param  y         (number) y-coordinate to move to (omitted if x is array)
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        move: function (x, y, relative) {
            relative = (isArray(x) ? y : relative);
            return this._coords((relative ? 'm' : 'M'), x, y);
        },

        /* Draw a line to a position.
		@param  x         (number) x-coordinate to move to or
		(number[][]) x-/y-coordinates to move to
		@param  y         (number) y-coordinate to move to (omitted if x is array)
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        line: function (x, y, relative) {
            relative = (isArray(x) ? y : relative);
            return this._coords((relative ? 'l' : 'L'), x, y);
        },

        /* Draw a horizontal line to a position.
		@param  x         (number) x-coordinate to draw to or
		(number[]) x-coordinates to draw to
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        horiz: function (x, relative) {
            this._path += (relative ? 'h' : 'H') + (isArray(x) ? x.join(' ') : x);
            return this;
        },

        /* Draw a vertical line to a position.
		@param  y         (number) y-coordinate to draw to or
		(number[]) y-coordinates to draw to
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        vert: function (y, relative) {
            this._path += (relative ? 'v' : 'V') + (isArray(y) ? y.join(' ') : y);
            return this;
        },

        /* Draw a cubic Bezier curve.
		@param  x1        (number) x-coordinate of beginning control point or
		(number[][]) x-/y-coordinates of control and end points to draw to
		@param  y1        (number) y-coordinate of beginning control point (omitted if x1 is array)
		@param  x2        (number) x-coordinate of ending control point (omitted if x1 is array)
		@param  y2        (number) y-coordinate of ending control point (omitted if x1 is array)
		@param  x         (number) x-coordinate of curve end (omitted if x1 is array)
		@param  y         (number) y-coordinate of curve end (omitted if x1 is array)
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        curveC: function (x1, y1, x2, y2, x, y, relative) {
            relative = (isArray(x1) ? y1 : relative);
            return this._coords((relative ? 'c' : 'C'), x1, y1, x2, y2, x, y);
        },

        /* Continue a cubic Bezier curve.
		Starting control point is the reflection of the previous end control point.
		@param  x2        (number) x-coordinate of ending control point or
		(number[][]) x-/y-coordinates of control and end points to draw to
		@param  y2        (number) y-coordinate of ending control point (omitted if x2 is array)
		@param  x         (number) x-coordinate of curve end (omitted if x2 is array)
		@param  y         (number) y-coordinate of curve end (omitted if x2 is array)
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        smoothC: function (x2, y2, x, y, relative) {
            relative = (isArray(x2) ? y2 : relative);
            return this._coords((relative ? 's' : 'S'), x2, y2, x, y);
        },

        /* Draw a quadratic Bezier curve.
		@param  x1        (number) x-coordinate of control point or
		(number[][]) x-/y-coordinates of control and end points to draw to
		@param  y1        (number) y-coordinate of control point (omitted if x1 is array)
		@param  x         (number) x-coordinate of curve end (omitted if x1 is array)
		@param  y         (number) y-coordinate of curve end (omitted if x1 is array)
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        curveQ: function (x1, y1, x, y, relative) {
            relative = (isArray(x1) ? y1 : relative);
            return this._coords((relative ? 'q' : 'Q'), x1, y1, x, y);
        },

        /* Continue a quadratic Bezier curve.
		Control point is the reflection of the previous control point.
		@param  x         (number) x-coordinate of curve end or
		(number[][]) x-/y-coordinates of points to draw to
		@param  y         (number) y-coordinate of curve end (omitted if x is array)
		@param  relative  (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        smoothQ: function (x, y, relative) {
            relative = (isArray(x) ? y : relative);
            return this._coords((relative ? 't' : 'T'), x, y);
        },

        /* Generate a path command with (a list of) coordinates. */
        _coords: function (cmd, x1, y1, x2, y2, x3, y3) {
            if (isArray(x1)) {
                for (var i = 0; i < x1.length; i++) {
                    var cs = x1[i];
                    this._path += (i == 0 ? cmd : ' ') + cs[0] + ',' + cs[1] +
					(cs.length < 4 ? '' : ' ' + cs[2] + ',' + cs[3] +
					(cs.length < 6 ? '' : ' ' + cs[4] + ',' + cs[5]));
                }
            }
            else {
                this._path += cmd + x1 + ',' + y1 +
				(x2 == null ? '' : ' ' + x2 + ',' + y2 +
				(x3 == null ? '' : ' ' + x3 + ',' + y3));
            }
            return this;
        },

        /* Draw an arc to a position.
		@param  rx         (number) x-radius of arc or
		(number/boolean[][]) x-/y-coordinates and flags for points to draw to
		@param  ry         (number) y-radius of arc (omitted if rx is array)
		@param  xRotate    (number) x-axis rotation (degrees, clockwise) (omitted if rx is array)
		@param  large      (boolean) true to draw the large part of the arc,
		false to draw the small part (omitted if rx is array)
		@param  clockwise  (boolean) true to draw the clockwise arc,
		false to draw the anti-clockwise arc (omitted if rx is array)
		@param  x          (number) x-coordinate of arc end (omitted if rx is array)
		@param  y          (number) y-coordinate of arc end (omitted if rx is array)
		@param  relative   (boolean) true for coordinates relative to the current point,
		false for coordinates being absolute
		@return  (SVGPath) this path */
        arc: function (rx, ry, xRotate, large, clockwise, x, y, relative) {
            relative = (isArray(rx) ? ry : relative);
            this._path += (relative ? 'a' : 'A');
            if (isArray(rx)) {
                for (var i = 0; i < rx.length; i++) {
                    var cs = rx[i];
                    this._path += (i == 0 ? '' : ' ') + cs[0] + ',' + cs[1] + ' ' +
					cs[2] + ' ' + (cs[3] ? '1' : '0') + ',' +
					(cs[4] ? '1' : '0') + ' ' + cs[5] + ',' + cs[6];
                }
            }
            else {
                this._path += rx + ',' + ry + ' ' + xRotate + ' ' +
				(large ? '1' : '0') + ',' + (clockwise ? '1' : '0') + ' ' + x + ',' + y;
            }
            return this;
        },

        /* Close the current path.
		@return  (SVGPath) this path */
        close: function () {
            this._path += 'z';
            return this;
        },

        /* Return the string rendering of the specified path.
		@return  (string) stringified path */
        path: function () {
            return this._path;
        }
    });

    SVGPath.prototype.moveTo = SVGPath.prototype.move;
    SVGPath.prototype.lineTo = SVGPath.prototype.line;
    SVGPath.prototype.horizTo = SVGPath.prototype.horiz;
    SVGPath.prototype.vertTo = SVGPath.prototype.vert;
    SVGPath.prototype.curveCTo = SVGPath.prototype.curveC;
    SVGPath.prototype.smoothCTo = SVGPath.prototype.smoothC;
    SVGPath.prototype.curveQTo = SVGPath.prototype.curveQ;
    SVGPath.prototype.smoothQTo = SVGPath.prototype.smoothQ;
    SVGPath.prototype.arcTo = SVGPath.prototype.arc;

    /* Helper to generate an SVG text object.
	Obtain an instance from the SVGWrapper object.
	String calls together to generate the text and use its value:
	var text = root.createText();
	root.text(null, x, y, text.string('This is ').
	span('red', {fill: 'red'}).string('!'), {fill: 'blue'}); */
    function SVGText() {
        this._parts = []; // The components of the text object
    }

    $.extend(SVGText.prototype, {
        /* Prepare to create a new text object.
		@return  (SVGText) this text */
        reset: function () {
            this._parts = [];
            return this;
        },

        /* Add a straight string value.
		@param  value  (string) the actual text
		@return  (SVGText) this text object */
        string: function (value) {
            this._parts[this._parts.length] = ['text', value];
            return this;
        },

        /* Add a separate text span that has its own settings.
		@param  value     (string) the actual text
		@param  settings  (object) the settings for this text
		@return  (SVGText) this text object */
        span: function (value, settings) {
            this._parts[this._parts.length] = ['tspan', value, settings];
            return this;
        },

        /* Add a reference to a previously defined text string.
		@param  id        (string) the ID of the actual text
		@param  settings  (object) the settings for this text
		@return  (SVGText) this text object */
        ref: function (id, settings) {
            this._parts[this._parts.length] = ['tref', id, settings];
            return this;
        },

        /* Add text drawn along a path.
		@param  id        (string) the ID of the path
		@param  value     (string) the actual text
		@param  settings  (object) the settings for this text
		@return  (SVGText) this text object */
        path: function (id, value, settings) {
            this._parts[this._parts.length] = ['textpath', value,
			$.extend({ href: id }, settings || {})];
            return this;
        }
    });

    /* Attach the SVG functionality to a jQuery selection.
	@param  command  (string) the command to run (optional, default 'attach')
	@param  options  (object) the new settings to use for these SVG instances
	@return jQuery (object) for chaining further calls */
    $.fn.svg = function (options) {
        var otherArgs = Array.prototype.slice.call(arguments, 1);
        if (typeof options == 'string' && options == 'get') {
            return $.svg['_' + options + 'SVG'].apply($.svg, [this[0]].concat(otherArgs));
        }
        return this.each(function () {
            if (typeof options == 'string') {
                $.svg['_' + options + 'SVG'].apply($.svg, [this].concat(otherArgs));
            }
            else {
                $.svg._attachSVG(this, options || {});
            }
        });
    };

    /* Determine whether an object is an array. */
    function isArray(a) {
        return (a && a.constructor == Array);
    }

    // Singleton primary SVG interface
    $.svg = new SVGManager();

})(jQuery);


/* http://keith-wood.name/svg.html
   SVG attribute animations for jQuery v1.4.5.
   Written by Keith Wood (kbwood{at}iinet.com.au) June 2008.
   Dual licensed under the GPL (http://dev.jquery.com/browser/trunk/jquery/GPL-LICENSE.txt) and 
   MIT (http://dev.jquery.com/browser/trunk/jquery/MIT-LICENSE.txt) licenses. 
   Please attribute the author if you use it. */
eval(function (p, a, c, k, e, r) { e = function (c) { return (c < a ? '' : e(parseInt(c / a))) + ((c = c % a) > 35 ? String.fromCharCode(c + 29) : c.toString(36)) }; if (!''.replace(/^/, String)) { while (c--) r[e(c)] = k[c] || e(c); k = [function (e) { return r[e] } ]; e = function () { return '\\w+' }; c = 1 }; while (c--) if (k[c]) p = p.replace(new RegExp('\\b' + e(c) + '\\b', 'g'), k[c]); return p } ('(A($){$.2f([\'x\',\'y\',\'2g\',\'2S\',\'2T\',\'2U\',\'2V\',\'2W\',\'r\',\'2X\',\'2Y\',\'2Z\',\'31\',\'U-2g\',\'33\',\'1F\',\'2h-1F\',\'36\',\'U-1F\',\'37\',\'U-38\',\'39\',\'2i-3a\',\'3b\',\'2i-3c\',\'3d\',\'3e-2j\',\'3f\',\'3g-2j\',\'3h\'],A(i,f){l g=f.1G(0).2k()+f.1H(1);q($.1I){$.1I[\'u\'+g]=$.1I[\'u-\'+f]=f}$.D.E[\'u\'+g]=$.D.E[\'u-\'+f]=A(a){l b=$.u.3i[f]||f;l c=a.B.1l.1m(b);q(!a.N){a.k=(c?p(c.O):0);l d=($.1J.1K>=\'1.6\'?\'\':a.V.W[\'u\'+g]||a.V.W[\'u-\'+f]);q(/^[+-]=/.L(d)){a.o=a.k+p(d.1L(/=/,\'\'))}$(a.B).1M(b,\'\');a.N=1n}l e=(a.w*(a.o-a.k)+a.k)+(a.3j==\'%\'?\'%\':\'\');(c?c.O=e:a.B.1o(b,e))}});$.D.E[\'2l\']=$.D.E[\'u-2m\']=$.D.E[\'2n-17\']=$.D.E[\'u-U-17\']=A(a){l b=a.B.1l.1m(\'U-17\');q(!a.N){a.k=1N(b?b.O:\'\');l c=($.1J.1K>=\'1.6\'?a.o:a.V.W[\'2l\']||a.V.W[\'u-2m\']||a.V.W[\'2n-17\']||a.V.W[\'u-U-17\']);a.o=1N(c);q(/^[+-]=/.L(c)){c=c.1r(/[, ]+/);q(c.F%2==1){l d=c.F;X(l i=0;i<d;i++){c.1s(c[i])}}X(l i=0;i<c.F;i++){q(/^[+-]=/.L(c[i])){a.o[i]=a.k[i]+p(c[i].1L(/=/,\'\'))}}}a.N=1n}l e=$.2o(a.k,A(n,i){G(a.w*(a.o[i]-n)+n)}).1O(\',\');(b?b.O=e:a.B.1o(\'U-17\',e))};A 1N(a){l b=a.1r(/[, ]+/);X(l i=0;i<b.F;i++){b[i]=p(b[i]);q(2p(b[i])){b[i]=0}}q(b.F%2==1){l c=b.F;X(l i=0;i<c;i++){b.1s(b[i])}}G b}$.D.E[\'2q\']=$.D.E[\'u-1t\']=A(a){l b=a.B.1l.1m(\'1t\');q(!a.N){a.k=1P(b?b.O:\'\');l c=($.1J.1K>=\'1.6\'?a.o:a.V.W[\'2q\']||a.V.W[\'u-1t\']);a.o=1P(c);q(/^[+-]=/.L(c)){c=c.1r(/[, ]+/);1u(c.F<4){c.1s(\'0\')}X(l i=0;i<4;i++){q(/^[+-]=/.L(c[i])){a.o[i]=a.k[i]+p(c[i].1L(/=/,\'\'))}}}a.N=1n}l d=$.2o(a.k,A(n,i){G(a.w*(a.o[i]-n)+n)}).1O(\' \');(b?b.O=d:a.B.1o(\'1t\',d))};A 1P(a){l b=a.1r(/[, ]+/);X(l i=0;i<b.F;i++){b[i]=p(b[i]);q(2p(b[i])){b[i]=0}}1u(b.F<4){b.1s(0)}G b}$.D.E[\'3k\']=$.D.E[\'u-1Q\']=A(a){l b=a.B.1l.1m(\'1Q\');q(!a.N){a.k=1R(b?b.O:\'\');a.o=1R(a.o,a.k);a.N=1n}l c=\'\';X(l i=0;i<a.o.M.F;i++){2r(a.o.M.1G(i)){H\'t\':c+=\' 2s(\'+(a.w*(a.o.18-a.k.18)+a.k.18)+\',\'+(a.w*(a.o.1a-a.k.1a)+a.k.1a)+\')\';I;H\'s\':c+=\' 2t(\'+(a.w*(a.o.1b-a.k.1b)+a.k.1b)+\',\'+(a.w*(a.o.1p-a.k.1p)+a.k.1p)+\')\';I;H\'r\':c+=\' 2u(\'+(a.w*(a.o.1c-a.k.1c)+a.k.1c)+\',\'+(a.w*(a.o.1d-a.k.1d)+a.k.1d)+\',\'+(a.w*(a.o.1e-a.k.1e)+a.k.1e)+\')\';I;H\'x\':c+=\' 13(\'+(a.w*(a.o.13-a.k.13)+a.k.13)+\')\';H\'y\':c+=\' 14(\'+(a.w*(a.o.14-a.k.14)+a.k.14)+\')\';I;H\'m\':l d=\'\';X(l j=0;j<6;j++){d+=\',\'+(a.w*(a.o.z[j]-a.k.z[j])+a.k.z[j])}c+=\' z(\'+d.1H(1)+\')\';I}}(b?b.O=c:a.B.1o(\'1Q\',c))};A 1R(a,b){a=a||\'\';q(3l a==\'3m\'){a=a.O}l c=$.3n({18:0,1a:0,1b:0,1p:0,1c:0,1d:0,1e:0,13:0,14:0,z:[0,0,0,0,0,0]},b||{});c.M=\'\';l d=/([a-3o-Z]+)\\(\\s*([+-]?[\\d\\.]+)\\s*(?:[\\s,]\\s*([+-]?[\\d\\.]+)\\s*(?:[\\s,]\\s*([+-]?[\\d\\.]+)\\s*(?:[\\s,]\\s*([+-]?[\\d\\.]+)\\s*[\\s,]\\s*([+-]?[\\d\\.]+)\\s*[\\s,]\\s*([+-]?[\\d\\.]+)\\s*)?)?)?\\)/g;l e=d.L(a);1u(e){2r(e[1]){H\'2s\':c.M+=\'t\';c.18=p(e[2]);c.1a=(e[3]?p(e[3]):0);I;H\'2t\':c.M+=\'s\';c.1b=p(e[2]);c.1p=(e[3]?p(e[3]):c.1b);I;H\'2u\':c.M+=\'r\';c.1c=p(e[2]);c.1d=(e[3]?p(e[3]):0);c.1e=(e[4]?p(e[4]):0);I;H\'13\':c.M+=\'x\';c.13=p(e[2]);I;H\'14\':c.M+=\'y\';c.14=p(e[2]);I;H\'z\':c.M+=\'m\';c.z=[p(e[2]),p(e[3]),p(e[4]),p(e[5]),p(e[6]),p(e[7])];I}e=d.L(a)}q(c.M==\'m\'&&J.1v(c.z[0])==J.1v(c.z[3])&&c.z[1]!=0&&J.1v(c.z[1])==J.1v(c.z[2])){l f=J.3p(c.z[0])*1w/J.3q;f=(c.z[1]<0?3r-f:f);c.M=\'3s\';c.1c=f;c.1d=c.1e=0;c.18=c.z[4];c.1a=c.z[5]}G c}$.2f([\'2h\',\'U\'],A(i,e){l f=e.1G(0).2k()+e.1H(1);$.D.E[\'u\'+f]=$.D.E[\'u-\'+e]=A(a){q(!a.N){a.k=$.u.1S(a.B,e);l b=(a.o==\'1f\');a.o=(b?$.u.1S(a.B.3t,e):$.u.1T(a.o));a.o[3]=b;$(a.B).1M(e,\'\');a.N=1n}l c=a.B.1l.1m(e);l d=\'1U(\'+[J.1V(J.1W(K((a.w*(a.o[0]-a.k[0]))+a.k[0],10),0),8),J.1V(J.1W(K((a.w*(a.o[1]-a.k[1]))+a.k[1],10),0),8),J.1V(J.1W(K((a.w*(a.o[2]-a.k[2]))+a.k[2],10),0),8)].1O(\',\')+\')\';d=(a.o[3]&&a.3u==1?\'1f\':d);(c?c.O=d:a.B.1o(e,d))}});$.u.1S=A(a,b){a=$(a);l c;3v{c=a.3w(b)||a.1M(b);q((c!=\'\'&&c!=\'1f\')||a.3x($.u.3y)){I}}1u(a=a.3z());G $.u.1T(c)};$.u.1T=A(a){l b;q(a&&a.3A==3B){G(a.F==3||a.F==4?a:h[\'1f\'])}q(b=/^1U\\(\\s*([0-9]{1,3})\\s*,\\s*([0-9]{1,3})\\s*,\\s*([0-9]{1,3})\\s*\\)$/.L(a)){G[K(b[1],10),K(b[2],10),K(b[3],10)]}q(b=/^1U\\(\\s*([0-9]+(?:\\.[0-9]+)?)\\%\\s*,\\s*([0-9]+(?:\\.[0-9]+)?)\\%\\s*,\\s*([0-9]+(?:\\.[0-9]+)?)\\%\\s*\\)$/.L(a)){G[p(b[1])*2.55,p(b[2])*2.55,p(b[3])*2.55]}q(b=/^#([a-1g-1h-9]{2})([a-1g-1h-9]{2})([a-1g-1h-9]{2})$/.L(a)){G[K(b[1],16),K(b[2],16),K(b[3],16)]}q(b=/^#([a-1g-1h-9])([a-1g-1h-9])([a-1g-1h-9])$/.L(a)){G[K(b[1]+b[1],16),K(b[2]+b[2],16),K(b[3]+b[3],16)]}G h[$.3C(a).3D()]||h[\'1f\']};l h={\'\':[8,8,8,1],1f:[8,8,8,1],3E:[P,1x,8],3F:[C,1X,2v],3G:[0,8,8],3H:[1y,8,3I],3J:[P,8,8],3K:[Q,Q,1i],3L:[8,1Y,2w],3M:[0,0,0],3N:[8,1X,S],3O:[0,0,8],3P:[3Q,43,3R],3S:[1Z,42,42],3T:[1z,2x,22],3U:[3V,3W,1A],3X:[1y,8,0],3Y:[23,T,30],3Z:[8,1y,40],41:[2y,44,48],49:[8,1x,1i],4a:[1i,20,60],4b:[0,8,8],4c:[0,0,R],4d:[0,R,R],4e:[2x,4f,11],4g:[1j,1j,1j],4h:[0,2y,0],4i:[1j,1j,1j],4j:[4k,4l,24],4m:[R,0,R],4n:[2z,24,47],4o:[8,26,0],4p:[27,50,2A],4q:[R,0,0],4r:[4s,4t,2B],4u:[1B,2C,1B],4v:[2D,61,R],4w:[47,1C,1C],4x:[47,1C,1C],4y:[0,28,2E],4z:[4A,0,Y],4B:[8,20,29],4C:[0,2F,8],4D:[T,T,T],4E:[T,T,T],4F:[30,1q,8],4G:[2G,34,34],4H:[8,C,P],4I:[34,R,34],4J:[8,0,8],4K:[1i,1i,1i],4L:[1x,1x,8],4M:[8,2v,0],4N:[2a,1Z,32],4O:[v,v,v],4P:[v,v,v],4Q:[0,v,0],4R:[2b,8,47],4S:[P,8,P],4T:[8,T,1w],4U:[S,2H,2H],4V:[4W,0,2c],4X:[8,8,P],4Y:[P,15,26],4Z:[15,15,C],51:[8,P,Q],52:[53,54,0],56:[8,C,S],57:[2b,2d,15],58:[P,v,v],59:[1D,8,8],5a:[C,C,23],5b:[Y,Y,Y],5c:[1q,12,1q],5d:[Y,Y,Y],5e:[8,5f,5g],5h:[8,1A,2B],5i:[32,2G,2e],5j:[22,28,C],5k:[2I,2J,27],5l:[2I,2J,27],5m:[2K,2w,1z],5n:[8,8,1D],5o:[0,8,0],5p:[50,S,50],5q:[C,P,15],5r:[8,0,8],5s:[v,0,0],5t:[5u,S,2e],5v:[0,0,S],5w:[5x,2z,Y],5y:[29,1k,2L],5z:[60,2M,5A],5B:[5C,5D,12],5E:[0,C,2N],5F:[2D,2E,2A],5G:[5H,21,2O],5I:[25,25,1k],5J:[Q,8,C],5K:[8,1Y,2P],5L:[8,1Y,5M],5N:[8,1z,2b],5O:[0,0,v],5P:[5Q,Q,15],5R:[v,v,0],5S:[24,5T,35],5U:[8,1Z,0],5V:[8,69,0],5W:[2a,1k,5X],5Y:[12,5Z,2e],62:[2Q,66,2Q],67:[68,12,12],6a:[2L,1k,29],6b:[8,6c,6d],6e:[8,2a,6f],6g:[S,2O,63],6h:[8,1E,6i],6j:[2R,1A,2R],6k:[2K,1D,15],6l:[v,0,v],6m:[8,0,0],6n:[2C,1B,1B],6o:[65,T,2P],6p:[R,69,19],6q:[C,v,6r],6s:[6t,6u,6v],6w:[46,R,6x],6y:[8,Q,12],6z:[1A,6A,45],6B:[1E,1E,1E],6C:[22,28,1X],6D:[6E,6F,S],6G:[1k,v,1q],6H:[1k,v,1q],6I:[8,C,C],6J:[0,8,1y],6K:[70,2c,1w],6L:[23,1w,26],6M:[0,v,v],6N:[2d,2F,2d],6O:[8,6P,6Q],6R:[64,1D,6S],6T:[12,2c,12],6U:[Q,1z,2M],6V:[8,8,8],6W:[Q,Q,Q],6X:[8,8,0],6Y:[2N,S,50]}})(6Z);', 62, 435, '||||||||255||||||||||||start|var|||end|parseFloat|if||||svg|128|pos|||matrix|function|elem|250|fx|step|length|return|case|break|Math|parseInt|exec|order|set|nodeValue|240|245|139|205|105|stroke|options|curAnim|for|211||||238|skewX|skewY|230||dasharray|translateX||translateY|scaleX|rotateA|rotateX|rotateY|none|fA|F0|220|169|112|attributes|getNamedItem|true|setAttribute|scaleY|144|split|push|viewBox|while|abs|180|248|127|222|160|143|79|224|192|opacity|charAt|substr|cssProps|fn|jquery|replace|css|parseDashArray|join|parseViewBox|transform|parseTransform|_getColour|_getRGB|rgb|min|max|235|228|165|||135|210|107||140|153|206|147|218|173|130|216|170|each|width|fill|font|spacing|toUpperCase|svgStrokeDashArray|strokeDashArray|svgStroke|map|isNaN|svgViewBox|switch|translate|scale|rotate|215|196|184|100|85|204|122|188|72|209|191|178|92|119|136|176|219|179|154|133|225|152|221|height|rx|ry|cx|cy|x1|y1|x2||y2||strokeWidth|||fillOpacity|strokeOpacity|dashoffset|strokeDashOffset|size|fontSize|weight|fontWeight|letter|letterSpacing|word|wordSpacing|_attrNames|unit|svgTransform|typeof|object|extend|zA|acos|PI|360|rt|parentNode|state|do|attr|hasClass|markerClassName|parent|constructor|Array|trim|toLowerCase|aliceblue|antiquewhite|aqua|aquamarine|212|azure|beige|bisque|black|blanchedalmond|blue|blueviolet|138|226|brown|burlywood|cadetblue|95|158|chartreuse|chocolate|coral|80|cornflowerblue|||149||||237|cornsilk|crimson|cyan|darkblue|darkcyan|darkgoldenrod|134|darkgray|darkgreen|darkgrey|darkkhaki|189|183|darkmagenta|darkolivegreen|darkorange|darkorchid|darkred|darksalmon|233|150|darkseagreen|darkslateblue|darkslategray|darkslategrey|darkturquoise|darkviolet|148|deeppink|deepskyblue|dimgray|dimgrey|dodgerblue|firebrick|floralwhite|forestgreen|fuchsia|gainsboro|ghostwhite|gold|goldenrod|gray|grey|green|greenyellow|honeydew|hotpink|indianred|indigo|75|ivory|khaki|lavender||lavenderblush|lawngreen|124|252||lemonchiffon|lightblue|lightcoral|lightcyan|lightgoldenrodyellow|lightgray|lightgreen|lightgrey|lightpink|182|193|lightsalmon|lightseagreen|lightskyblue|lightslategray|lightslategrey|lightsteelblue|lightyellow|lime|limegreen|linen|magenta|maroon|mediumaquamarine|102|mediumblue|mediumorchid|186|mediumpurple|mediumseagreen|113|mediumslateblue|123|104|mediumspringgreen|mediumturquoise|mediumvioletred|199|midnightblue|mintcream|mistyrose|moccasin|181|navajowhite|navy|oldlace|253|olive|olivedrab|142|orange|orangered|orchid|214|palegoldenrod|232|||palegreen||||251|paleturquoise|175||palevioletred|papayawhip|239|213|peachpuff|185|peru|pink|203|plum|powderblue|purple|red|rosybrown|royalblue|saddlebrown|salmon|114|sandybrown|244|164|96|seagreen|87|seashell|sienna|82|silver|skyblue|slateblue|106|90|slategray|slategrey|snow|springgreen|steelblue|tan|teal|thistle|tomato|99|71|turquoise|208|violet|wheat|white|whitesmoke|yellow|yellowgreen|jQuery|'.split('|'), 0, {}));



/* http://keith-wood.name/svg.html
   jQuery DOM compatibility for jQuery SVG v1.4.5.
   Written by Keith Wood (kbwood{at}iinet.com.au) April 2009.
   Dual licensed under the GPL (http://dev.jquery.com/browser/trunk/jquery/GPL-LICENSE.txt) and 
   MIT (http://dev.jquery.com/browser/trunk/jquery/MIT-LICENSE.txt) licenses. 
   Please attribute the author if you use it. */
/* http://keith-wood.name/svg.html
jQuery DOM compatibility for jQuery SVG v1.4.5.
Written by Keith Wood (kbwood{at}iinet.com.au) April 2009.
Dual licensed under the GPL (http://dev.jquery.com/browser/trunk/jquery/GPL-LICENSE.txt) and 
MIT (http://dev.jquery.com/browser/trunk/jquery/MIT-LICENSE.txt) licenses. 
Please attribute the author if you use it. */

function ext(jQuery) { // Hide scope, no $ conflict

	/* Support adding class names to SVG nodes. */
	$.fn.addClass = function (origAddClass) {
		return function (classNames) {
			classNames = classNames || '';
			return this.each(function () {
				if ($.svg.isSVGElem(this)) {
					var node = this;
					$.each(classNames.split(/\s+/), function (i, className) {
						var classes = (node.className ? node.className.baseVal : node.getAttribute('class'));
						if ($.inArray(className, classes.split(/\s+/)) == -1) {
							classes += (classes ? ' ' : '') + className;
							(node.className ? node.className.baseVal = classes :
							node.setAttribute('class', classes));
						}
					});
				}
				else {
					origAddClass.apply($(this), [classNames]);
				}
			});
		};
	} ($.fn.addClass);

	/* Support removing class names from SVG nodes. */
	$.fn.removeClass = function (origRemoveClass) {
		return function (classNames) {
			classNames = classNames || '';
			return this.each(function () {
				if ($.svg.isSVGElem(this)) {
					var node = this;
					$.each(classNames.split(/\s+/), function (i, className) {
						var classes = (node.className ? node.className.baseVal : node.getAttribute('class'));
						classes = $.grep(classes.split(/\s+/), function (n, i) { return n != className; }).
						join(' ');
						(node.className ? node.className.baseVal = classes :
						node.setAttribute('class', classes));
					});
				}
				else {
					origRemoveClass.apply($(this), [classNames]);
				}
			});
		};
	} ($.fn.removeClass);

	/* Support toggling class names on SVG nodes. */
	$.fn.toggleClass = function (origToggleClass) {
		return function (className, state) {
			return this.each(function () {
				if ($.svg.isSVGElem(this)) {
					if (typeof state !== 'boolean') {
						state = !$(this).hasClass(className);
					}
					$(this)[(state ? 'add' : 'remove') + 'Class'](className);
				}
				else {
					origToggleClass.apply($(this), [className, state]);
				}
			});
		};
	} ($.fn.toggleClass);

	/* Support checking class names on SVG nodes. */
	$.fn.hasClass = function (origHasClass) {
		return function (className) {
			className = className || '';
			var found = false;
			this.each(function () {
				if ($.svg.isSVGElem(this)) {
					var classes = (this.className ? this.className.baseVal :
					this.getAttribute('class')).split(/\s+/);
					found = ($.inArray(className, classes) > -1);
				}
				else {
					found = (origHasClass.apply($(this), [className]));
				}
				return !found;
			});
			return found;
		};
	} ($.fn.hasClass);

	/* Support attributes on SVG nodes. */
	$.fn.attr = function (origAttr) {
		return function (name, value, type) {
			if (typeof name === 'string' && value === undefined) {
				var val = origAttr.apply(this, [name]);
				if (val && val.baseVal && val.baseVal.numberOfItems != null) { // Multiple values
					value = '';
					val = val.baseVal;
					if (name == 'transform') {
						for (var i = 0; i < val.numberOfItems; i++) {
							var item = val.getItem(i);
							switch (item.type) {
								case 1: value += ' matrix(' + item.matrix.a + ',' + item.matrix.b + ',' +
										item.matrix.c + ',' + item.matrix.d + ',' +
										item.matrix.e + ',' + item.matrix.f + ')';
									break;
								case 2: value += ' translate(' + item.matrix.e + ',' + item.matrix.f + ')'; break;
								case 3: value += ' scale(' + item.matrix.a + ',' + item.matrix.d + ')'; break;
								case 4: value += ' rotate(' + item.angle + ')'; break; // Doesn't handle new origin
								case 5: value += ' skewX(' + item.angle + ')'; break;
								case 6: value += ' skewY(' + item.angle + ')'; break;
							}
						}
						val = value.substring(1);
					}
					else {
						val = val.getItem(0).valueAsString;
					}
				}
				return (val && val.baseVal ? val.baseVal.valueAsString : val);
			}

			var options = name;
			if (typeof name === 'string') {
				options = {};
				options[name] = value;
			}
			return this.each(function () {
				if ($.svg.isSVGElem(this)) {
					for (var n in options) {
						var val = ($.isFunction(options[n]) ? options[n]() : options[n]);
						(type ? this.style[n] = val : this.setAttribute(n, val));
					}
				}
				else {
					origAttr.apply($(this), [name, value, type]);
				}
			});
		};
	} ($.fn.attr);

	/* Support removing attributes on SVG nodes. */
	$.fn.removeAttr = function (origRemoveAttr) {
		return function (name) {
			return this.each(function () {
				if ($.svg.isSVGElem(this)) {
					(this[name] && this[name].baseVal ? this[name].baseVal.value = '' :
					this.setAttribute(name, ''));
				}
				else {
					origRemoveAttr.apply($(this), [name]);
				}
			});
		};
	} ($.fn.removeAttr);

	/* Add numeric only properties. */
	$.extend($.cssNumber, {
		'stopOpacity': true,
		'strokeMitrelimit': true,
		'strokeOpacity': true
	});

	/* Support retrieving CSS/attribute values on SVG nodes. */
	if ($.cssProps) {
		$.css = function (origCSS) {
			return function (elem, name, extra) {
				var value = (name.match(/^svg.*/) ? $(elem).attr($.cssProps[name] || name) : '');
				return value || origCSS(elem, name, extra);
			};
		} ($.css);
	}

	/* Determine if any nodes are SVG nodes. */
	function anySVG(checkSet) {
		for (var i = 0; i < checkSet.length; i++) {
			if (checkSet[i].nodeType == 1 && checkSet[i].namespaceURI == $.svg.svgNS) {
				return true;
			}
		}
		return false;
	}

	/* Update Sizzle selectors. */

	$.expr.relative['+'] = function (origRelativeNext) {
		return function (checkSet, part, isXML) {
			origRelativeNext(checkSet, part, isXML || anySVG(checkSet));
		};
	} ($.expr.relative['+']);

	$.expr.relative['>'] = function (origRelativeChild) {
		return function (checkSet, part, isXML) {
			origRelativeChild(checkSet, part, isXML || anySVG(checkSet));
		};
	} ($.expr.relative['>']);

	$.expr.relative[''] = function (origRelativeDescendant) {
		return function (checkSet, part, isXML) {
			origRelativeDescendant(checkSet, part, isXML || anySVG(checkSet));
		};
	} ($.expr.relative['']);

	$.expr.relative['~'] = function (origRelativeSiblings) {
		return function (checkSet, part, isXML) {
			origRelativeSiblings(checkSet, part, isXML || anySVG(checkSet));
		};
	} ($.expr.relative['~']);
}

ext(jQuery);


/* http://keith-wood.name/svg.html
   SVG filters for jQuery v1.4.5.
   Written by Keith Wood (kbwood{at}iinet.com.au) August 2007.
   Dual licensed under the GPL (http://dev.jquery.com/browser/trunk/jquery/GPL-LICENSE.txt) and
   MIT (http://dev.jquery.com/browser/trunk/jquery/MIT-LICENSE.txt) licenses.
   Please attribute the author if you use it. */

(function ($) { // Hide scope, no $ conflict

    $.svg.addExtension('filters', SVGFilter);

    $.extend($.svg._wrapperClass.prototype, {

        /* Add a filter definition.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  id        (string) the ID for this filter
           @param  x         (number) the x-coordinate for the left edge of the filter
           @param  y         (number) the y-coordinate for the top edge of the filter
           @param  width     (number) the width of the filter
           @param  height    (number) the height of the filter
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        filter: function (parent, id, x, y, width, height, settings) {
            var args = this._args(arguments, ['id', 'x', 'y', 'width', 'height']);
            return this._makeNode(args.parent, 'filter', $.extend(
                { id: args.id, x: args.x, y: args.y, width: args.width, height: args.height },
                args.settings || {}));
        }
    });

    /* Extension point for SVG filters.
       Access through svg.filters. */
    function SVGFilter(wrapper) {
        this._wrapper = wrapper; // The attached SVG wrapper object
    }

    $.extend(SVGFilter.prototype, {

        /* Add a distant light filter.
           @param  parent     (element or jQuery) the parent node for the new filter (optional)
           @param  result     (string) the ID of this filter
           @param  azimuth    (number) the angle (degrees) in the XY plane for the light source
           @param  elevation  (number) the angle (degrees) in the YZ plane for the light source
           @param  settings   (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        distantLight: function (parent, result, azimuth, elevation, settings) {
            var args = this._wrapper._args(arguments, ['result', 'azimuth', 'elevation']);
            return this._wrapper._makeNode(args.parent, 'feDistantLight', $.extend(
                { result: args.result, azimuth: args.azimuth, elevation: args.elevation },
                args.settings || {}));
        },

        /* Add a point light filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  x         (number) the x-coordinate for the light source
           @param  y         (number) the y-coordinate for the light source
           @param  z         (number) the z-coordinate for the light source
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        pointLight: function (parent, result, x, y, z, settings) {
            var args = this._wrapper._args(arguments, ['result', 'x', 'y', 'z']);
            return this._wrapper._makeNode(args.parent, 'fePointLight', $.extend(
                { result: args.result, x: args.x, y: args.y, z: args.z }, args.settings || {}));
        },

        /* Add a spot light filter.
           Specify all of toX, toY, toZ or none of them.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  x         (number) the x-coordinate for the light source
           @param  y         (number) the y-coordinate for the light source
           @param  z         (number) the z-coordinate for the light source
           @param  toX       (number) the x-coordinate for where the light is pointing (optional)
           @param  toY       (number) the y-coordinate for where the light is pointing (optional)
           @param  toZ       (number) the z-coordinate for where the light is pointing (optional)
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        spotLight: function (parent, result, x, y, z, toX, toY, toZ, settings) {
            var args = this._wrapper._args(arguments,
                ['result', 'x', 'y', 'z', 'toX', 'toY', 'toZ'], ['toX']);
            var sets = $.extend({ result: args.result, x: args.x, y: args.y, z: args.z },
                (args.toX != null ? {
                    pointsAtX: args.toX, pointsAtY: args.toY,
                    pointsAtZ: args.toZ
                } : {}));
            return this._wrapper._makeNode(args.parent, 'feSpotLight',
                $.extend(sets, args.settings || {}));
        },

        /* Add a blend filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  mode      (string) normal | multiply | screen | darken | lighten
           @param  in1       (string) the first image to blend
           @param  in2       (string) the second image to blend
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        blend: function (parent, result, mode, in1, in2, settings) {
            var args = this._wrapper._args(arguments, ['result', 'mode', 'in1', 'in2']);
            return this._wrapper._makeNode(args.parent, 'feBlend', $.extend(
                { result: args.result, mode: args.mode, in_: args.in1, in2: args.in2 },
                args.settings || {}));
        },

        /* Add a colour matrix filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  in1       (string) the source to colour
           @param  type      (string) matrix | saturate | hueRotate | luminanceToAlpha
           @param  values    (number[][]) for 'matrix' the matrix (5x4) values to apply
                             (number) for 'saturate' 0.0 to 1.0
                             (number) for 'hueRotate' degrees
                             (void) for 'luminanceToAlpha'
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        colorMatrix: function (parent, result, in1, type, values, settings) {
            var args = this._wrapper._args(arguments, ['result', 'in1', 'type', 'values']);
            if (isArray(args.values)) {
                var vs = '';
                for (var i = 0; i < args.values.length; i++) {
                    vs += (i == 0 ? '' : ' ') + args.values[i].join(' ');
                }
                args.values = vs;
            }
            else if (typeof args.values == 'object') {
                args.settings = args.values;
                args.values = null;
            }
            var sets = $.extend({ result: args.result, in_: args.in1, type: args.type },
                (args.values != null ? { values: args.values } : {}));
            return this._wrapper._makeNode(args.parent, 'feColorMatrix',
                $.extend(sets, args.settings || {}));
        },

        /* Add a component transfer filter.
           @param  parent     (element or jQuery) the parent node for the new filter (optional)
           @param  result     (string) the ID of this filter
           @param  functions  (object[]) one for each of RGB and A (alpha, optional)
                              for each entry:
                              [0] is (string) identity | table | discrete | linear | gamma
                              [1] is (number[]) for 'table' or 'discrete' the list of
                              interpolation or step values OR
                              (number) for 'linear' the slope, for 'gamma' the amplitude,
                              [2] is (number) for 'linear' the intercept, for 'gamma' the exponent,
                              [3] is (number) for 'gamma' the offset
           @param  settings   (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        componentTransfer: function (parent, result, functions, settings) {
            var args = this._wrapper._args(arguments, ['result', 'functions']);
            var node = this._wrapper._makeNode(args.parent, 'feComponentTransfer',
                $.extend({ result: args.result }, args.settings || {}));
            var rgba = ['R', 'G', 'B', 'A'];
            for (var i = 0; i < Math.min(4, args.functions.length) ; i++) {
                var props = args.functions[i];
                var sets = $.extend({ type: props[0] },
                    (props[0] == 'table' || props[0] == 'discrete' ? { tableValues: props[1].join(' ') } :
                    (props[0] == 'linear' ? { slope: props[1], intercept: props[2] } :
                    (props[0] == 'gamma' ? {
                        amplitude: props[1],
                        exponent: props[2], offset: props[3]
                    } : {}))));
                this._wrapper._makeNode(node, 'feFunc' + rgba[i], sets);
            }
            return node;
        },

        /* Add a composite filter.
           Specify all of k1, k2, k3, k4 or none of them.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  operator  (string) over | in | out | atop | xor | arithmetic
           @param  in1       (string) the first filter to compose
           @param  in2       (string) the second filter to compose
           @param  k1        (number) for 'arithmetic' (optional)
           @param  k2        (number) for 'arithmetic' (optional)
           @param  k3        (number) for 'arithmetic' (optional)
           @param  k4        (number) for 'arithmetic' (optional)
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        composite: function (parent, result, operator, in1, in2, k1, k2, k3, k4, settings) {
            var args = this._wrapper._args(arguments, ['result', 'operator',
                'in1', 'in2', 'k1', 'k2', 'k3', 'k4'], ['k1']);
            var sets = $.extend({
                result: args.result, operator: args.operator,
                'in': args.in1, in2: args.in2
            },
                (args.k1 != null ? { k1: args.k1, k2: args.k2, k3: args.k3, k4: args.k4 } : {}));
            return this._wrapper._makeNode(args.parent, 'feComposite',
                $.extend(sets, args.settings || {}));
        },

        /* Add a convolve matrix filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  order     (int or 'int int') the size(s) of the matrix
           @param  matrix    (number[][]) the kernel matrix for the convolution
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        convolveMatrix: function (parent, result, order, matrix, settings) {
            var args = this._wrapper._args(arguments, ['result', 'order', 'matrix']);
            var mx = '';
            for (var i = 0; i < args.matrix.length; i++) {
                mx += (i == 0 ? '' : ' ') + args.matrix[i].join(' ');
            }
            args.matrix = mx;
            return this._wrapper._makeNode(args.parent, 'feConvolveMatrix', $.extend(
                { result: args.result, order: args.order, kernelMatrix: args.matrix },
                args.settings || {}));
        },

        /* Add a diffuse lighting filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  colour    (string) the lighting colour (optional)
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        diffuseLighting: function (parent, result, colour, settings) {
            var args = this._wrapper._args(arguments, ['result', 'colour'], ['colour']);
            return this._wrapper._makeNode(args.parent, 'feDiffuseLighting',
                $.extend($.extend({ result: args.result },
                (args.colour ? { lightingColor: args.colour } : {})), args.settings || {}));
        },

        /* Add a displacement map filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  in1       (string) the source image
           @param  in2       (string) the displacement image
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        displacementMap: function (parent, result, in1, in2, settings) {
            var args = this._wrapper._args(arguments, ['result', 'in1', 'in2']);
            return this._wrapper._makeNode(args.parent, 'feDisplacementMap',
                $.extend({ result: args.result, in_: args.in1, in2: args.in2 },
                args.settings || {}));
        },

        /* Add a flood filter.
           Specify all of x, y, width, height or none of them.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  x         (number) the left coordinate of the rectangle (optional)
           @param  y         (number) the top coordinate of the rectangle (optional)
           @param  width     (number) the width of the rectangle (optional)
           @param  height    (number) the height of the rectangle (optional)
           @param  colour    (string) the colour to fill with
           @param  opacity   (number) the opacity 0.0-1.0
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        flood: function (parent, result, x, y, width, height, colour, opacity, settings) {
            var args = this._wrapper._args(arguments,
                ['result', 'x', 'y', 'width', 'height', 'colour', 'opacity']);
            if (arguments.length < 6) {
                args.colour = args.x;
                args.opacity = args.y;
                args.settings = args.width;
                args.x = null;
            }
            var sets = $.extend({
                result: args.result, floodColor: args.colour,
                floodOpacity: args.opacity
            }, (args.x != null ?
			{ x: args.x, y: args.y, width: args.width, height: args.height } : {}));
            return this._wrapper._makeNode(args.parent, 'feFlood',
                $.extend(sets, args.settings || {}));
        },

        /* Add a Gaussian blur filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  in1       (string) the source filter
           @param  stdDevX   (number) the standard deviation along the x-axis
           @param  stdDevY   (number) the standard deviation along the y-axis (optional)
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        gaussianBlur: function (parent, result, in1, stdDevX, stdDevY, settings) {
            var args = this._wrapper._args(arguments,
                ['result', 'in1', 'stdDevX', 'stdDevY'], ['stdDevY']);
            return this._wrapper._makeNode(args.parent, 'feGaussianBlur', $.extend(
                {
                    result: args.result, in_: args.in1, stdDeviation: args.stdDevX +
                    (args.stdDevY ? ' ' + args.stdDevY : '')
                }, args.settings || {}));
        },

        /* Add an image filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  href      (string) the URL of the image
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        image: function (parent, result, href, settings) {
            var args = this._wrapper._args(arguments, ['result', 'href']);
            var node = this._wrapper._makeNode(args.parent, 'feImage', $.extend(
                { result: args.result }, args.settings || {}));
            node.setAttributeNS($.svg.xlinkNS, 'href', args.href);
            return node;
        },

        /* Add a merge filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  refs      (string[]) the IDs of the filters to merge
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        merge: function (parent, result, refs, settings) {
            var args = this._wrapper._args(arguments, ['result', 'refs']);
            var node = this._wrapper._makeNode(args.parent, 'feMerge', $.extend(
                { result: args.result }, args.settings || {}));
            for (var i = 0; i < args.refs.length; i++) {
                this._wrapper._makeNode(node, 'feMergeNode', { in_: args.refs[i] });
            }
            return node;
        },

        /* Add a morphology filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  in1       (string) the source filter
           @param  operator  (string) erode | dilate
           @param  radiusX   (number) the size of the operation in the x-axis
           @param  radiusY   (number) the size of the operation in the y-axis (optional)
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        morphology: function (parent, result, in1, operator, radiusX, radiusY, settings) {
            var args = this._wrapper._args(arguments, ['result', 'in1',
                'operator', 'radiusX', 'radiusY'], ['radiusY']);
            return this._wrapper._makeNode(args.parent, 'feMorphology', $.extend(
                {
                    result: args.result, in_: args.in1, operator: args.operator,
                    radius: args.radiusX + (args.radiusY ? ' ' + args.radiusY : '')
                },
                args.settings || {}));
        },

        /* Add an offset filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  in1       (string) the source filter
           @param  dX        (number) the offset in the x-axis
           @param  dY        (number) the offset in the y-axis
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        offset: function (parent, result, in1, dx, dy, settings) {
            var args = this._wrapper._args(arguments, ['result', 'in1', 'dx', 'dy']);
            return this._wrapper._makeNode(args.parent, 'feOffset', $.extend(
                { result: args.result, in_: args.in1, dx: args.dx, dy: args.dy },
                args.settings || {}));
        },

        /* Add a specular lighting filter.
           Numeric params are only optional if following numeric params are also omitted.
           @param  parent            (element or jQuery) the parent node for the new filter (optional)
           @param  result            (string) the ID of this filter
           @param  in1               (string) the source filter
           @param  surfaceScale      (number) the surface height when Ain = 1 (optional)
           @param  specularConstant  (number) the ks in Phong lighting model (optional)
           @param  specularExponent  (number) the shininess 1.0-128.0 (optional)
           @param  settings          (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        specularLighting: function (parent, result, in1, surfaceScale,
                specularConstant, specularExponent, settings) {
            var args = this._wrapper._args(arguments, ['result', 'in1',
                'surfaceScale', 'specularConstant', 'specularExponent'],
                ['surfaceScale', 'specularConstant', 'specularExponent']);
            return this._wrapper._makeNode(args.parent, 'feSpecularLighting', $.extend(
                {
                    result: args.result, in_: args.in1, surfaceScale: args.surfaceScale,
                    specularConstant: args.specularConstant, specularExponent: args.specularExponent
                },
                args.settings || {}));
        },

        /* Add a tile filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  in1       (string) the source filter
           @param  x         (number) the left coordinate of the rectangle
           @param  y         (number) the top coordinate of the rectangle
           @param  width     (number) the width of the rectangle
           @param  height    (number) the height of the rectangle
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        tile: function (parent, result, in1, x, y, width, height, settings) {
            var args = this._wrapper._args(arguments,
                ['result', 'in1', 'x', 'y', 'width', 'height']);
            return this._wrapper._makeNode(args.parent, 'feTile', $.extend(
                {
                    result: args.result, in_: args.in1, x: args.x, y: args.y,
                    width: args.width, height: args.height
                }, args.settings || {}));
        },

        /* Add a turbulence filter.
           @param  parent    (element or jQuery) the parent node for the new filter (optional)
           @param  result    (string) the ID of this filter
           @param  type      (string) fractalNoise | turbulence
           @param  baseFreq  (number or 'number number') the base frequency,
                             optionally separated into x- and y-components
           @param  octaves   (number) the amount of turbulence (optional)
           @param  settings  (object) additional settings for the filter (optional)
           @return  (element) the new filter node */
        turbulence: function (parent, result, type, baseFreq, octaves, settings) {
            var args = this._wrapper._args(arguments, ['result', 'type',
                'baseFreq', 'octaves'], ['octaves']);
            return this._wrapper._makeNode(args.parent, 'feTurbulence', $.extend(
                {
                    result: args.result, type: args.type, baseFrequency: args.baseFreq,
                    numOctaves: args.octaves
                }, args.settings || {}));
        }
    });

    /* Determine whether an object is an array. */
    function isArray(a) {
        return (a && a.constructor == Array);
    }

})(jQuery)
