﻿// request manager module
define(['server/adapters/EventsAdapter',
    'when', 'common/Utilites',
    'common/Enums', 'base/EventTarget',
    'server/requestManager/RequestManagerState',
    'core/Timer'], function (EventsAdapter, when,
        Utilites, Enums, EventTarget, RequestManagerState, Timer) {

        var RequestManager = Class.extend({
            MESSAGES_CHANGED: 'MessagesChanged',
            ARCHIVE_MESSAGES_RECEIVED: 'ArchiveMessagesReceived',
            TIME_UPDATED: 'TimeUpdated',
            stateVars: Enums.serverState,
            clientHandleKey: 'clientHandle',
            keyFieldName: 'EventId',
            activeTimeField: 'ActiveTime',

            recType: {
                refresh: 'Refresh',
                update: 'Update',
                archive: 'Archive'
            },

            init: function (options) {
                this.eventTarget = new EventTarget();
                this.serverStateManager = new RequestManagerState(options);
                this.adapter = new EventsAdapter(this.serverStateManager);

                this.clientHandle = 0;
                this.monitoredItemId = null;
                this.statusCode = 0;
                this.initialized = false;
                //this.messages = [];
                this.fieldsNames = [];

                this.columns = [];
                this.filters = [];

                this.ackSequenceNumber = 0;
                this.getAllWrap = this.getAll.bind(this);
                this.sGet = this.serverStateManager.get.bind(this.serverStateManager);
                this.initialize();
            },

            initialize: function () {
                this.timer = new Timer(this.sGet(this.stateVars.requestedPublishingInterval));
                this.startTimer();
            },
            setObject: function(objectUid) {
                if (isNaN(objectUid)) {
                    this.serverStateManager.state.itemId = 0;
                    this.serverStateManager.state.path = objectUid;
                } else {
                    this.serverStateManager.state.itemId = objectUid;
                    this.serverStateManager.state.path = "";
                }
            },
            startTimer: function () {
                this.timer.start();
            },

            setColumns: function (columns) {
                this.columns = $.extend(true, [], columns);
                this.fieldsNames = _.map(this.columns, function (columns) {
                    return columns.name;
                });
            },

            setFilters: function (filters, slient) {
                var def = when.defer();
                var ctx = this;
                this.filters = $.extend(true, [], filters);
                if (slient === false) {
                    this.deleteMonitoredEvent().then(function () {
                        ctx.subscribe().then(function () {
                            def.resolve();
                        });
                    }, function () {
                        def.reject();
                    });
                } else {
                    def.resolve();
                }

                return def.promise;
            },

            deleteMonitoredEvent: function () {
                if (this.monitoredItemId == null) {
                    console.warn("deleteMonitoredEvent: попытка отписаться от данных, на которые еще не подписались");
                    return when.resolve();
                }
                //зарание увеличиваем, 
                //что бы отсеять ответы по старым запросам
                this.clientHandle++;
                //this.messages = [];
                //предотвращаем запросы getAll
                this.initialized = false;
                this.timer.remove(this.getAllWrap);
                return this.adapter.deleteMonitoredEvent(this.monitoredItemId).then(function (result) {
                    this.monitoredItemId = null; //подписка удалилась, id больше использовать не надо
                }.bind(this));
            },


            subscribe: function () {
                var def = when.defer();
                var ctx = this;
                //if (this.columns.length > 0) {
                this.adapter.initialize().then(function () {
                    ctx._addMonitoredEvents().then(function (result) {
                        var item = result.data.items[0];
                        ctx.monitoredItemId = item.monitoredItemId;
                        ctx.statusCode = item.statusCode;
                        ctx.adapter.refresh().then(function (result) {
                            if (ctx.initialized === false) {
                                ctx.initialized = true;
                                ctx.timer.setInterval(ctx.sGet(ctx.stateVars.requestedPublishingInterval));
                                ctx.timer.prepend(ctx.getAllWrap);
                            }

                            def.resolve();
                        });
                    });
                });
                //  } else {
                //      def.resolve();
                //   }

                return def.promise;
            },

            refresh: function () {
                this.adapter.refresh();
            },

            _addMonitoredEvents: function () {
                return this.adapter.addMonitoredEvents([{
                    clientHandle: ++this.clientHandle,
                    fields: this.columns,
                    filter: this.filters
                }]);
            },

            getAll: function () {
                var that = this;
                if (this.initialized && this.serverStateManager.get(this.stateVars.Connected)) {
                    return this.adapter.getAll().then(function (recs) {
                        return that._getAllSucceed(recs);
                    }).catch(function (result) {
                        return that._getAllFailed();
                    });
                } else {
                    return when.resolve();
                }
            },

            _getAllSucceed: function (responce) {
                var result = {
                    deletedMessages: [],
                    newOrUpdatedMessages: []
                };

                if (responce.deleted && responce.deleted.length > 0) {
                    var deletedGroups = _.groupBy(responce.deleted, this.clientHandleKey);
                    var deleted = [];

                    _.forEach(deletedGroups[this.clientHandle], function (arg) {
                        deleted = deleted.concat(arg.eventIds);
                    });

                    result.deletedMessages = deleted;                   
                }

                if (responce.recs.length > 0) {
                    //берем только сообщения, которые пришли для активной выборки (this.clientHandle)            
                    var newGroups = _.groupBy(responce.recs, this.clientHandleKey);
                    var messages = _.map(newGroups[this.clientHandle], function (message) {
                        return {
                            fields: message.fields,
                            recType: message.recType
                        };
                    });

                    result.newOrUpdatedMessages = this._processMessages(messages);
                }
                this.fireTimeUpdated(+responce.serverTime);
                if (result.deletedMessages.length > 0 || result.newOrUpdatedMessages.length > 0) {
                    this.fireMessagesChanged(result.deletedMessages, result.newOrUpdatedMessages);
                }

            },

            _processMessage: function (fields) {
                var msg = {};

                for (var i = 0; i < this.fieldsNames.length; i++) {
                    msg[this.fieldsNames[i]] = fields[i];
                    if (Utilites.STTypeToJsType(this.columns[i].type) === Enums.parameterType.date &&
                         fields[i] !== null) {
                        // дата приходит в виде UTC Timestamp
                        // null даты, 0 и "0" не конвертируем 
                        msg[this.fieldsNames[i]] = (fields[i] === null || fields[i] == 0) ? null : new Date(parseInt(fields[i]));
                    } else {
                        msg[this.fieldsNames[i]] = fields[i];
                    }
                }

                return msg;
            },

            _getAllFailed: function () {
                when.reject(new Error("getAll failed"));
            },

            getFieldType: function (fieldName) {
                var field = _.find(this.columns, { name: fieldName });
                if (field === undefined) {
                    return undefined;
                }

                return Utilites.STTypeToJsType(field.type);
            },            

            acknowledgeEvents: function (eventIds, comment) {
                var recs = _.map(eventIds, function(eventId){
                    return{
                        monitoredItemId: this.monitoredItemId,
                        eventId: eventId,
                        comment: comment || ''
                    }
                }, this);

                return this.adapter.acknowledgeEvents(recs);
            },

            getArchiveEvents: function (startTime, endTime, count, inverse) {
                var params = {
                    startTime: startTime,
                    endTime: endTime,
                    fields: this.columns,
                    filter: this.filters,
                    maxRecs: count,
                    inverse: inverse
                };

                var messages = [];

                return this._getArchiveEvents(params, messages);
            },

            _getArchiveEvents: function (params, messages) {
                var ctx = this;
                //console.debug('Read from archive: ' + params.maxRecs);

                return this.adapter.getArchiveData(params).then(function (response) {
                    if (response.recs && response.recs.length > 0) {
                        var newOrUpdatedMessages = ctx._processArchiveMessages(response.recs);
                        if (newOrUpdatedMessages.length > 0) {
                            ctx.fireArchiveMessagesReceived(newOrUpdatedMessages);
                            messages.push.apply(messages, newOrUpdatedMessages);
                        }
                    }

                    if (response.hasMore === true &&
                        response.continuationPoint !== undefined
                        && newOrUpdatedMessages && newOrUpdatedMessages.length > 0) {

                        //console.debug('Read more from ' + response.continuationPoint);
                        //console.debug('Read count: ' + response.continuationPoint);
                        //т.к читаем архив, то все сообщения - новые
                        if (params.maxRecs != 0) {
                            var newMaxRecs = params.maxRecs - newOrUpdatedMessages.length;
                            if (newMaxRecs === 0) {
                                //вычитали весь запрошенный объем
                                return messages;
                            }
                        }
                        params.continuationPoint = response.continuationPoint;
                        return ctx._getArchiveEvents(params, messages);
                    }

                    return messages;
                });
            },

            getArchiveLeftBorder: function () {
                var params = {
                    startTime: 0,
                    endTime: 0,
                    fields: this.columns,
                    filter: [],
                    maxRecs: 1
                };
                return this.adapter.getArchiveData(params).then(function (response) {
                    if (response.recs.length > 0) {
                        return this._processMessage(response.recs[0]);
                    }

                    return undefined;
                }.bind(this));
            },

            _processMessages: function (messages) {
                var newOrUpdatedMessages = [];

                var i;
                for (i = 0; i < messages.length; i++) {
                    newOrUpdatedMessages.push(this._processMessage(messages[i].fields));
                }

                return newOrUpdatedMessages;
            },
            //not messages  - just recs
            _processArchiveMessages: function (recs) {
                var newOrUpdatedMessages = [];

                var i;
                for (i = 0; i < recs.length; i++) {
                    newOrUpdatedMessages.push(this._processMessage(recs[i]));
                }

                return newOrUpdatedMessages;
            },

            /*getMessages: function () {
                return this.messages;
            },
            */
            unsubscribe: function () {
                if (this.initialized === true) {
                    this.initialized = false;
                    this.timer.remove(this.getAllWrap);
                    this.timer.stop();
                }
                return this.adapter.destroy();
            },

            subscribeMessagesChanged: function (handler, context) {
                this.eventTarget.addListener(this.MESSAGES_CHANGED, handler, context);
            },

            subscribeTimeUpdated: function (handler, context) {
                this.eventTarget.addListener(this.TIME_UPDATED, handler, context);
            },

            fireTimeUpdated: function (time) {
                this.eventTarget.fire({
                    type: this.TIME_UPDATED,
                    target: this,
                    serverTime: time
                });
            },

            fireMessagesChanged: function (deletedMessages, newOrUpdated) {
                this.eventTarget.fire({
                    type: this.MESSAGES_CHANGED,
                    target: this,
                    deleted: deletedMessages,
                    newOrUpdated: newOrUpdated
                });
            },

            unsubscribeMessagesChanged: function (handler) {
                this.eventTarget.removeListener(this.MESSAGES_CHANGED, handler);
            },

            subscribeArchiveMessagesReceived: function (handler, context) {
                this.eventTarget.addListener(this.ARCHIVE_MESSAGES_RECEIVED, handler, context);
            },

            fireArchiveMessagesReceived: function (newOrUpdated) {
                this.eventTarget.fire({
                    type: this.ARCHIVE_MESSAGES_RECEIVED,
                    target: this,
                    newOrUpdated: newOrUpdated
                });
            },

            unsubscribeArchiveMessagesReceived: function (handler) {
                this.eventTarget.removeListener(this.ARCHIVE_MESSAGES_RECEIVED, handler);
            },
        });

        return RequestManager;
    });
