﻿
define(['common/Enums', 'common/Error', 'common/GeometryNormalizer/Processor/PathSyntaxCommandFactory', 'core/L10n'],
function (Enums, Error, PathSyntaxCommandFactory, L10n) {

    var CustomGeometryProcessor = Class.extend({
        whiteSpaceRegx: /\s/,
        init: function () {
            this.pathCommands = ["F", "M", "L", "H", "V", "C", "Q", "S", "T", "A", "Z", 
                                            "f", "m", "l", "h", "v", "c", "q", "s", "t", "a", "z"];
        },

        process: function (geometryPath, width, height, parentWidth, parentHeight) {
            geometryPath = geometryPath.replace('\n', ' ').replace('\r', ' ');
            var parsedString = [], //{command: arguments}  
                commands = [];

            parsedString = this.parseInputString(geometryPath);
            if (parsedString.length === 0) {
                Error.onerror('Error parsing geometry: ' + geometryPath);
                return '';
            }

            _.forEach(parsedString, function(parsedItem) {
                var key = Object.keys(parsedItem)[0];
                var command = PathSyntaxCommandFactory.create(key); //todo item.item1
                command.parse(parsedItem[key]);
                commands.push(command);
            });

            var elements = [];
            _.forEach(commands, function(command) {
                var filtered = _.filter(command.getElements(), function(command) { 
                    return command.type == Enums.pathSyntaxCommandElementType.X || 
                    command.type == Enums.pathSyntaxCommandElementType.Width
                }).map(function(command) {
                    return Math.abs(parseFloat(command.getValue()))
                });

                elements = elements.concat(filtered);
                
            });

            if (elements.length === 0) {
                Error.onerror('Error parsing geometry: ' + geometryPath);
                return '';
            }

            var maxWidthElement = _.max(elements),
                elements = [];

            _.forEach(commands, function(command) {
                var filtered = _.filter(command.getElements(), function(command) { 
                    return command.type == Enums.pathSyntaxCommandElementType.Y || 
                    command.type == Enums.pathSyntaxCommandElementType.Height
                }).map(function (command) {
                    return Math.abs(parseFloat(command.getValue()))
                });

                elements = elements.concat(filtered);
            });      
            
            var maxHeightElement = _.max(elements);

            if (maxHeightElement === 0 || maxWidthElement === 0) {
                // нечего нормальизовать-команда состоит из одних нулей
                Error.debug(String.format('{0}, {1}', L10n.get('messages.invalidGeometry'), geometryPath));
                return geometryPath;
            }

            return commands.map(function (c) {
                c.normalize(parentWidth || width, parentHeight || height, maxWidthElement, maxHeightElement);
                return c.getStringRepresentation();
            }).join(' ');
        },

        parseInputString: function(geometryPath) {
            //will contain a pair command-arguments
            var parsedString = [],
                leftBound = 0,
                rightBound = leftBound + 1;
            //Fill dictionary with valid pairs
            while (leftBound < geometryPath.length) {
                if (this.pathCommands.indexOf(geometryPath[leftBound])!==-1) {
                    var commandArgs = '';
                    while ((rightBound < geometryPath.length) &&
                        (!isNaN(geometryPath[rightBound]) ||
                        geometryPath[rightBound] == ',' ||
                        geometryPath[rightBound] == '-' ||
                         geometryPath[rightBound] == '.' ||
                        this._isWhiteSpace(geometryPath[rightBound])
                        )) {
                        commandArgs += geometryPath[rightBound];
                        rightBound++;
                    }

                    var pathCommand = {};
                    pathCommand[geometryPath[leftBound]] = commandArgs;
                    parsedString.push(pathCommand);
                }

                leftBound = rightBound;
                rightBound = leftBound + 1;
            }

            return parsedString;
        },

        _isWhiteSpace: function(ch) {
            this.whiteSpaceRegx.test(ch);
        }
    });

    return new CustomGeometryProcessor();
});
