﻿// <copyright file="NumericUpDownController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class NumericUpDownController controller class for NumericUpDown control
*/

define(['common/Enums', 'common/Appearence', 'base/TextEditController', 'common/Utilites'],
function (Enums, Appearence, TextEditController, Utilites) {
    //here we will define controller for NumericUpDown

    var NumericUpDownController = TextEditController.extend({

        //numbers after decimal separator
        precision: null,
        filterOptions: null,

        init: function () {
            this._super();

            this.ClassName = 'NumericUpDownController';
            this.filterOptions = {
                live: false,
                negkey: '-',
                singlechar: '.'
            };
        },
        onModelInit: function () {
            this._super();
            this.precision = Utilites.precision(this.mvc.model.getIncrementStep());
        },

        _changeControlByMultiline: function () {
        },
        validateProperty: function (propertyName, newProperty) {
            switch (propertyName) {
                case Enums.ParameterRoles.VALUE:
                    this.validator.validateMinMax(
                        {
                            propertyName: propertyName,
                            maxValue: this.mvc.model.get(Enums.ParameterRoles.MAX_VALUE),
                            minValue: this.mvc.model.get(Enums.ParameterRoles.MIN_VALUE),
                            property: newProperty
                        });
                    break;
            }
        },

        textElement: function () {
            return this.mvc.view.$('input');
        },

        displayText: function () {
            //do nothing because we display value in onValueChanged
            this.textElement().attr('value', this.mvc.model.getValue());
        },

        getCurrentValue: function () {
            var val = parseFloat(this.textElement().val());
            if (val == null || isNaN(val) || !isFinite(val)) {
                val = this.mvc.model.getMinValue();
            }
            return val;
        },

        checkLower: function (val) {
            var value = parseFloat(val);
            if (value < this.mvc.model.getMinValue()) {
                return false;
            }

            return true;
        },

        checkUpper: function (val) {
            var value = parseFloat(val);
            if (value > this.mvc.model.getMaxValue()) {
                return false;
            }

            return true;
        },

        incrementValue: function () {
            var val = this.getCurrentValue();
            val += this.mvc.model.getIncrementStep();
            val = this.normalize(val);
            this.mvc.model.setValue(val);
        },

        decrementValue: function () {
            var val = this.getCurrentValue();
            val -= this.mvc.model.getIncrementStep();
            val = this.normalize(val);
            this.mvc.model.setValue(val);
        },

        normalize: function(val) {
            if (!this.checkUpper(val)) {
                val = this.mvc.model.getMaxValue();
            }

            if (!this.checkLower(val)) {
                val = this.mvc.model.getMinValue();
            }

            return val;
        },

        checkInputChars: function () {
            var inputValue = this.textElement().val();
            var oldValue = inputValue;

            if (inputValue === '' || inputValue === "") {
                inputValue = this.mvc.model.getMinValue();
            }

            if (!isFinite(inputValue)) {
                inputValue = parseFloat(inputValue);
                if (isNaN(inputValue)) {
                    inputValue = this.mvc.model.getMinValue();
                }
            }

            inputValue = this.normalize(inputValue);

            if (oldValue !== inputValue) {
                this.textElement().val(inputValue);
            }
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            var value = this.mvc.model.get(event.property);

            switch (event.property) {
                case Enums.ParameterRoles.VALUE:
                    this.onValueChanged(value);
                    break;
                case Enums.ParameterRoles.MIN_VALUE:
                    this.onMinValueChanged(value);
                    break;
                case Enums.ParameterRoles.MAX_VALUE:
                    this.onMaxValueChanged(value);
                    break;
                case Enums.ParameterRoles.INCREMENT_STEP:
                    this.onIncrementStepChanged(value);
                    break;
            }
        },

        onTextChanged: function (value) {
            //do nothing because we use onValueChanged
        },

        onValueChanged: function (value) {

            if (this.isSettingText()) {
                return;
            }
            value = Utilites.decimalAdjust(value, -this.precision);
            value = this.normalize(value);
            this.textElement().val(value);
            this.mvc.model.setText(value);
        },

        onMinValueChanged: function (value) {
            var inputValue = this.getCurrentValue();
            if (!this.checkLower(inputValue)) {
                this.mvc.model.setValue(value);
            }
        },

        onMaxValueChanged: function (value) {
            var inputValue = this.getCurrentValue();
            if (!this.checkUpper(inputValue)) {
                this.mvc.model.setValue(value);
            }
        },

        onIncrementStepChanged: function (value) {

        },

        onIsEnabledChanged: function (value) {
            this._super(value);

            if (!value) {
                this.mvc.view.$('.up, .down').attr('disabled', 'true');
            } else {
                this.mvc.view.$('.up, .down').removeAttr('disabled');
            }
        },

        //
        // Events
        //      

        attachEvents: function (elem) {
            this._super(elem);
            var self = this;
            this.mvc.view.$('.up').mousedown(function (event) { self.onUpButtonMouseDown(event); });
            this.mvc.view.$('.down').mousedown(function (event) { self.onDownButtonMouseDown(event); });
            this.mvc.view.$('input').on('paste', function (event) { return self.onPaste(event); });
            this.mvc.view.$().mousewheel(function (e, delta) {
                self.onWheel(e, delta);
            }.bind(this));
        },

        onWheel: function (e, delta) {
            var value = this.mvc.model.getValue() + Math.floor(delta) * this.mvc.model.getIncrementStep();
            if ((value > this.mvc.model.getMaxValue()) || value < this.mvc.model.getMinValue()) {
                return;
            }
            else {
                this.mvc.model.setValue(value);
            }
        },
          

        allowAct: function() {
            this.setSettingText(true);
            this.checkInputChars(event);
            var val = this.getCurrentValue();
            this.mvc.model.setValue(val);
            //this.mvc.model.setText(val);
            this.setSettingText(false); 
            return Promise.resolve();       
        },

        denyAct: function() {        	
			var val = this.mvc.model.getValue();			
			this.textElement().val(val);			
            return Promise.resolve();
        },

        onUpButtonMouseDown: function (event) {
            if (!this.mvc.model.getIsEnabled()) {
                return;
            }

            this.incrementValue();
        },

        onDownButtonMouseDown: function (event) {
            if (!this.mvc.model.getIsEnabled()) {
                return;
            }

            this.decrementValue();
        },
        
        onKeyDown: function (event) {
            this._super(event);
            if (!this.mvc.model.getIsEnabled()) {
                return false;
            }
        },

        onKeyUp: function (event) {
            this._super(event);
       //    this.checkInputChars(event);
        },

        onKeyPress: function (event) {
            this._super(event);
            if (event.keyCode === 13 || event.keyCode === 10) {
                this.textElement().trigger("blur");
                this.checkInputChars(event);
                return;
            }
            return this.filter_input_function(this.filterOptions, event);
        },

        onPaste: function (event) {
            return this.filter_input_function(this.filterOptions, event);
         //   this.checkInputChars(event);
        },

        onTabIndexChanged: function(value) {
            if (value === 0) {
                value++;
            }
            this.textElement().attr('tabindex', value);
        },

        //
        // Utilites
        //
        /*
        * @method filter_input_function used to filter user input
        * http://www.thimbleopensource.com/tutorials-snippets/jquery-plugin-filter-text-input
        */
        filter_input_function: function (options, event) {

            var input = (event.input) ? event.input : this.textElement();
            if (event.ctrlKey || event.altKey) return;
            if (event.type == 'keypress') {

                var key = event.charCode ? event.charCode : event.keyCode ? event.keyCode : 0;

                // 8 = backspace, 9 = tab, 13 = enter, 35 = end, 36 = home, 37 = left, 39 = right, 46 = delete
                if (key == 8 || key == 9 || key == 13 || key == 35 || key == 36 || key == 37 || key == 39 || key == 46) {

                    // if charCode = key & keyCode = 0
                    // 35 = #, 36 = $, 37 = %, 39 = ', 46 = .

                    if (event.charCode == 0 && event.keyCode == key) {
                        return true;
                    }
                }
                var string = String.fromCharCode(key);
                // if they pressed the defined negative key
                if (string == options.singlechar) { //bug 13240, временно разрешено вводить точки
                    return true;
                }
                if (options.negkey && string == options.negkey) {
                    // if there is already one at the beginning, remove it
                    if (input.val().length == 0) {
                        return true;
                    }
                    if (input.val().substr(0, 1) == string) {
                        input.val(input.val().substring(1, input.val().length)).change();
                    } else {
                        // it isn't there so add it to the beginning of the string
                        input.val(string + input.val()).change();
                    }
                    return false;
                } else {
                    //verify for single char and skip if it already exists
                    if (options.singlechar.indexOf(string) > -1) {
                        if (input.val().indexOf(string) > -1) {
                            return false;
                        }
                    }
                }
            } else if (event.type == 'paste') {
                input.data('value_before_paste', event.target.value);
                setTimeout(function (that) {
                    return function () {
                        that.filter_input_function(options, { type: 'after_paste', input: input });
                    };
                }(this), 1);

                return true;
            } else if (event.type == 'after_paste') {
                var string = input.val();
            } else {
                return false;
            }

            if (!isNaN(parseFloat(string))) {
                return true;
            } else if (typeof (options.feedback) == 'function') {
                options.feedback.call(this, string);
            }
            if (event.type == 'after_paste') input.val(input.data('value_before_paste'));
            return false;
        },

        onRender: function () {

        }
    });

    return NumericUpDownController;
});
