﻿// <copyright file="TrendController.js" company="ИнСАТ">
// ИнСАТ, 2017
// </copyright>

/**
 * @class TrendController класс для контрола Тренд
 */
define(['common/ParameterRoles', 'common/Error', 'base/ControlController',
    'server/sources/TrendVariablesSource', 'common/Utilites', 'common/Appearence', 'core/global/GlobalSettingsType'
],
    function(Role, Error, ControlController, TrendVariablesSource, Utilites, Appearence, GlobalSettings) {
        return ControlController.extend({
            /**
             * @constructor 
             */
            init: function() {
                this._super();
                this.initialized = false;
                this.ClassName = 'TrendController';
                this.itemIdToGraphId = {};
            },
            /**
             * Установить конвертированное свойство для echart
             * @param {function} converter - указатель на функцию конвертер 
             * @param {value} value - значение, которое сначала конвертируется, а потом установится
             * @returns {} 
             */
            setChartOptionWithConverter: function(converter, value) {
                this.myChart.setOption(converter(value));
            },
            /**
            * Установить свойство для echart
            * @param {string} path - путь свойства(станет структурой объекта)
            * @param {any} value - значение
            */
            setChartOption: function(path, value) {
                var option = {};
                _.set(option, path, value);
                this.myChart.setOption(option);
            },
            /**
             * Установить свойство цвета
             * @param {string} path - путь свойства 
             * @param {string} value - значение цвета
             * @returns {} 
             */
            setColorChartOption: function(path, value) {
                this.setChartOption(path, Appearence.color.toCssColor(value));
            },
            /**
            * Установить свойство для пера
            * @param {string} path - путь свойства, начиная с series
            * @param {number} id - номер пера в массиве перьев
            * @param {any} value - значение
            */
            setSeriesOption: function(path, id, value) {
                _.set(this.series[id], path, value);
                this.myChart.setOption({
                    series: this.series
                });
            },

            /**
            * Установить свойство для оси Y
            * @param {string} path - путь свойства, начиная с yAxis
            * @param {number} id - номер пера в массиве перьев (номер оси)
            * @param {any} value - значение
            */
            setAxisOption: function (path, id, value) {
                _.set(this.current_y_axis[id], path, value);
                this.myChart.setOption({
                    yAxis: this.current_y_axis
                });
            },

            /**
            * Обработчик изменения параметра Цвет пера
            * @param {number} id - номер пера в массиве перьев
            * @param {string} value - цвет
            */
            onPenColorChanged: function (id, value) {
                this.setSeriesOption("lineStyle.normal.color", id, value);
                this.setAxisOption("axisLine.lineStyle.color", id, value);
            },
            /**
             * Обработчик изменения параметра Тип графика у пера
             * @param {number} id - номер пера в массиве перьев
             * @param {enum} value - значение (1 - ступенька, 0 - линия)
             * @returns {} 
             */
            onLineJoinChanged: function (id, value) {
                if (value) {
                    this.setSeriesOption("step", id, 'end');
                } else {
                    this.setSeriesOption("step", id, false);
                }
                this.setChartOption("xAxis.axisPointer.snap", false);
            },
            /**
             * Обработчик изменения стиля линии пера
             * @param {number} id - номер пера в массиве перьев
             * @param {enum} value - стиль
             * @returns {} 
             */
            onPenLineStyleChanged: function (id, value) {
                var style = Appearence.border.getBorderStyle(value);
                if (style == "none") {
                    this.setSeriesOption("lineStyle.normal.opacity", id, 0);
                } else {
                    this.setSeriesOption("lineStyle.normal.opacity", id, 1);
                    this.setSeriesOption("lineStyle.normal.type", id, Appearence.border.getBorderStyle(value));
                }
            },
            /**
             * Обработчик изменения минимума и максимума пера
             * @param {string} path - путь свойства, начиная с yAxis
             * @param {number} id - номер пера в массиве перьев
             * @param {number} value - значение
             * @returns {} 
             */
            onPenMinMaxChanged: function (path, id, value) {
                if (!this.mvc.model.getPenAutoScale(id)) { //если включено автомасштабирование, то менять границы не нужно
                    _.set(this.current_y_axis[id], path, value);
                    this.setChartOption("yAxis", this.current_y_axis);
                }
            },
            /**
             * Обработчик изменения видимости пера
             * @param {number} id - номер пера в массиве перьев
             * @param {boolean} value - значение 
             * @returns {} 
             */
            onPenIsVisibleChanged: function(id, value) {
                if (value) {
                    this.myChart.dispatchAction({
                        type: 'legendSelect',
                        name: this.series[id].name
                    });
                } else {
                    this.myChart.dispatchAction({
                        type: 'legendUnSelect',
                        name: this.series[id].name
                    });
                }
                var selected = {};
                for (var i = 0; i < this.series.length; i++) {
                    selected[this.series[i].name] = this.mvc.model.getPen(i).IsVisible.value;
                }
                this.hideAxis({ name: this.series[id].name, selected: selected });
            },
            onPenDrawConstantChanged: function (index, value) {
                var i = this.drawConstants.findIndex(function (element) {
                    return element.idx === index;
                });
                if (value) {
                    if (i == -1)
                        this.drawConstants.push({ idx: index, draw: true, redraw: true });
                }
                else {
                    this.drawConstants.splice(i, 1);
                }
            },
            /**
             * Функция-заглушка для неопределенных обработчиков параметров
             * @returns {} 
             */
            doNothing: function() {

            },
            onAddedToDOM: function() {
                this._super();
                this.initialized = true;
                this.variablesSource = new TrendVariablesSource(this.clientId());
                this.variablesSource.itemIdToGraphId = this.itemIdToGraphId;
                var connectionSettings = this.addConnectionsForParameter(Role.TREND_LINES);
                this.mvc.model.serverVariables = this.getServerVarsFromConnectionsSettings(connectionSettings, Role.DATA_SOURCE);
                this.myChart = echarts.init(this.mvc.view.$()[0],
                    null,
                    {
                        width: this.getControlWidthPx(),
                        height: this.getControlHeightPx()
                    });
                this.endTime = this.mvc.model.get(Role.TILL);
                this.startTime = this.mvc.model.getTill() - this.mvc.model.getInterval();
                this.modelChanged[Role.TITLE] = this.setChartOption.bind(this, "title.text");
                this.modelChanged[Role.CHART_BACKGROUND] = this.setColorChartOption.bind(this, "grid.backgroundColor");
                this.modelChanged[Role.BACKGROUND_COLOR] = this.setColorChartOption.bind(this, "backgroundColor");
                this.modelChanged[Role.GRID_LINE_THICKNESS] = this.setChartOptionWithConverter.bind(this, this.mvc.model.getGridLineThicknessOption);
                this.modelChanged[Role.GRID_LINE_STROKE_COLOR] = this.setChartOptionWithConverter.bind(this, this.mvc.model.getGridLineStrokeColorOption);
                this.modelChanged[Role.TILL] = this.onTillChanged.bind(this);
                this.modelChanged[Role.AUTOSCROLL] = this.onAutoscrollChanged.bind(this);
                this.modelChanged[Role.INTERVAL] = this.onIntervalChanged.bind(this);
                this.modelChanged[Role.SHIFT] = this.doNothing;
                this.modelChanged[Role.RESAMPLE_INTERVAL] = this.doNothing;
                this.modelChanged[Role.INTERVAL_PAGES_TURN] = this.doNothing;
                this.modelChanged[Role.UPDATE_TIME] = this.onUpdateTimeChanged.bind(this);
                this.modelChanged[Role.TEXT_COLOR] = this.setChartOptionWithConverter.bind(this, this.mvc.model.getTextColorOption.bind(this.mvc.model));
                this.modelChanged[Role.CURSOR_COLOR] = this.setColorChartOption.bind(this, "xAxis.axisPointer.lineStyle.color");
                this.modelChanged[Role.CURSOR_THICKNESS] = this.setChartOption.bind(this, "xAxis.axisPointer.lineStyle.width");
                this.modelChanged[Role.SHOW_CURSOR] = this.setChartOption.bind(this, "xAxis.axisPointer.show");
                this.modelChanged[Role.X_MAJOR_TICK_PADDING_TOP] = this.setChartOption.bind(this, "xAxis.offset");
                this.modelChanged[Role.Y_MAJOR_TICK_PADDING_RIGHT] = this.setYAxisOption.bind(this, "axisLabel.padding");
                this.modelChanged[Role.X_AXIS_STROKE_COLOR] = this.setColorChartOption.bind(this, "xAxis.axisLine.lineStyle.color");
                this.modelChanged[Role.Y_AXIS_STROKE_COLOR] = this.setColorChartOption.bind(this, "yAxis.axisLine.lineStyle.color");
                this.modelChanged[Role.X_AXIS_LABEL] = this.setChartOption.bind(this, "xAxis.name");
                this.modelChanged[Role.Y_AXIS_LABEL] = this.setChartOption.bind(this, "yAxis.name");
                this.modelChanged[Role.X_AXIS_THICKNESS] = this.setChartOption.bind(this, "xAxis.axisLine.lineStyle.width");
                this.modelChanged[Role.Y_AXIS_THICKNESS] = this.setYAxisOption.bind(this, "axisLine.lineStyle.width");
                this.modelChanged[Role.X_MAJOR_TICK_COUNT] = this.setChartOption.bind(this, "xAxis.splitNumber");
                this.modelChanged[Role.Y_MAJOR_TICK_COUNT] = this.setYAxisOption.bind(this, "splitNumber");
                this.modelChanged[Role.X_MAJOR_TICK_WIDTH] = this.setChartOption.bind(this, "xAxis.axisTick.lineStyle.width");
                this.modelChanged[Role.Y_MAJOR_TICK_WIDTH] = this.setYAxisOption.bind(this, "axisTick.lineStyle.width");
                this.modelChanged[Role.X_MAJOR_TICK_COLOR] = this.setColorChartOption.bind(this, "xAxis.axisTick.lineStyle.color");
                this.modelChanged[Role.Y_MAJOR_TICK_COLOR] = this.setColorChartOption.bind(this, "yAxis.axisTick.lineStyle.color");
                this.modelChanged[Role.X_MAJOR_TICK_HEIGHT] = this.setChartOption.bind(this, "xAxis.axisTick.length");
                this.modelChanged[Role.Y_MAJOR_TICK_HEIGHT] = this.setYAxisOption.bind(this, "axisTick.length");
                this.modelChanged[Role.SHOW_LEGEND] = this.setChartOption.bind(this, "legend.show");
                this.modelChanged[Role.LOW_RESOLUTION_MODE] = this.onLowResolutionModeChanged.bind(this);
                this.modelChanged[Role.RESAMPLE_INTERVAL] = this.onResampleIntervalChanged.bind(this);
                this.modelChanged[Role.INTERVAL_PAGES_TURN] = this.onIntervalPagesTurnChanged.bind(this);
                //Role.X_AXIS_FORMAT - формат значений Х обновится автоматически

                var ctx = this;
                this.drawConstants = [];
                this.series = this.mvc.model.getSeriesOption();
                this.current_y_axis = this.mvc.model.getMultiAxisOption();
                if (this.mvc.model.getPens().length !== 0) {
                    this.mvc.model.getPens().forEach(function (element, index) {
                        if ((ctx.mvc.model.serverVariables.getValues[index] &&
                            ctx.mvc.model.serverVariables.getValues[index].type == "STRING") ||
                            (ctx.mvc.model.serverVariables.getValues.length == 0) //перьев нет
                        ) { //подписка только на string-перья, для остальных есть onDataSounceChanged
                            ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.DATA_SOURCE] =
                                ctx.onStringDataSourceChanged.bind(ctx, index);
                        }
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.COLOR] = ctx.onPenColorChanged.bind(ctx, index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.MIN_Y] = ctx.onPenMinMaxChanged.bind(ctx, "min", index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.MAX_Y] = ctx.onPenMinMaxChanged.bind(ctx, "max", index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.LINE_JOIN] = ctx.onLineJoinChanged.bind(ctx, index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.IS_VISIBLE] = ctx.onPenIsVisibleChanged.bind(ctx, index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.LINE_STYLE] = ctx.onPenLineStyleChanged.bind(ctx, index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.AUTOSCALE] = ctx.onPenAutoscaleChanged.bind(ctx, index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.DRAW_CONSTANT] = ctx.onPenDrawConstantChanged.bind(ctx, index);
                        //   ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.TICK_COUNT_Y] = ctx.setSeriesOption.bind(ctx, "series.data", index);
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.UNIT] = ctx.onUnitChanged.bind(ctx, index);
                        //Role.UNIT - надпись в минилегенде изменится автоматически
                        ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.LINE_WIDTH] = ctx.setSeriesOption.bind(ctx, "lineStyle.normal.width", index);
                        //    ctx.modelChanged[Role.TREND_LINES + '[' + index + '].' + Role.FALSEY_VALUES_LINESTYLE] = ctx.setSeriesOption.bind(ctx, "series.data", index);
                        if (element.value.DrawConstant.value) {
                            ctx.drawConstants.push({ idx: index, redraw: true, draw: true});
                        }
                    });
                }
                var option = this.mvc.model.getInitialChartOption();
                setTimeout(function () {
                    ctx.myChart.setOption(option);
                    if (ctx.mvc.model.getPens().length != 0 || !GlobalSettings.OfflineMode.value) {
                        ctx._startupDataExchange();
                    }
                    ctx.subscribeDataSourceChanged();
                    ctx.myChart.on("legendselectchanged",
                        function (params) {
                            var offset = 0;
                            ctx.current_y_axis[ctx.mvc.model.series_names[params.name].idx].show = params.selected[params.name];
                            ctx.current_y_axis.forEach(function (element, index) {
                                element.offset = 30 * offset;
                                if (params.selected[ctx.series[index].name])
                                    offset++;
                            });
                            ctx.setChartOption("yAxis", ctx.current_y_axis);
                        });
                    ctx.myChart.on("datazoom",
                        function (params) {
                            var datazoom = ctx.myChart.getModel().option.dataZoom[0];
                            if (datazoom.endValue - datazoom.startValue <= 0 || !datazoom.startValue || !datazoom.endValue) { //не надо достигать 0 в интервале
                                return;
                            }
                            ctx.startTime = datazoom.startValue;
                            ctx.endTime = datazoom.endValue;
                            ctx.mvc.model.stopAutoscroll();
                            ctx.mvc.model.set(Role.TILL, ctx.endTime, true);
                            ctx.mvc.model.setInterval(ctx.endTime - ctx.startTime);
                            ctx.drawConstants.forEach(function (el) {
                                el.redraw = true;
                            });
                            ctx.getBlocksIfNeeded();
                        });
                    ctx.mvc.model.fireAllPropertiesChanged();
                }, 1);
            },
            /**
             * Спрятать ось Y, пересчитать отступы для оставшихся осей
             * @param {object} params - параметры перьев (структура echarts) 
             * @returns {} 
             */
            hideAxis: function (params) {
                var offset = 0;
                this.current_y_axis[this.mvc.model.series_names[params.name].idx].show = params.selected[params.name];
                this.current_y_axis.forEach(function (element, index) {
                    element.offset = 30 * offset;
                    element.axisLabel.padding = this.mvc.model.get(Role.Y_MAJOR_TICK_PADDING_RIGHT);
                    if (params.selected[this.series[index].name])
                        offset++;
                    this.mvc.model.getPen(index).IsVisible.value = params.selected[this.series[index].name];
                }.bind(this));
                this.setChartOption("yAxis", this.current_y_axis);
            },
            setYAxisOption: function (style, value) {
                this.current_y_axis.forEach(function (element) {
                    _.set(element, style, value);
                });
                this.setChartOption("yAxis", this.current_y_axis);
            },
            /** Получение блоков архива с сервера
             * @async
             * @param {boolean} needUpdate - обязательно ли запрашивать архив
             * @returns {promise} 
             */
            getBlocksIfNeeded: function(needUpdate) {
                var K = 2;
                var lnK = Math.log(K);

                var interval = this.mvc.model.getInterval();
                var bsRank = Math.ceil(Math.log(interval) / lnK);
                if (bsRank != this.currentBlockSizeRank) {
                    needUpdate = true;
                    this.currentBlockSizeRank = bsRank;
                }

                var blockSize = Math.pow(K, bsRank);
                var block = Math.floor(this.endTime / blockSize);
                if (this.currentBlock != block) {
                    needUpdate = true;
                    this.currentBlock = block;
                }

                if (needUpdate) {
                    return this.getAsyncArchiveData((block - 1) * blockSize, (block + 1) * blockSize);
                }
                return Promise.resolve();
            },
            /**
             * Установить видимость для курсора выбранной позиции
             * @param {boolean} value 
             * @returns {} 
             */
            setXAxisPointerVisibility: function(value) {
                this.myChart.setOption({
                    xAxis: {
                        axisPointer: {
                            show: value
                        }
                    }
                });
            },
            onParentActualWidthChanged: function(event) {
                this.resizeTrend();
            },
            onParentActualHeightChanged: function (event) {
                this.resizeTrend();
            },
            /**
             * Пересчет размеров Тренда
             * @returns {} 
             */
            onResourceChanged: function (value) {
                if (value) {
                    var resourceUrl = this.mvc.model.resourceManager.resolve(value);
                    if (this.mvc.view.$('div'))
                        Appearence.background.applyBgImage(this.mvc.view.$('div'), resourceUrl);
                }
            },
            onBackgroundTileChanged: function (value) {
                Appearence.background.applyBackgroundTile(this.mvc.view.$('div'), value);
            },
            resizeTrend: function() {
                if (window.initialized && this.myChart) {
                    this.myChart.resize({
                        width: this.getControlWidthPx(),
                        height: this.getControlHeightPx()
                    });
                }
            },
            /**
             * Добавить фиктивную точку для пересечения с границами видимой области Тренда и дорисовки константы
             * @param {} seriesData - оригинальные точки в Тренде
             * @param {} time - время границы, относительно которой поиск точки пересечения
             * @param {} step - пересечение справа или слева
             * @param {} shift - в какую сторону смещать результат
             * @returns {} 
             */
            addFakeIntersectionPoint: function(seriesData, time, step, shift) {
                var secondpointindex = _.findIndex(seriesData, function (o) { return o[0] > time; }); //найти первую точку, время которой больше левой границы области видимости
                if (secondpointindex != -1 && secondpointindex != 0) {
                    var firstpointindex = secondpointindex - 1;
                    var result;
                    if (step) {
                        result = [time, seriesData[firstpointindex][1], seriesData[firstpointindex][2]];
                    } else {
                        result = Utilites.getIntersection(
                            {
                                x: seriesData[firstpointindex][0],
                                y: seriesData[firstpointindex][1]
                            },
                            {
                                x: seriesData[secondpointindex][0],
                                y: seriesData[secondpointindex][1]
                            },
                            {
                                x: time,
                                y: 0
                            },
                            {
                                x: time,
                                y: seriesData[firstpointindex][1] + seriesData[secondpointindex][1]
                            });
                    }
                    seriesData.splice(secondpointindex, 0, [result[0] + shift, result[1], seriesData[firstpointindex][2], true]);
                    return true;
                }
                return false;
            },
            /**
             * Обновление Тренда
             * @returns {} 
             */
            update: function () {
                var datazoom = this.myChart.getModel().option.dataZoom[0];
                var ctx = this;
                var lastElement, i;
                if (this.mvc.model.getIsAutoScroll()) {
                    this.endTime = this.variablesSource.getServerTime();
                    this.startTime = this.endTime - this.mvc.model.getInterval();
                    this.mvc.model.set(Role.TILL, this.variablesSource.getServerTime());
                }
                this.drawConstants.forEach(function (el) {
                    if (el.redraw || el.draw) {
                        _.remove(ctx.series[el.idx].data,
                            function (o) {
                                return (o[3] == true);
                            });
                        el.redraw = false;
                    }
                });
                for (i = 0; i < ctx.series.length; i++) { //для каждого пера (точки отсортированы по возрастанию)
                    if (this.series[i].data.length > 2) {
                        var index=_.findIndex(this.series[i].data, function(o) { return o[0] > ctx.startTime - ctx.mvc.model.getInterval(); });
                        if (index === -1) { //все значения находятся слева от интервала
                            this.series[i].data = [this.series[i].data[this.series[i].data.length - 1]]; //можно оставить только одно из них
                            continue;
                        }
                        var leftpoint = this.series[i].data[index - 1];
                        var rightpoint = this.series[i].data[_.findIndex(this.series[i].data,
                            function(o) { return o[0] > ctx.endTime + ctx.mvc.model.getInterval(); })];
                        _.remove(this.series[i].data,
                            function(o) {
                                return (o[0] < (ctx.startTime - ctx.mvc.model.getInterval()) ||
                                    o[0] > (ctx.endTime + ctx.mvc.model.getInterval()));
                            });
                        if (leftpoint)
                            this.series[i].data.unshift(leftpoint);
                        if (rightpoint)
                            this.series[i].data.push(rightpoint);
                    }
                }
                this.setChartOption("xAxis.max", this.variablesSource.getServerTime() + this.mvc.model.getInterval()); //допускаем перемещение на 1 интервал в будущее
                this.updateChartDataZoom();
                for (i = 0; i < ctx.series.length; i++) { //для каждого пера (точки отсортированы по возрастанию)
                    lastElement = _.last(this.series[i].data);
                    if (!this.series[i].data.length && lastElement) {
                        this.series[i].data.push(lastElement);
                    }
                    this.addFakeIntersectionPoint(this.series[i].data, datazoom.startValue, this.series[i].step, +1);
                    if (!this.mvc.model.getIsAutoScroll()) {
                        if (datazoom.endValue > this.variablesSource.getServerTime()) {
                            this.addFakeIntersectionPoint(this.series[i].data, this.variablesSource.getServerTime(), this.series[i].step, 0);
                        } else {
                            this.addFakeIntersectionPoint(this.series[i].data, datazoom.endValue, this.series[i].step, -1);
                        }
                    }
                }
                this.drawConstants.forEach(function (el) {
                    lastElement = _.last(ctx.series[el.idx].data);
                    if (el.draw && lastElement) {
                        if (lastElement[0] < datazoom.startValue) {
                            this.series[el.idx].data.push([
                                datazoom.startValue + 1, _.last(this.series[el.idx].data)[1],
                                _.last(this.series[el.idx].data)[2], true
                            ]);
                        }
                        if (lastElement[0] > datazoom.endValue || datazoom.endValue < this.variablesSource.getServerTime()) {
                            if (_.findIndex(ctx.series[el.idx].data, function(o) { return o[0] == datazoom.endValue - 1 }) == -1) {
                                this.series[el.idx].data.push([
                                    datazoom.endValue - 1, _.last(this.series[el.idx].data)[1],
                                    _.last(this.series[el.idx].data)[2], true
                                ]);
                            }
                        }
                        this.series[el.idx].data.push([
                            this.variablesSource.getServerTime() - 1, lastElement[1], lastElement[2], true
                        ]);
                    } else {
                        el.draw = true;
                    }
                }.bind(this));
                this.myChart.setOption({
                    series: this.series
                });
            },
            /**
             * Начать обмен данными с сервером
             * Подписка на параметры Значения перьев, получение границ архива, запрос данных
             * @returns {} 
             */
            _startupDataExchange: function() {
                this.variablesSource.addToSubscription(this.mvc.model.serverVariables, this.mvc.model.getId());
                this.variablesSource.setSourcesOptions({
                    interval: this.mvc.model.get(Role.UPDATE_TIME)
                });
                this.variablesSource.subscribe(this.mvc.model.getId()).then(function() {
                    var pens = this.variablesSource.getPensInfo();
                    for (var i = 0, j = 0; i < this.mvc.model.getPens().length; i++) {
                        if (this.mvc.model.getPens()[i].value.DataSource.connections.length > 0) {
                            this.itemIdToGraphId[pens[j].itemId + pens[j].path] = i;
                            j++;
                        }
                    }
                    this.variablesSource.initArchive().then(function() {
                        this.variablesSource.startTimer();
                        this.variablesSource.updateArchiveInfo().then(function() {
                            this.mvc.model.set(Role.ARCHIVE_END, this.variablesSource.getServerTime());
                            this.mvc.model.set(Role.TILL, this.variablesSource.getServerTime());
                            this.endTime = this.variablesSource.getServerTime();
                            this.startTime = this.endTime - this.mvc.model.getInterval();
                            this.mvc.model.set(Role.ARCHIVE_START, this.variablesSource.getArchiveInfo().minStartTime);
                            this.setChartOption("xAxis.min", this.mvc.model.get(Role.ARCHIVE_START));
                            this.setChartOption("xAxis.max", this.mvc.model.get(Role.ARCHIVE_END) + this.mvc.model.getInterval());
                            this.initialized = true;
                            this.getBlocksIfNeeded(true);
                            this.updateTime = setInterval(this.update.bind(this), this.mvc.model.get(Role.UPDATE_TIME));
                        }.bind(this));
                    }.bind(this));
                }.bind(this));
            },
            /**
             * Подписаться на изменения Значения перьев
             * @returns {} 
             */
            subscribeDataSourceChanged: function() {
                this.variablesSource.subscribePropertyChanged(this.onDataSourceChanged, this);
            },
            /**
             * Обработка изменений стринговых Значений перьев
             * @param {} index - 
             * @param {} path 
             * @returns {} 
             */
            onStringDataSourceChanged: function (index, path) {
                if (!path || path == this.series[index].path)
                    return;
                if (this.series[index].path) {
                    this.variablesSource._getTrendVM().unsubscribePen(this.variablesSource.createVarForStringPen(this.series[index].path));
                    this.itemIdToGraphId['0'+this.series[index].path] = null;
                }
                this.series[index].path = path;
                this.itemIdToGraphId['0' + path] = index;
                this.series[index].data = [];
                this.variablesSource.addToSubscription(this.variablesSource.createVarForStringPen(path), this.mvc.model.getId());
                this.variablesSource._getTrendVM().subscribePen();
                this.variablesSource.initArchive().then(function (result) {
                    this.getBlocksIfNeeded(true);
                }.bind(this));
            },

            onUnitChanged: function (index, value) {
                this.mvc.model.series_names[this.series[index].name].unit = value;
            },
            /**
             * Обработка изменения численных Значений перьев
             * @param {} event 
             * @returns {} 
             */
            onDataSourceChanged: function (event) {
                if (event.idx == undefined) {
                    event.idx = this.itemIdToGraphId[event.property];
                }
                if (!this.series[event.idx]) {
                    return;
                }
                this.mvc.model.set(Role.ARCHIVE_END, this.variablesSource.getServerTime());
                this.mvc.model.series_names[this.series[event.idx].name].lastValue = _.last(event.newValue).value;
                if ((this.endTime > this.variablesSource.getServerTime() - this.mvc.model.getInterval() && this.endTime < this.variablesSource.getServerTime()+this.mvc.model.getInterval()) || this.mvc.model.getIsAutoScroll() || !this.series[event.idx].data.length) {
                    if (this.drawConstants.length && _.findIndex(this.drawConstants, { idx: event.idx }) != -1) {
                        this.drawConstants[_.findIndex(this.drawConstants, { idx: event.idx })].redraw = true; //нужно удалить все ненастоящие точки и дорисовать константу снова
                        this.drawConstants[_.findIndex(this.drawConstants, { idx: event.idx })].draw = false; //но не нужно дорисовывать ее в первый же раз, ведь у нас и так есть новые данные
                    }
                    this.series[event.idx].data = _.concat(this.series[event.idx].data, _.map(event.newValue,
                        function(el) { return [el.sourceTime, +el.value, el.statusCode] }));
                }
            },
            /**
             * Обработка изменения параметра Минимальный вид
             * @param {boolean} value - true - убирается тулбар, уменьшаются отступы 
             * @returns {} 
             */
            onLowResolutionModeChanged: function (value) {
                this.setChartOption("toolbox.show", !value);
                if (value == true) {
                    this.setChartOption("grid.top", 25);
                    this.setChartOption("grid.bottom", 25);
                    this.myChart.setOption({
                        dataZoom: [{}, {
                            show: false
                        }]
                    });

                } else {
                    this.setChartOption("grid.top", 60);
                    this.setChartOption("grid.bottom", 60);
                    this.myChart.setOption({
                        dataZoom: [{}, {
                            show: true
                        }]
                    });
                }
            },
            /**
             * Обработка изменения Интервала листания
             * При действиях Следующая страница и Предыдущая страница перелистывание будет происходить на Интервал * value в процентах
             * Новый_Конец=Конец+Интервал * Шаг_листания
             * @returns {} 
             */
            onIntervalPagesTurnChanged: function(value) {
                var interval = this.mvc.model.get(Role.INTERVAL_PAGES_TURN);
                if (interval >= 0) {
                    return;
                } else {
                    this.mvc.model.set(Role.INTERVAL_PAGES_TURN, 100);
                    Error.warn(String.format(
                        "Интервал листания у тренда {0} принял недопустимое значение {1}. Выставлено значение по умолчанию 100%",
                        this.mvc.model.getId(),
                        interval));
                }
            },
            /**
             * Обработка изменения Интервала прореживания
             * Расчет интервала прореживания
             * Интервал / (Ширина Тренда в пикселях * Шаг прореживания)
             * @param {} event 
             * @returns {} 
             */
            onResampleIntervalChanged: function(event) {
                var resample = this.mvc.model.get(Role.RESAMPLE_INTERVAL);
                if (this.mvc.model.get(Role.RESAMPLE_INTERVAL) < 0) {
                    this.mvc.model.set(Role.RESAMPLE_INTERVAL, 1);
                    Error.warn(String.format(
                        "Коэффициент прореживания у тренда {0} принял недопустимое значение {1}. Выставлено значение по умолчанию 1",
                        this.mvc.model.getId(),
                        resample));
                }
            },
            /**
             * Обработка изменения Конца Тренда
             * @param {time} value - время правой границы области видимости
             * @returns {} 
             */
            onTillChanged: function(value) {
                //    this.mvc.model.stopAutoscroll();
                if (this.endTime != value && value) {
                    this.endTime = value;
                    this.startTime = this.endTime - this.mvc.model.get(Role.INTERVAL);
                    this.getBlocksIfNeeded();
                }
            },
            /**
             * Обработка изменения автомасштабирования пера
             * @param {number} id - номер пера в массиве перьев
             * @param {boolean} value - значение автомасштабирования
             * @returns {} 
             */
            onPenAutoscaleChanged: function(id, value) {
                if (value) {
                    this.current_y_axis[id].min = null;
                    this.current_y_axis[id].max = null;
                } else {
                    this.current_y_axis[id].min = this.mvc.model.getPenMin(id);
                    this.current_y_axis[id].max = this.mvc.model.getPenMax(id);
                }
                this.setChartOption("yAxis", this.current_y_axis);
            },
            onUpdateTimeChanged: function () {
                this.variablesSource.setSourcesOptions({
                    interval: this.mvc.model.get(Role.UPDATE_TIME)
                });
                this.variablesSource.startTimer();
            },
            /**
             * Обработка изменения автопрокрутки Тренда
             * @returns {} 
             */
            onAutoscrollChanged: function() {
                if (this.initialized) {
                    if (this.mvc.model.getIsAutoScroll() === true) {
                        this.mvc.model.set(Role.TILL, this.variablesSource.getServerTime());
                    }
                }
            },
            /**
             * Обработка изменения Интервала Тренда
             * @param {time} value 
             * @returns {} 
             */
            onIntervalChanged: function (value) {
                if (value < this.mvc.model.get(Role.UPDATE_TIME)) {
                    this.mvc.model.setInterval(this.mvc.model.get(Role.UPDATE_TIME));
                    return;
                }
                if (this.startTime != this.endTime - this.mvc.model.getInterval()) {
                    this.startTime = this.endTime - this.mvc.model.getInterval();
                    if (Number.isFinite(this.mvc.model.get(Role.ARCHIVE_START))) {
                        if (this.startTime < this.mvc.model.get(Role.ARCHIVE_START)) {
                            this.startTime = this.mvc.model.get(Role.ARCHIVE_START);
                            this.endTime = this.startTime + this.mvc.model.getInterval();
                        }
                        this.updateViewPort();
                    } else {
                        this.updateChartDataZoom();
                    }
                }
            },
            /**
             * Обновить
             * @returns {} 
             */
            updateViewPort: function() {
                this.mvc.model.stopAutoscroll();
                this.updateChartDataZoom();
                return this.getBlocksIfNeeded();
            },
            /**
            * Перелистнуть на следующую страницу
            */
            nextPage: function() {
                var interval = this.mvc.model.getInterval() * this.mvc.model.getIntervalTurn() / 100;
                this.startTime += interval;
                this.endTime += interval;
                return this.updateViewPort();
            },
            /**
            * Перелистнуть на предыдущую страницу
            */
            previousPage: function() {
                var interval = this.mvc.model.getInterval() * this.mvc.model.getIntervalTurn() / 100;
                this.startTime -= interval;
                this.endTime -= interval;
                return this.updateViewPort();
            },
            /**
            * Перелистать на первую страницу
            */
            firstPage: function() {
                this.startTime = this.mvc.model.get(Role.ARCHIVE_START);
                this.endTime = this.startTime + this.mvc.model.getInterval();
                return this.updateViewPort();
            },
            /**
            * Перелистать на последнюю страницу
            */
            lastPage: function() {
                this.endTime = this.mvc.model.get(Role.ARCHIVE_END);
                this.startTime = this.endTime - this.mvc.model.getInterval();
                return this.updateViewPort();
            },
            /**
             * Обновить позлунок времени внизу
             * @returns {} 
             */
            updateChartDataZoom: function() {
                this.myChart.setOption({
                    dataZoom: {
                        startValue: this.startTime,
                        endValue: this.endTime
                    }
                });
            },
            /**
             * Обновить ось Х
             * @returns {} 
             */
            updateXAxis: function() {
                this.setChartOption("xAxis.min", this.startTime);
                this.setChartOption("xAxis.max", this.endTime);
            },
            /**
             * @async
             * Запрос архива с сервера
             * @param {number} startTime Время начала интервала в мс 
             * @param {number} endTime Время конца интервала в мс
             * @returns {promise} 
             */
            getArchiveData: function (startTime, endTime) {
                for (var i = 0; i < this.series.length; i++) {
                    this.series[i].data = [];
                }
                return this.variablesSource.getArchiveData(startTime, endTime, this.calcResampleInterval()).then(
                    function (result) {
                        if (result) {
                            for (var i = 0; i < this.mvc.model.getPens().length && i < result.length; i++) {
                                this.series[i].data = result[i].data;
                            }
                            this.myChart.setOption({
                                series: this.series
                            });
                        }
                    }.bind(this));
            },
            /**
             * @async
             * Асинхронный запрос архива с сервера
             * @param {number} startTime Время начала интервала в мс 
             * @param {number} endTime Время конца интервала в мс
             * @returns {promise} 
             */
            getAsyncArchiveData: function (startTime, endTime) {
                return this.variablesSource.getAsyncArchiveData(startTime, endTime, this.calcResampleInterval()).then(
                    function (result) {
                        if (result) {
                            for (var i = 0; i < this.mvc.model.getPens().length && i < result.length; i++) {
                                if (result[i] && result[i].length > 0)
                                    this.series[i].data = result[i];
                            }
                            this.myChart.setOption({
                                series: this.series
                            });
                        }
                    }.bind(this));
            },
            /**
             * Подписаться на события
             * @param {} elem 
             * @returns {} 
             */
            attachEvents: function (elem) {
                this._super(elem);
                var self = this;
                elem.dblclick(function (event) { self.onDoubleClick(event); });
            },
            /**
             * Событие двойного клика - включить автоскролл
             * @returns {} 
             */
            onDoubleClick: function() {
                this.mvc.model.set(Role.AUTOSCROLL, !this.mvc.model.get(Role.AUTOSCROLL));
            },
            /**
             * Расчет интервала прореживания в мс. Сервер вернет не более 4 значений на интервал
             * @returns {number} Интервал в мс 
             */
            calcResampleInterval: function() {
                return this.mvc.model.getInterval() /
                    this.getControlWidthPx() *
                    this.mvc.model.get(Role.RESAMPLE_INTERVAL);
            },
            /**
            * Деструктор
            */
            onDestroy: function() {
                this._super();
                clearInterval(this.updateTime);
                this.variablesSource.unsubscribeAll();
            }
        });
    });