#!/bin/bash

rootfs_dir="/tmp/ubi_rootfs"

create_mtd() {
	if mtdpart add /dev/mtd0 "$1" "$2" "$3" >/dev/null 2>&1 ; then
		echo "/dev/$(awk -v n="\"$1\"" 'FNR > 1 { if($4 == n)  print substr($1, 1, length($1)-1) }' /proc/mtd)"
	fi
}

backup_impl() {
	insat_mount=0
	ubifs_mount=0
	attach=0

	mtd_ubifs=$(create_mtd ubifs 0x00f20000 0x070E0000)

	if ! [ -n "$mtd_ubifs" ] ; then
		echo "Not create mtd partition"
		return 8
	fi

	if ! ubiattach -p "$mtd_ubifs" ; then
		echo "Not attach UBI partition"
		return 9
	else
		attach=1
	fi

	mkdir -p "$rootfs_dir"
	if mount -t ubifs -o ro ubi0:rootfs "$rootfs_dir" ; then
		ubifs_mount=1
	else
		echo "Not mount UbiFs rootfs"
		return 10
	fi

	if [ -d "$rootfs_dir/opt" ] ; then
		if mount -o ro /dev/mmcblk1p1 "$rootfs_dir/opt/" ; then
			insat_mount=1
		else
			echo "InSat filesystem not mount"
			return 11
		fi
	fi

	echo "Packing backup...."
	if tar -cp -C "$rootfs_dir" . | xz -zf - > "$3" ; then
		echo "Backup sucess"
		return 0
	else
		echo "Create arch error"
		return 12
	fi
}

backup() {
	backup_impl "$@"
	local ret="$?"

	[ $insat_mount -ne 0 ] && umount /dev/mmcblk1p1
	[ $ubifs_mount -ne 0 ] && umount "$rootfs_dir"
	[ -d "$rootfs_dir" ] && rm -rf "$rootfs_dir"
	[ $attach -ne 0 ] && ubidetach -p "$mtd_ubifs"
	[ -n "$mtd_ubifs" ] && [ -c "$mtd_ubifs" ] && mtdpart del /dev/mtd0 "${mtd_ubifs: -1}"
	return $ret
}

install_recovery_impl() {
	insat_mount=0
	ubifs_mount=0
	attach=0

	mtd_spl=$(create_mtd spl 0x00000000 0x00140000)
	mtd_uboot=$(create_mtd uboot_main 0x00140000 0x00060000)
	mtd_ubifs=$(create_mtd ubifs 0x00f20000 0x070E0000)

	if ! [ -n "$mtd_spl" -a -n "$mtd_uboot" -a -n "$mtd_ubifs" ] ; then
		echo "Not create mtd partition"
		return 6
	fi

	if ( ubiformat "$mtd_ubifs" && ubiattach -p "$mtd_ubifs" ) ; then
		attach=1
	else
		echo "Not format ubifs filesystem"
		return 7
	fi

	if ! mkdir "$rootfs_dir" ; then 
		echo "ubifs tmp dir create error"
		return 8
	fi


	if ( ubimkvol /dev/ubi0 -N rootfs -m -t dynamic  && mount -t ubifs -o rw /dev/ubi0_0 "$rootfs_dir" ) ; then
		ubifs_mount=1
	else
		echo "Unable create volume or mount"
		return 9
	fi

	if ! mkdir "$rootfs_dir/opt" ; then
		echo "Not create /opt folder"
		return 10
	fi

	fdisk /dev/mmcblk1 <<EOF_SB
o
n
p
1


w

EOF_SB
	if [ $? -ne 0 ] ; then
		echo "Format eMMC error"
		return 11
	fi

	fdisk /dev/mmcblk1 <<EOF_SB && echo "MMC reloading" || exit 13
p
EOF_SB

	if ( mkfs.ext4 -t ext4 -F -L InSatFs /dev/mmcblk1p1 && mount -o rw /dev/mmcblk1p1 "$rootfs_dir/opt/" ) ; then
		insat_mount=1
	else
		echo "Create FS om eMMC error"
		return 12
	fi

	echo "Unpacking rootfs....."
	if ! tar -xpJf "$2" --exclude "./raw_images" -C "$rootfs_dir" ; then
		echo "Unpack archive error"
		return 13
	fi
	sync
	return 0
}

install_recovery() {
	install_recovery_impl "$@"
	local ret="$?"

	[ $insat_mount -ne 0 ] && umount /dev/mmcblk1p1
	[ $ubifs_mount -ne 0 ] && umount /dev/ubi0_0
	[ -d "$rootfs_dir" ] && rm -rf "$rootfs_dir"
	[ $attach -ne 0 ] && ubidetach -p "$mtd_ubifs"

	[ -n "$mtd_spl" ] && [ -c "$mtd_spl" ] && mtdpart del /dev/mtd0 "${mtd_spl: -1}"
	[ -n "$mtd_uboot" ] && [ -c "$mtd_uboot" ] && mtdpart del /dev/mtd0 "${mtd_uboot: -1}"
	[ -n "$mtd_ubifs" ] && [ -c "$mtd_ubifs" ] && mtdpart del /dev/mtd0 "${mtd_ubifs: -1}"
	return $ret
}

############################ Production part ############################

spl_path="/mnt/imgs/SPL.izm2-nand"
u_boot_path="/mnt/imgs/u-boot.izm2-nand.img"
img_path="/mnt/imgs/c3000t-image-c3000t.ubi"
reocov_path="/mnt/imgs/fitImage-c3000hub-recovery-orion-initramfs.bin"

send_status() {
	case "$1" in
		init)
			exec 2>&1
			if [ "$#" -eq 1 ] || [ "$2" == "-" ] || [ -z "$2" ] ; then
				exec 5>&1
			else
				exec 5<> "$2"
			fi
			;;
		nand_erase)
			local s="$2"
			local e="$3"
			local m="$4"
			shift 4
			eval "$@"
			local r="$?"
			echo -e "\n@@@@@ $e @@@@@ $m" >&5
			return "$r"
			;;
		ubi_format)
			local s="$2"
			local e="$3"
			local m="$4"
			shift 4
			eval "$@" | awk -vRS='\r' -vs="$s" -ve="$e" 'BEGIN { p1=0; p2=0; p3=0; s1=(e-s)/1000; s2=(e-s)*7/1000; s3=(e-s)/500; e=s-1 } (NR>7){ if($2 == "scanning") {p1=$6} else if($2 == "flashing") {p2=$6} else if($2 == "formatting") {p3=$6} ; m = int(p1*s1 + p2*s2 + p3*s3) + s; if(m != e) { e = m; print "\n@@@@@ " m " @@@@@ " } } { print }'
			local r="$?"
			echo -e "\n@@@@@ $e @@@@@ $m" >&5
			return "$r"
			;;
		*)
			if [ "$#" -eq 2 ] ; then
				echo -e "\n@@@@@ $1 @@@@@ $2" >&5
			else
				local s=$1
				local m=$2
				shift 2
				eval "$@"
				local r="$?"
				echo -e "\n@@@@@ $s @@@@@ $m" >&5
				return "$r"
			fi
			;;
	esac
}

write_firmware_impl() {
	insat_mount=0
	ubifs_mount=0
	attach=0
	recovery_mount=0

	send_status init "$1"

	mtd_spl=$(create_mtd spl 0x00000000 0x00140000)
	mtd_uboot=$(create_mtd uboot_main 0x00140000 0x00060000)
	mtd_ubootenv=$(create_mtd uboot_env 0x001A0000 0x00020000)
	mtd_recover=$(create_mtd recover 0x008A0000 0x00680000)
	mtd_ubifs=$(create_mtd ubifs 0x00f20000 0x070E0000)

	if ! [ -n "$mtd_spl" -a -n "$mtd_uboot" -a -n "$mtd_ubifs" -a -n "$mtd_recover" ] ; then
		echo "Not create mtd partition"
		echo "mtd_spl=$mtd_spl mtd_uboot=$mtd_uboot mtd_ubifs=$mtd_ubifs mtd_recover=$mtd_recover"
		return 1
	fi

	send_status 1 "Create NAND partitions"

	if [[ -f "$spl_path" && -f "$u_boot_path" && -f "$img_path" && -f "$reocov_path"  ]] ; then 
		send_status 2 "Check needed files"
	else
		echo "Not found need files"
		echo "spl_path=$spl_path u_boot_path=$u_boot_path img_path=$img_path reocov_path=$reocov_path"
		return 2
	fi

	# U-boot install
	send_status 3 "Erase NAND SPL" flash_erase "$mtd_spl" 0 0 || return 3
	send_status 4 "Write NAND SPL" kobs-ng init -x "$spl_path" --search_exponent=1 -v --chip_0_device_path="$mtd_spl" || return 4
	flash_erase "$mtd_ubootenv" 0 0 && echo "NAND u-boot env section erased" || return 5
	send_status 5 "Erase NAND u-boot" flash_erase "$mtd_uboot" 0 0 || return 6
	send_status 6 "Write NAND u-boot" nandwrite -p "$mtd_uboot" "$u_boot_path" || return 7

	# Install recovery
	send_status nand_erase 7 24 "Erase NAND recovery FIT image" flash_erase "$mtd_recover" 0 0 || return 8
	send_status 42 "Write NAND recovery FIT image" nandwrite -p "$mtd_recover" "$reocov_path" || return 9

	# UbiFs install
	send_status nand_erase 43 60 "Erase NAND rootfs image" flash_erase "$mtd_ubifs" 0 0 || return 10
	send_status ubi_format 61 78 "Write NAND rootfs" ubiformat "$mtd_ubifs" -s 2048 -O 2048 -f "$img_path" || return 11
	ubiattach -p "$mtd_ubifs" && echo "UBIFS attached" || return 12
	attach=1
	mkdir -p /tmp/ubi_rootfs
	send_status 79 "UBIFS mounted" mount -t ubifs /dev/ubi0_0 /tmp/ubi_rootfs || return 13
	ubifs_mount=1
	
	send_status 81 "MMC formated" fdisk /dev/mmcblk1 <<EOF_SB || return 14
	o
	n
	p
	1


	w

EOF_SB
	send_status 82 "MMC syncing" sync
	send_status 83 "MMC reloading" fdisk /dev/mmcblk1 <<EOF_SB || return 15
	p
EOF_SB

	send_status 95 "MMC datafs formating" mkfs.ext4 -t ext4 -F -L InfatFs /dev/mmcblk1p1 || return 16
	mkdir -p /tmp/mmc_datafs
	send_status 96 "MMC datafs mounted" mount /dev/mmcblk1p1 /tmp/mmc_datafs || return 17
	insat_mount=1

	cp -rp /tmp/ubi_rootfs/opt/* /tmp/mmc_datafs && echo "Insat soft moved" || return 18
	rm -rf /tmp/ubi_rootfs/opt/*
	send_status 97 "Insat soft installed"

	echo "Install recovery"
	mkdir -p /tmp/recovery_datafs
	echo '0' > "/sys/block/mmcblk1boot0/force_ro" && dd if=/dev/zero of=/dev/mmcblk1boot0 bs=512 count="$(cat /sys/block/mmcblk1boot0/size)" && mke2fs -L CfgFs /dev/mmcblk1boot0 && mount -o rw /dev/mmcblk1boot0 /tmp/recovery_datafs || return 19
	recovery_mount=1
	cp /tmp/ubi_rootfs/install.sh /tmp/recovery_datafs/backup_1234.sh && sha3sum /tmp/recovery_datafs/backup_1234.sh > /tmp/recovery_datafs/backup_1234.sh.sig || return 20
	send_status 98 "Backup script installed"
	sync
	send_status 99 "All Syncing" sync
	echo "############ All soft installed ############"
	return 0
}

write_firmware() {
	write_firmware_impl "$@"
	local r=$?
	[ $recovery_mount -ne 0 ] && umount /dev/mmcblk1boot0
	[ $insat_mount -ne 0 ] && umount /dev/mmcblk1p1
	[ $ubifs_mount -ne 0 ] && umount /dev/ubi0_0
	[ -d "/tmp/recovery_datafs" ] && rm -rf "/tmp/recovery_datafs"
	[ -d "/tmp/ubi_rootfs" ] && rm -rf "/tmp/ubi_rootfs"
	[ -d "/tmp/mmc_datafs" ] && rm -rf "/tmp/mmc_datafs"
	[ $attach -ne 0 ] && ubidetach -p "$mtd_ubifs"

	[ -n "$mtd_spl" ] && [ -c "$mtd_spl" ] && mtdpart del /dev/mtd0 "${mtd_spl: -1}"
	[ -n "$mtd_uboot" ] && [ -c "$mtd_uboot" ] && mtdpart del /dev/mtd0 "${mtd_uboot: -1}"
	[ -n "$mtd_ubootenv" ] && [ -c "$mtd_ubootenv" ] && mtdpart del /dev/mtd0 "${mtd_ubootenv: -1}"
	[ -n "$mtd_recover" ] && [ -c "$mtd_recover" ] && mtdpart del /dev/mtd0 "${mtd_recover: -1}"
	[ -n "$mtd_ubifs" ] && [ -c "$mtd_ubifs" ] && mtdpart del /dev/mtd0 "${mtd_ubifs: -1}"
	if (( r == 0 )) ; then
		send_status 100 "Umount all"
	fi
	return $r
}

insat_mount() {
	mkdir -p /tmp/instat_mount 1>&2
	mount /dev/mmcblk1p1 /tmp/instat_mount 1>&2
	local r=$?
	echo "/tmp/instat_mount/mplc4"
	exit $r
}

insat_umount() {
	sync 1>&2
	umount -f /dev/mmcblk1p1 1>&2
	local r=$?
	rmdir /tmp/instat_mount/ 1>&2
	sync  1>&2
	exit $r
}


case "$1" in
	-i) 
		install_recovery "$@"
		exit "$?" 
		;;
	-b) 
		backup "$@"
		exit "$?" 
		;;
	-w)
		shift
		write_firmware "$@"
		exit "$?" 
		;;
	-mi)
		shift
		insat_mount
		exit "$?" 
		;;
	-ui)
		shift
		insat_umount
		exit "$?" 
		;;
	*) 
		echo "Unknown action" >&2
		exit 15
esac
