﻿// <copyright file="CanvasContainerController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class CanvasContainerController
* Base class for controller of container controls 
*   (like window and other controls with absolute positioning)
*/
define(['base/ContainerController', 'common/Enums', 'common/Utilites'],
function (ContainerController, Enums, Utilites) {

    var CanvasContainerController = ContainerController.extend({

        childModelPropertyChangedInternal: function (event) {
            this._super();

            var childControl;

            switch (event.property) {
                case Enums.ParameterRoles.X_UNITS:
                case Enums.ParameterRoles.X:
                    childControl = this.controls[event.target.getId()];
                    this.onChildXChanged(childControl);
                    break;
                case Enums.ParameterRoles.Y_UNITS:
                case Enums.ParameterRoles.Y:
                    childControl = this.controls[event.target.getId()];
                    this.onChildYChanged(childControl);
                    break;
                case Enums.ParameterRoles.WIDTH:
                    this.onChildWidthChanged(event);
                    break;
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildWidthUnitsChanged(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                    this.onChildHeightChanged(event);
                    break;
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildHeightUnitsChanged(event);
                    break;
                case Enums.ParameterRoles.ANCHOR:
                    childControl = this.controls[event.target.getId()];
                    this.onAnchorChanged(childControl);
                    break;
            }
        },

        // изменяем привязку
        onAnchorChanged: function (childControl) {
            switch (childControl.model.getAnchor()) {
                case Enums.AnchorType.Up: {
                    this.setChildX(childControl, this.getChildX(childControl));
                    this.setChildY(childControl, 0);
                }
                    break;
                case Enums.AnchorType.Down: {
                    this.setChildX(childControl, this.getChildX(childControl));
                    this.setChildY(childControl, this.getOwnHeight() - childControl.model.getHeight());
                }
                    break;
                case Enums.AnchorType.Right: {
                    this.setChildX(childControl, this.getOwnWidth() - childControl.model.getWidth());
                    this.setChildY(childControl, this.getChildY(childControl));
                }
                    break;
                case Enums.AnchorType.Left: {
                    this.setChildX(childControl, 0);
                    this.setChildY(childControl, this.getChildY(childControl));
                }
                    break;
            }
        },

        applyChildX: function (childControl, layoutTypeValue) {

            //abs is for line support
            //в редакторе позиция указывается без учета толщины border
            var x = this.getChildX(childControl),
                xunits = childControl.model.getXUnits(),               
                parentWidth = this.getOwnWidth(),
                displayValue;

            if (xunits == Enums.SizeType.Relative) {
                displayValue = (100 * x / parentWidth);
            }
            else {
                displayValue = x;
            }

            if (childControl.model.getWidth() < 0) {
                //we need to shift control up
                displayValue += this.calcChildWidth(childControl.model, childControl.model.getXUnits());
            }

            layoutTypeValue === Enums.LayoutType.Right ?
                childControl.controller.setControlRight(displayValue + Utilites.getMeasureSuffix(xunits)) :
                childControl.controller.setControlLeft(displayValue + Utilites.getMeasureSuffix(xunits));
        },

        getOwnWidth: function () {
            return this.mvc.model.getWidth();
        },

        getChildX: function (childControl) {
            if (childControl.model.getX() != undefined) {
                return childControl.model.getX();
                //TODO определить правила позиционирования в редакторе
                //в окне редактора Х учитывает бордер -this.mvc.model.getBorderThickness();
            }

            return undefined;
        },

        setChildX:function(childControl, value){
            return childControl.model.setX(value);
        },


        applyChildY: function (childControl, layoutTypeValue) {

            //abs is for line support
            //в редакторе позиция указывается без учета толщины родительского border
            var y = this.getChildY(childControl),
                yunits = childControl.model.getYUnits(),             
                parentHeight = this.getOwnHeight(),
                displayValue;

            if (yunits == Enums.SizeType.Relative) {
                displayValue = (100 * y / parentHeight);
            }
            else {
                displayValue = y;
            }

            if (childControl.model.getHeight() < 0) {
                //we need to shift control up
                displayValue += this.calcChildHeigth(childControl.model, childControl.model.getYUnits());
            }

            layoutTypeValue === Enums.LayoutType.Down ?
                childControl.controller.setControlBottom(displayValue + Utilites.getMeasureSuffix(yunits)) :
                childControl.controller.setControlTop(displayValue + Utilites.getMeasureSuffix(yunits));
        },

        setControlTop: function (value) {
            this.mvc.view.$().css('bottom', '');
            this._super(value);
        },

        setControlLeft: function (value) {
            this.mvc.view.$().css('right', '');
            this._super(value);
        },

        setControlBottom: function (value) {
            this.mvc.view.$().css('top', '');
            this._super(value);
        },

        setControlRight: function (value) {
            this.mvc.view.$().css('left', '');
            this._super(value);
        },


        getOwnHeight: function () {
            return this.mvc.model.getHeight();
        },

        getChildY: function (childControl) {
            if (childControl.model.getY() != undefined) {
                return childControl.model.getY(); // - this.mvc.model.getBorderThickness();
            }

            return undefined;
        },

        setChildY: function (childControl, value) {
            return childControl.model.setY(value);
        },

        /*
        * @method normalizeCoord normalizes relative coordinates for WebKit browsers
        *   to avoid rounding in browser, which may result in disappearing 1px things.
        *   will be called in context of child control
        * @param c {number} coord of child
        * @param p {number} dimension of parent
        * @returns normalized coord in percents
        */
        _normalizeCoord: function (c, p) {
            if (c === undefined || p === undefined) {
                return 0;
            }

            var parentDiv = document.createElement('div'),
                childDiv = document.createElement('div'),
                percent = Utilites.getMeasureSuffix(Enums.SizeType.Relative),
                computedOffset,
                result;

            parentDiv.style.width = p + 'px';
            // use width attribute. No matter, what property we are rounding.
            // just sure that we are using percents.
            childDiv.style.width = (c * 100 / Math.floor(p - 2 * this.mvc.model.getBorderThickness())) + percent;
            parentDiv.appendChild(childDiv);
            document.body.appendChild(parentDiv);
            computedOffset = Math.floor(childDiv.offsetWidth);
            document.body.removeChild(parentDiv);

            result = computedOffset * 100 / p;
            return result;
        },



        /*
        * @method calcChildWidth calculates width for child control
        *   will be called in context of child control
        * @param model {object} model of control for which to calculate
        * @param measureUnits {Enums.MeasureUnits} says in which units should be result
        * @returns width of control in pixels or percents
        */
        calcChildWidth: function (model, measureUnits) {

            //width can be negative, we need to keep sign

            var width = model.getWidth();
            var parentWidth;

            var displayValue;
            if (measureUnits == Enums.SizeType.Relative) {
                parentWidth = Math.abs(this.mvc.model.getWidth());
                if (!parentWidth) {
                    return 0;
                }
                displayValue = (100 * width / parentWidth);
            }
            else {
                displayValue = width;
            }

            return displayValue;
        },

        /*
        * @method calcChildHeigth calculates heigth for child control
        *   will be called in context of child control
        * @param model {object} model of control for which to calculate
        * @param measureUnits {Enums.MeasureUnits} says in which units should be result
        * @returns heigth of control in pixels or percents
        */
        calcChildHeigth: function (model, measureUnits) {

            //heigth can be negative, we need to keep sign

            var height = model.getHeight();
            var parentHeight;

            var displayValue;
            if (measureUnits == Enums.SizeType.Relative) {
                parentHeight = Math.abs(this.mvc.model.getHeight());
                if (!parentHeight) {
                    return 0;
                }
                displayValue = (100 * height / parentHeight);
            }
            else {
                displayValue = height;
            }

            return displayValue;
        },

        onChildXChanged: function (childControl) {
            this.applyChildX(childControl);
        },

        onChildYChanged: function (childControl) {
            this.applyChildY(childControl);
        },

        onChildXUnitsChanged: function (childControl) {
            this.applyChildX(childControl);
        },

        onChildYUnitsChanged: function (childControl) {
            this.applyChildY(childControl);
        },

        //support for negative width
        onChildWidthChanged: function (event) {
            var newValue = event.target.getWidth();

            if ((newValue <= 0) ||
                (newValue >= 0 && event.oldValue >= 0)) {
                return;
            }

            // width has changed its sign so we need to move control
            // this.applyChildX(this.controls[event.target.getId()]);
        },

        //support for negative height
        onChildHeightChanged: function (event) {
            var newValue = event.target.getHeight();

            if ((newValue <= 0) ||
                (newValue >= 0 && event.oldValue >= 0)) {
                return;
            }

            // height has changed its sign so we need to move control
            // this.applyChildY(this.controls[event.target.getId()]);
        },

        //support for negative width
        onChildWidthUnitsChanged: function (event) {
            if (event.target.getWidth() < 0) {
                //width has changed so we need to move control
                this.applyChildX(this.controls[event.target.getId()]);
            }
        },

        //support for negative height
        onChildHeightUnitsChanged: function (event) {
            if (event.target.getHeight() < 0) {
                //height has changed so we need to move control
                this.applyChildY(this.controls[event.target.getId()]);
            }
        },

        onProportionTypeChanged: function (event) {

        }
    });

    return CanvasContainerController;
});
