﻿// <copyright file="ControlController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class ContainerController
* Base class for controller of container controls (like window and panels)
*/
define(['base/ControlController', 'common/Enums', 'common/Utilites', 'core/ModulesProvider', 'common/Error',
    'helpers/contextResolver'],
    function (ControlController, Enums, Utilites, ModulesProvider, Error, contextResolver) {

        var ContainerController = ControlController.extend({
            _currentChild: undefined,
            controls: null,
            orderedControls: null,

            init: function () {
                this._super();

                this.controls = {};
                this.orderedControls = [];
                this.initialized = false;
                //panel Data Source connections            
                this.childConnections = {}; //id : {to, from conns}
                this.PDSChildGroups = []; //{conns, elements}
            },

            getContainerTagSelector: function () {
                //undefined means root tag
                return undefined;
            },

            addTypeChildControls: function () {
                var type = this.mvc.model.getType(),
                    childModels = type.getChildModels();

                this.addChildControls(childModels);
            },

            //context was added for handling panel connections in window
            addChildControls: function (childModels, context) {

                var i;

                if (!childModels || childModels.length === 0) {
                    return;
                }

                for (i = 0; i < childModels.length; i += 1) {
                    this.addChildControl(context || this.mvc, childModels[i], this.mvc.model.resourceManager);
                }
            },

            addChildControl: function (context, childModel, parentResourceManager) {
                try {
                    if (childModel.cssClass) {
                        childModel.parameters.optimization = true;
                    }
                    var childControl = ModulesProvider.getControlFactory().createControl(childModel, parentResourceManager);

                    childControl.model.subscribeOnInit(this.childModelInit, this);
                    childControl.model.subscribePropertyChanged(this.childModelPropertyChanged, this);
                    childControl.model.subscribePropertyChanging(this.childModelPropertyChanging, this);

                    this._currentChild = childControl;
                    this.orderedControls.push(childControl);
                    childControl.model.initModel(childModel, context);
                    this.appendChild(childControl);
                    this.addChildControlConnections(context, childControl);
                    this._currentChild = undefined;
                    //end statistic
                    return childControl;
                }
                catch (e) {
                    Error.exception(e);
                }
            },

            getPermissionsContext: function (childModel) {
                try {
                    var childControl = ModulesProvider.getControlFactory().createControl(childModel);
                    if (childModel && childModel.permissions) {
                        childControl.permissions = $.extend(true, childControl.permissions, childModel.permissions);
                    }
                    return childControl;
                }
                catch (e) {
                    Error.exception(e);
                }
            },

            onDestroy: function () {
                this._super();
                if (this.mvc.model.isPropertyExists(Enums.ParameterRoles.PANEL_DATA_SOURCE) && this.mvc.model.get(Enums.ParameterRoles.PANEL_DATA_SOURCE)) {
                    var parentWindowObject = contextResolver.getParentWindowControl(this);
                    var parentWindowCH = parentWindowObject.model.getConnectionHandler();
                    parentWindowCH.connections = this.init_connections;
                }
            },
            addChildControlConnections: function (context, childControl) {
                // Все связи обрабатываются в контексте узла родительского элемента и принадлежат ему.
                // Нам нужно подписаться после того, как модель проинициализировалась и сработал firePropertyChanged.
                // Это нужно потому что в данный момент еще не созданы все дочерние элементы контрола.
                childControl.model.subscribePropertyChanged(function (childControlController) {
                    return function (event) {
                        var path = childControlController.buildPath() + event.property;
                        this.model.updateTypeConnection(path, this, event.oldValue);
                    }
                }(childControl.controller), context);
                _.forOwn(childControl.controller.actions, function (action) {
                    action.subscribePropertyChanged(function(childControlController, action) {
                        return function(event) {
                            var path = childControlController.buildPath() + action.id + '/' + event.property;
                            this.model.overrideConnectionsOperation(path, action.constants.operation);
                            this.model.updateTypeConnection(path, this);
                            this.model.restoreConnectionsOperation(action.constants.operation);
                        }
                    }(childControl.controller, action), context);
                });

                _.forOwn(childControl.controller.events, function (actionEvent) {
                    actionEvent.subscribePropertyChanged(function (childControlController, actionEvent) {
                        return function (event) {
                            var path = childControlController.buildPath() + actionEvent.type + '/' + event.property;
                            this.model.updateTypeConnection(path, this);
                        }
                    }(childControl.controller, actionEvent), context);
                });
            },

            removeChildControls: function () {
                for (var i = 0; i < this.orderedControls.length; i++) {
                    this.removeChildControlInternal(this.orderedControls[i]);
                }

                this.orderedControls = [];
            },

            removeChildControl: function (childControl) {
                this.removeChildControlInternal(childControl);
                _.remove(this.orderedControls, function (orderedControl) { return orderedControl.model.getId() === childControl.model.getId() });
            },

            removeChildControlInternal: function (childControl) {
                childControl.model.destroy();
                childControl.destroy();
            },

            appendChild: function (childControl) {
                this.mvc.append(childControl, this.getContainerTagSelector());
            },

            /**
            *  Fired upon object creation         
            */
            onCreate: function () {
                this._super();
                this.mvc.model.subscribeOnChildrenAdded(this.onChildrenAdded, this);
                this.addTypeChildControls();

                function processConnections(connections) {
                    var i;
                    for (i = connections.length; i--; ) {
                        this.mvc.model.getType()._connectionsHandler.add(connections[i], this);
                    }
                }

                // связи типа обрабатываем один раз
                if (!this.mvc.model.getType().connectionsProcessed) {
                    processConnections.call(this, this.mvc.model.getType().connections || []);
                    this.mvc.model.getType().connectionsProcessed = true;
                }

                if (this.mvc.model && this.mvc.model.connections) {
                    //apply connections
                    processConnections.call(this, this.mvc.model.connections || []);
                }
            },

            onOneTimeConnectionsProcess: function(){
                this._super();

                for (var i in this.mvc._container.children) {
                    this.mvc._container.children[i].controller.onOneTimeConnectionsProcess();
                }
            },


            onAddedToDOM: function () {
                this._super();

                for (var i in this.mvc._container.children) {
                    this.mvc._container.children[i].controller.onAddedToDOM();
                }

                if (this.mvc.model.isPropertyExists(Enums.ParameterRoles.PANEL_DATA_SOURCE)) {
                    this.init_connections = $.extend(true, {}, contextResolver.getParentWindowControl(this).model.getConnectionHandler().connections); //сохраняем связи, чтобы потом при уничтожении их можно было откатить в изначальное состояние
                    this._addPanelSoruceControls();
                    this.initialized = true;
                } else {
                    this.fullReposition();
                }
            },

            onAfterOneTimeConnectionsProcessed: function () {
                this._super();

                for (var i in this.mvc._container.children) {
                    this.mvc._container.children[i].controller.onAfterOneTimeConnectionsProcessed();
                }
            },

            fullReposition: function () {
            },

            _addPanelSoruceControls: function () {
                if (this.mvc.model.isPropertyExists(Enums.ParameterRoles.PANEL_DATA_SOURCE)) {
                    //remove all child controls - we will create new by array
                    this.removeChildControls();
                    var parentWindowObject = contextResolver.getParentWindowControl(this);
                    var parentWindowCH = parentWindowObject.model.getConnectionHandler();
                    var containerPath = this.buildPath().replace(/\//g, '\\/');
                    var pdsPath = containerPath + Enums.ParameterRoles.PANEL_DATA_SOURCE;
                    this.pdsRegexp = new RegExp(pdsPath, 'i');
                    this.pdsNameRegexp = new RegExp(Enums.ParameterRoles.PANEL_DATA_SOURCE);

                    var childConns,
                        childModelRegx;

                    this.mvc.model.childModels.forEach(function (childModel) {
                        childModelRegx = new RegExp(containerPath + childModel.parameters.Id.value, 'i');
                        childConns = this.childConnections[childModel.parameters.Id.value] = {};

                        childConns.fromChild = parentWindowCH.getConnections(childModelRegx, undefined, Enums.connectionSearchType.source);
                        childConns.toChild = parentWindowCH.getConnections(undefined, childModelRegx, Enums.connectionSearchType.target);
                        parentWindowCH.removeConnections(childConns.fromChild);
                        parentWindowCH.removeConnections(childConns.toChild);
                    }, this);

                    this.onPanelDataSourceChanged(true);
                }
            },


            onPanelDataSourceChanged: function (force) {
                if (!(this.initialized === true || force === true)) {
                    return;
                }
                var pds = this.mvc.model.get(Enums.ParameterRoles.PANEL_DATA_SOURCE);
                var pdsArr = JSON.parse(JSON.stringify(pds));
                if (!(pds instanceof Array)) {
                    this.structure = $.extend({}, pds);
                    return;
                } else {
                    for (var i = 0; i < pdsArr.length; i++) {
                        for (var attrname in pdsArr[i]) {
                            if (attrname == "__type")
                                continue;
                            var value = pdsArr[i][attrname];
                            pdsArr[i][attrname] = $.extend({}, this.structure[attrname]);
                            pdsArr[i][attrname].value = value;
                        }
                    }
                    this.mvc.model.set(Enums.ParameterRoles.PANEL_DATA_SOURCE, pdsArr, true);
                }
                var delta = pdsArr.length - this._getPDSControlsCount();

                if (delta > 0) {
                    this._createControlsByPanelSoruce();
                } else if (delta < 0) {
                    this._deleteControlsByPanelSoruce()
                }
                this.fullReposition();
                var windowContext = contextResolver.getParentWindowControl(this);
                //обновляем только коннекшены дочерних контролов панели    
                var groupI, connI;
                for (groupI = 0; groupI < this.PDSChildGroups.length; groupI++) {
                    for (connI = 0; connI < this.PDSChildGroups[groupI].conns.length; connI++) {
                        if (this.PDSChildGroups[groupI].conns[connI].source.adapterType!="ActionAdapter") //не надо обновлять значения параметров по действиям
                            this.PDSChildGroups[groupI].conns[connI].update(windowContext);
                    }
                }
            },

            _createControlsByPanelSoruce: function () {

                var pdsArr = this.mvc.model.get(Enums.ParameterRoles.PANEL_DATA_SOURCE),
                    childModels = this.mvc.model.childModels,
                    windowContext = contextResolver.getParentWindowControl(this),
                    parentWindowCH = windowContext.model.getConnectionHandler(),
                    i, childModelIndex, pDSchildGroup,
                    currentPDSControlsCount = this._getPDSControlsCount();

                //для каждого элемента (он структура с полями) в PanelDataSource
                for (i = currentPDSControlsCount; i < pdsArr.length; i++) {
                    pDSchildGroup = {
                        controls: [],
                        conns: []
                    };
                    //создаем набор контролов из тех, что положили в контейнер
                    for (childModelIndex = 0; childModelIndex < childModels.length; childModelIndex++) {
                        //т.к меняем только Id, то его запоминаем/восстанавливаем
                        var originalId = childModels[childModelIndex].parameters.Id.value;
                        var newId = originalId + '_' + i;
                        childModels[childModelIndex].parameters.Id.value = newId;

                        pDSchildGroup.controls.push(this.addChildControl(windowContext, childModels[childModelIndex],
                            this.mvc.model.resourceManager));

                        //досоздаем связи элемента		            
                        this.childConnections[originalId].fromChild.forEach(function (childConnection) {
                            var settings = $.extend(true, {}, childConnection.settings);
                            //меняем id элемента в пути 
                            settings.source.path = settings.source.path.replace(originalId, newId);

                            //Если связь на PDS - подменяем индекс
                            if (this.pdsNameRegexp.test(settings.target.propertyPath)) {
                                settings.target.propertyPath = settings.target.propertyPath.replace('PanelDataSource[0]', 'PanelDataSource[' + i + ']');
                            }//если связь на параметр этого же элемента, то подменяем Id
                            else if (settings.target.path.indexOf(originalId) !== -1) {
                                settings.target.path = settings.target.path.replace(originalId, newId);
                            }
                            //иначе таргет где-то снаружи и мы путь не трогаем

                            pDSchildGroup.conns.push(parentWindowCH.add(settings, windowContext));
                        }, this);


                        this.childConnections[originalId].toChild.forEach(function (childConnection) {
                            var settings = $.extend(true, {}, childConnection.settings);
                            //меняем id элемента в пути 
                            settings.target.path = settings.target.path.replace(originalId, newId);

                            var p = Utilites.buildFullPath(settings.source);
                            //Если связь c PD
                            if (this.pdsRegexp.test(p)) {
                                settings.source.propertyPath = settings.source.propertyPath.replace('PanelDataSource', 'PanelDataSource[' + i + ']');
                            }//если связь с параметра этого же элемента, то подменяем Id
                            else if (settings.source.path.indexOf(originalId) !== -1) {
                                settings.source.path = settings.source.path.replace(originalId, newId);
                            }
                            //иначе таргет где-то снаружи и мы путь не трогаем


                            pDSchildGroup.conns.push(parentWindowCH.add(settings, windowContext));
                        }, this);


                        childModels[childModelIndex].parameters.Id.value = originalId;
                    }

                    this.PDSChildGroups.push(pDSchildGroup);
                }
            },

            _deleteControlsByPanelSoruce: function () {
                var i,
                    pdsArr = this.mvc.model.get(Enums.ParameterRoles.PANEL_DATA_SOURCE),
                    parentWindowObject = contextResolver.getParentWindowControl(this),
                    parentWindowCH = parentWindowObject.model.getConnectionHandler(),
                    currentPDSControlsCount = this._getPDSControlsCount();

                //контролов больше чем размер массива - удаляем лишние
                for (i = pdsArr.length; i < currentPDSControlsCount; i++) {
                    //удаляем коннекшены элемента
                    parentWindowCH.removeConnections(this.PDSChildGroups[i].conns);
                    delete this.PDSChildGroups[i].conns;
                    //а затем сам элемент
                    this.PDSChildGroups[i].controls.forEach(function (control) {
                        this.removeChildControl(control);
                    }, this);

                    delete this.PDSChildGroups[i].controls;
                }
                this.PDSChildGroups.splice(pdsArr.length, currentPDSControlsCount - pdsArr.length);
            },

            _getPDSControlsCount: function () {
                //т.к один элемент массива - это кол-во childModels
                return this.orderedControls.length / this.mvc.model.childModels.length;
            },

            onAfterModelInit: function () {
                this._super();
                this.updateTypeConnections();
            },

            updateTypeConnections: function() {
                // после того, как все дети добавлены и модель полностью проинициализирована
                // можно запустить обновление связей.
                this.mvc.model.updateTypeConnections(this.mvc);
            },

            childModelInit: function (event) {
                this.controls[event.target.getId()] = this._currentChild;
            },

            onChildrenAdded: function (event) {

                if (!event.childModels || event.childModels.length === 0) {
                    return;
                }

                this.addChildControls(event.childModels, event.context)
            },

            modelPropertyChangedInternal: function (event) {

                this._super(event);

                switch (event.property) {
                    case Enums.ParameterRoles.PANEL_DATA_SOURCE:
                        this.onPanelDataSourceChanged(event);
                        break;
                }
            },

            childModelPropertyChanged: function (event) {
                if (event.property) {
                    this.childModelPropertyChangedInternal(event);
                }
            },

            childModelPropertyChanging: function (event) {
                //do something in descendants
            },

            childModelPropertyChangedInternal: function (event) {
                //do something in descendants
            },

            indexOf: function (contol) {
                return this.indexById(control.model.getID());
            },

            childById: function (childId) {
                return this.controls[childId];
            },

            childIndexById: function (childId) {
                var i;

                for (i = 0; i < this.orderedControls.length; i++) {
                    if (this.orderedControls[i].model.getId() === childId) {
                        return i;
                    }
                }

                return -1;
            },

            askAddPermissions: function () {
                this.askPermissions(this.doNothing, this.doNothing, true);
            }
        });

        return ContainerController;
    });
