﻿// <copyright file="ControlController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 

/*
* @class ControlController
* Base class for controllers of controls
*/
define(['common/ParameterRoles', 'common/Enums', 'common/Error', 'common/Utilites', 'common/Appearence', 'base/EventTarget',
        'core/trigger/TriggerHandler', 'core/ActionFactory',
    'core/Event', 'core/Validator', 'helpers/contextResolver', 'helpers/PermissionChecker', 'core/connection/ParameterConnections', 'core/Statistic'],
    function (Roles, Enums, Error, Utilites, Appearence, EventTarget,
        TriggerHandler, ActionFactory, Event, Validator, contextResolver, PermissionChecker,
        ParameterConnections, Statistic) {

        var ACTUAL_WIDTH_CHANGED = 'actual width changed',
            ACTUAL_HEIGHT_CHANGED = 'actual height changed',
            ACTUAL_ENABLED_CHANGED = 'actual enabled changed';


        var ControlController = Class.extend({

            eventTarget: null,
            actions: null,
            triggerHandler: null,
            flashClass: 'flash',

            init: function (statisticId) {
                this.statisticId = statisticId;
                Statistic.addControlCount();
                this.eventTarget = new EventTarget();
                this.events = {};
                this.validator = null;
                this.modelChanged = {};
                this.modelChanged[Roles.WIDTH] = this.onWidthChanged.bind(this);
                this.modelChanged[Roles.WIDTH_UNITS] = this.onWidthUnitsChanged.bind(this);
                this.modelChanged[Roles.HEIGHT] = this.onHeightChanged.bind(this);
                this.modelChanged[Roles.HEIGHT_UNITS] = this.onHeightUnitsChanged.bind(this);
                this.modelChanged[Roles.Z] = this.onZChanged.bind(this);
                this.modelChanged[Roles.SKIN] = this.onSkinChanged.bind(this);
                this.modelChanged[Roles.ANGLE] = this.onAngleChanged.bind(this);
                this.modelChanged[Roles.SCALE_X] = this.onScaleXChanged.bind(this);
                this.modelChanged[Roles.SCALE_Y] = this.onScaleYChanged.bind(this);
                this.modelChanged[Roles.IS_ENABLED] = this.onIsEnabledChanged.bind(this);
                this.modelChanged[Roles.IS_VISIBLE] = this.onIsVisibleChanged.bind(this);
                this.modelChanged[Roles.FLASH] = this.onFlashChanged.bind(this);
                this.modelChanged[Roles.OPACITY] = this.onOpacityChanged.bind(this);
                this.modelChanged[Roles.SHADOW_SIZE] = this.onShadowSizeChanged.bind(this);
                this.modelChanged[Roles.SHADOW_COLOR] = this.onShadowColorChanged.bind(this);
                this.modelChanged[Roles.BACKGROUND_COLOR] = this.onBackgroundColorChanged.bind(this);
                this.modelChanged[Roles.BACKGROUND_TILE] = this.onBackgroundTileChanged.bind(this);
                this.modelChanged[Roles.RESOURCE] = this.onResourceChanged.bind(this);
                this.modelChanged[Roles.BORDER_COLOR] = this.applyBorder.bind(this);
                this.modelChanged[Roles.BORDER_STYLE] = this.applyBorder.bind(this);
                this.modelChanged[Roles.BORDER_THIKNESS] = this.applyBorder.bind(this);
                this.modelChanged[Roles.CORNER_RADIUS] = this.onCornerRadiusChanged.bind(this);
                this.modelChanged[Roles.THEME] = this.onThemeChanged.bind(this);
                this.modelChanged[Roles.ID] = this.renderId.bind(this);
                this.modelChanged[Roles.TAB_INDEX] = this.onTabIndexChanged.bind(this);
            },

            //
            //agility events handlers
            //

            /**
             *  @method Fired upon object creation         
             */
            onCreate: function () {
                this.mvc.model.subscribePropertyChanged(this.modelPropertyChanged, this);

                mergeActions.call(this, this.mvc.model.actions);
                this.triggerHandler = new TriggerHandler(this.actions);
                this.triggerHandler.add(this.mvc.model.triggers, this);
                this.mvc.model.subscribePropertyChanging(this.onPropertyChanging, this);
                this.mvc.model.subscribeOnInit(this.onModelInit, this);
                this.mvc.model.subscribeOnAfterInit(this.onAfterModelInit, this);
                this.onRender();
            },


            onBeforeDestroy: function () {
            },

            /**
             *  @method Fired before object is destroyed.
             */
            onDestroy: function () {
                Window.num--;
                var eventActions;
                for (var eventType in this.events) {
                    eventActions = this.events[eventType].getActions();
                    eventActions.forEach(function (eventAction) {
                        this.actions[eventAction].destroy();
                    }, this);
                }
                this.mvc.model.eventTarget.removeAllListeners();
                this.mvc.view.$().off();
                this.mvc.view.$().empty();
                this.mvc.view.$().remove();
                this.mvc._controller = null;
                this.mvc._container = null;
            },
            getParentObject: function(){
                var parent=this.mvc._parent;
                while (!parent.model.objectId && parent && this._parent) {
                    parent = parent._parent;
                }
                return parent.model.objectId;
            },

            buildPath: function () {
                var parents = [this.mvc.model.getId()],
                    parent = this.mvc._parent;

                if (parent === null || (this.mvc.model.ClassName === "WindowModel" && parent === null)) {
                    return '/';
                }

                while (parent !== null && parent.model.getId !== undefined && parent.model.ClassName != "WindowModel") {
                    parents.push(parent.model.getId());
                    parent = parent._parent;
                }

                return '/' + parents.reverse().join('/') + '/';
            },

            buildFullPath: function () {
                var parents = [this.mvc.model.getId()],
                    parent = this.mvc._parent;

                if (parent === null) {
                    return '/';
                }

                while (parent !== null && parent.model.getId !== undefined && parent.model.ClassName != "SchemeModel") {
                    parents.push(parent.model.getId());
                    parent = parent._parent;
                }

                return '/' + parents.reverse().join('/') + '/';
            },

            /**
             *  @method Fired when a new Agility object is added to the object's container.
             */
            onChildAdded: function (child) {

            },

            /**
             *  @method Fired with an Agility object is removed from the object's container.
             */
            onChildRemoved: function (child) {

            },

            /*
            * @method Fired when control added to parent container
            */
            onAddedToParent: function () {

                //хитрый ход: запускаем onAddedToDOM child-элементов, 
                //только когда корневое окно appendChild к Scheme,
                //а контейнеры уже запустят onAddedToDOM  у своих child
                //т.е onAddedToParent - снизу-вверх, onAddedToDOM - сверху-вниз
                var parents = this.mvc.parent().view.$().parents('html');
                if (parents.length > 0) {
                    this.mvc.controller.onOneTimeConnectionsProcess();
                    this.mvc.controller.onAddedToDOM();
                }
                var parentController = this.mvc.parent().controller;
                if (parentController) {
                    //subscribe on parent events here

                    if (parentController.subscribeActualEnabledChanged) {
                        parentController.subscribeActualEnabledChanged(
                            this.onParentActualEnabledChanged, this);
                    }

                    if (parentController.subscribeActualWidthChanged && this.widthType!=Enums.SizeType.Pixel) {
                        parentController.subscribeActualWidthChanged(
                            this.onParentActualWidthChanged, this);
                    }

                    if (parentController.subscribeActualHeightChanged && this.heightType!=Enums.SizeType.Pixel) {
                        parentController.subscribeActualHeightChanged(
                            this.onParentActualHeightChanged, this);
                    }
                }

                this.applyWidth();
                this.applyHeight();
            },

            /*
            * @method Fired just before control removed from its container
            */
            onRemovingFromParent: function () {
                var parentController = this.mvc.parent().controller;

                //unsubscribe from parent events here
                if (parentController) {

                    if (parentController.unsubscribeActualEnabledChanged) {
                        parentController.unsubscribeActualEnabledChanged(
                            this.onParentActualEnabledChanged, this);
                    }

                    if (parentController.unsubscribeActualWidthChanged) {
                        parentController.unsubscribeActualWidthChanged(
                            this.onParentActualWidthChanged);
                    }

                    if (parentController.unsubscribeActualHeightChanged) {
                        parentController.unsubscribeActualHeightChanged(
                            this.onParentActualHeightChanged);
                    }
                }
            },

            onAddedToDOM: function () {
                if (this.mvc.model.isPropertyExists(Roles.TOOL_TIP))
                {
                    try {
                        this.mvc.view.$().attr('title', "");
                        this.mvc.view.$().tooltip({
                            track: true,
                            content: ""
                        });
                        this.modelChanged[Roles.TOOL_TIP] = this.onToolTipChanged.bind(this);
                        this.onToolTipChanged(this.mvc.model.get(Roles.TOOL_TIP));
                    } catch (err) {

                    }
                }
                if (this.mvc.model.get(Roles.OPACITY) == 0) {
                    this.mvc.view.$().css('pointer-events', 'none');
                }
                var ctx = this.mvc;
                _.forOwn(this.actions, function (action) {
                    action.subscribePropertyChanged(function (ControlController, action) {
                        return function (event) {
                            // ControlController.buildPath()
                            var path = "/" + action.id + '/' + event.property;
                            ctx.model.overrideConnectionsOperation(path, action.constants.operation);
                            ctx.model.updateTypeConnection(path, ctx);
                            ctx.model.restoreConnectionsOperation(action.constants.operation);
                        }
                    }(this, action));
                });

                _.forOwn(this.events, function (ev) {
                    ev.subscribePropertyChanged(function (ControlController, ev) {
                        return function (event) {
                            // ControlController.buildPath()
                            var path = "/" + ev.type + '/' + event.property;
                            ctx.model.updateTypeConnection(path, ctx);
                        }
                    }(this, ev));
                });
                if (this.mvc.model.isPropertyExists(Enums.ParameterRoles.SKIN) && this.mvc.model.get(Enums.ParameterRoles.SKIN)) {
                    this.applySkin();
                }
                else {
                    //применяем триггеры, когда тип и модель обработаны + 
                    //построены и отработали все связи
                    this.triggerHandler.init(this);
                    //    this.triggerHandler.applyAll();
                }

                this.askAddPermissions(this.doNothing, this.mvc.model.setIsEnabled.bind(this.mvc.model, false), true);

            },

            applySkin: function () {
                var value = this.mvc.model.get(Enums.ParameterRoles.SKIN);
                var resourceUrl;
                if (value) {
                    resourceUrl = this.mvc.model.resourceManager.resolve(value);
                } else {
                    return;
                }
                var embed = document.createElement('embed');
                embed.setAttribute("type", "image/svg+xml");
                embed.setAttribute("src", resourceUrl);
                var attrs = this.mvc.view.$()[0].attributes;
                for (var i = 0; i < attrs.length; i++) {
                    embed.setAttribute(attrs[i].name, attrs[i].value);
                }
                embed.style.border = '';
                embed.style.background = '';
                this.mvc.view.$().replaceWith(embed);
                this.mvc.view.$root = $(embed);
                this.modelChanged[Enums.ParameterRoles.BORDER_COLOR] = this.onSvgModelChanged.bind(this, Enums.ParameterRoles.BORDER_COLOR + ":values", Appearence.color.color2Matrix);
                this.modelChanged[Enums.ParameterRoles.BORDER_STYLE] = this.onSvgBorderChanged.bind(this);
                this.modelChanged[Enums.ParameterRoles.BORDER_THIKNESS] = this.onSvgBorderChanged.bind(this);
                this.modelChanged[Enums.ParameterRoles.BACKGROUND_COLOR] = this.onSvgModelChanged.bind(this, Enums.ParameterRoles.BACKGROUND_COLOR + ":values", Appearence.color.color2Matrix);
                this.modelChanged[Enums.ParameterRoles.IS_ENABLED] = this.onSvgIsEnabledChanged.bind(this);
                //загрузка
                this.loadSVG();

            },
            loadSVG: function() {
                var embed = this.mvc.view.$()[0];
                try {
                    embed.getSVGDocument();
                } catch (ex) {
                    console.error(ex.message || ex);
                }
                embed.addEventListener('load', function () {
                    this.svgRoot = this._getSvgRoot(embed);
                    if (this.svgRoot) { //бывает возвращается null
                        this.initSvgModelChangedMethods();
                        this.addSvgEventListener.call(this, this.svgRoot);
                    } else {
                        Error.warn("svg DataControlId = " + this.mvc.model.getId() + " getSVGDocument=null");
                    }
                }.bind(this), { capture: false, once: true });
            },

            initSvgModelChangedMethods: function () {
                //у образа нет особенных параметров модели    
            },

            onSvgIsEnabledChanged: function(){
                if (!this.actualEnabled()) {
                    this.mvc.view.$().attr('disabled', 'true');
                    this.mvc.view.$().addClass('disabled');
                } else {
                    this.mvc.view.$().removeAttr('disabled');
                    this.mvc.view.$().removeClass('disabled');
                }
            },

            onSvgModelChanged: function (property, converter, value) {
                var newvalue;
                if (converter) {
                    newvalue = converter(value);
                }
                else {
                    newvalue = value;
                }
                this.onCustomParameterChanged("svg:"+property, newvalue);
            },

            onCustomParameterChanged: function(path, value) {
                this._setSvgParameter(path, value);
            },
            getSVGElementById: function(id){
                var element = this.mvc.view.$()[0].getSVGDocument().getElementById(id);
                if (!element) {
                    Error.warn(String.format("У контрола {0} {1} не найден элемент с id {2}", this.mvc.model.getRole(), this.mvc.model.getId(), id));
                }
                return element;

            },
            getSVGAttributeFromElement: function(element, attribute){
                if (element) {
                    var value = element.getAttribute(attribute);
                    if (!value) {
                        Error.warn(String.format("У контрола {0} {1} в svg параметре Образ не найден атрибут {2}", this.mvc.model.getRole(), this.mvc.model.getId(), attribute));
                    }
                    return value;
                }
            },
            _setSvgParameter: function (path, value) {
                if (this.svgRoot === null) {
                    return;
                }

                var tokenizedPath = path.split(':');
                var selector = this._getSelector(tokenizedPath[1]);

                if (tokenizedPath[2] === 'fill') {
                    value = Appearence.color.toCssColor(value);
                }
                if (tokenizedPath[2] === 'values') {
                    if (!Appearence.color.isColorMatrix(value)) value = Appearence.color.color2Matrix(value);
                }
                if (tokenizedPath[2] === 'Content') {
                    if (Utilites.isWebkit()) {
                        $(selector, this.svgRoot)[0].textContent = value;
                    } else {
                        $(selector, this.svgRoot).html(value);
                    }

                } else {
                    this._getSetParameterMethod(tokenizedPath[3]).call($(selector, this.svgRoot), tokenizedPath[2], value);
                }
                if (Utilites.isWebkit()) {
                    this.onResize();
                }
            },
            onResize: function () {
                if (!window.initialized)
                    return;
                var w = this.mvc.view.$()[0].style.width;
                this.mvc.view.$().width(0);
                this.mvc.view.$()[0].style.width = w;
            },

            _getSelector: function (name) {
                if (this._isClassParameter(name)) {
                    return name;
                } else {
                    return '#' + name;
                }
            },
            _getSetParameterMethod: function (name) {
                if (name === null || name === undefined || name !== 'style') {
                    return $.fn.attr;
                } else {
                    return $.fn.css;
                }
            },

            _isClassParameter: function (name) {
                return name.indexOf('.') === 0;
            },

            _getSvgRoot: function () {
                try {
                    if (this.mvc.view.$()[0].getSVGDocument() !== null && this.mvc.view.$()[0].getSVGDocument().readyState === 'complete') {
                        return this.mvc.view.$()[0].getSVGDocument().documentElement;
                    } else {
                        return null;
                    }
                } catch (e) {
                    Error.debug(String.format('Reason: {0} \n Stack trace: \n {1}', e.message || e, e.stack));
                    return null;
                }
            },

            addSvgEventListener: function (embed) {
                var elem;
                if (embed.getElementById("body")) {
                    elem = $(embed.getElementById("body"));
                } else {
                    elem = $(embed.querySelector('svg'));
                }
                this.addEvents(elem);
                this.attachEvents(elem);
            },

            onModelInit: function () {
                this.validator = new Validator(this.mvc.model.getId());
                mergeActions.call(this, this.mvc.model.actions);
                //когда все события собраны и все акшены смержены - создаем события
                this.addEvents(this.mvc.view.$());
                this.attachEvents(this.mvc.view.$());
                this.triggerHandler.add(this.mvc.model.triggers, this);
            },

            addEvents: function (elem) {
                /*if (_.isEmpty(this.mvc.model.events)) {
                    this.mvc.view.$().css('pointer-events', 'none');
                    return;
                }*/
                if (!elem)
                    elem = this.mvc.view.$();
                var ctx = this;
                _.forOwn(this.mvc.model.events, function (event, eventType) {
                    ctx.events[eventType] = new Event(event); //надо привести названия событий к стандартным названиям событий jQuery
                    if (eventType == Enums.EventType.doubleClick)
                        elem.dblclick(ctx.handleEvent.bind(ctx, eventType));
                    else if (eventType==Enums.EventType.lostFocus)
                        elem.focusout(ctx.handleEvent.bind(ctx, eventType));
                    else
                        elem.on(_.toLower(eventType), ctx.handleEvent.bind(ctx, eventType));
                });

            },

            onAfterModelInit: function () {
                this.renderBorder();
            },

            onOneTimeConnectionsProcess: function () {

            },

            addConnectionsForParameter: function (parameterName) {
                return ParameterConnections.process(parameterName, this.mvc.model.getProperty(parameterName), this.mvc);
            },

            getServerVarsFromConnectionsSettings: function (connectionSettings, role) {
                var serverVars = {
                    getValues: [],
                    setValues: []
                };
                var windowServerVariables = contextResolver.getRootWindow(this).model.serverVariables.getValues
                    .concat(contextResolver.getParentWindowControl(this).model.serverVariables.getValues); //переменная может быть либо в основном окне, либо в родительском
                connectionSettings.forEach(function (connSetting) {
                        if (connSetting.source.type == Enums.connectionType.server &&
                            connSetting.target.propertyPath.indexOf(role) != -1) {
                            for (var i = 0; i < windowServerVariables.length; i++) {
                                if (windowServerVariables[i].dataSourceId == connSetting.source.dataSource &&
                                    (windowServerVariables[i].itemId == connSetting.source.itemId || connSetting.source.itemId == 0) &&
                                    windowServerVariables[i].path == connSetting.source.propertyPath) {
                                    serverVars.getValues.push($.extend(true, {}, windowServerVariables[i]));
                                    break;
                                }
                            }
                        }
                    },
                    this);

                return serverVars;
            },
            //
            // rendering
            //

            onRender: function () {

                this.mvc.view.$().attr('data-control-role', this.mvc.model.getRole());
                this.renderId();
                this.renderBorder();
                this.renderOverflow();
                this.renderPosition();
            },

            renderId: function () {
                this.mvc.view.$().attr('data-control-id', this.mvc.model.getId());
            },

            renderBorder: function () {
                var borderWidth = this.mvc.model.get(Roles.BORDER_THIKNESS);

                if (borderWidth) {
                    Appearence.border.applyBoxSizing(this.mvc.view.$(), 'border-box');
                }
            },

            renderOverflow: function () {
                this.mvc.view.$().css('overflow', 'hidden');
            },

            renderPosition: function () {
                this.mvc.view.$().css('position', 'absolute');
            },

            onParentActualEnabledChanged: function (newValue) {
                if (!window.initialized)
                    return;
                this.applyIsEnabled();
                if (this.mvc.model.getIsEnabled()) {
                    this.fireActualEnabledChanged(newValue);
                }
            },

            onParentActualWidthChanged: function () {
                if (!window.initialized)
                    return;
                if (Utilites.isPercents(this.mvc.view.$()[0].style.width)) {
                    this.fireActualWidthChanged();
                }
            },

            onParentActualHeightChanged: function () {
                if (!window.initialized)
                    return;
                if (Utilites.isPercents(this.mvc.view.$()[0].style.height)) {
                    this.fireActualHeightChanged();
                }
            },

            clientId: function () {
                var p = this.mvc.parent();

                var clientId = this.mvc.model.getId();
                if (this.mvc.model.objectId !== undefined)
                    clientId = clientId + '_' + this.mvc.model.objectId;

                if (p && p.controller && typeof p.controller.clientId == 'function') {
                    clientId = p.controller.clientId() + '_' + clientId;
                }

                return clientId;
            },

            getVariablesSource: function (id) {
                return this.variablesSource.getVMById(id);
            },
            getControlLeftPx: function () {
                var value = this.mvc.view.$().css('left');
                if (value === 'auto') {
                    this.mvc.view.$()[0].offsetLeft;
                }
                else {
                    return parseInt(value, 10);
                }
            },

            setControlLeft: function (value) {
                var oldValue = this.mvc.view.$().css('left');

                if (value !== oldValue) {
                    this.mvc.view.$().css('left', value);
                }
            },

            getControlTopPx: function () {
                return parseInt(this.mvc.view.$().css('top'), 10);
            },

            setControlTop: function (value) {
                var oldValue = this.mvc.view.$().css('top');

                if (value !== oldValue) {
                    this.mvc.view.$().css('top', value);
                }
            },

            setControlRight: function (value) {
                var oldValue = this.mvc.view.$().css('right');

                if (value !== oldValue) {
                    this.mvc.view.$().css('right', value);
                }
            },

            getControlBottomPx: function () {
                return parseInt(this.mvc.view.$().css('bottom'), 10);
            },

            setControlBottom: function (value) {
                var oldValue = this.mvc.view.$().css('bottom');

                if (value !== oldValue) {
                    this.mvc.view.$().css('bottom', value);
                }
            },

            setControlWidth: function (value) {
                var oldValue = this.mvc.view.$().css('width');
                if (value == "0%") value = 0;
                if (value !== oldValue) {
                    this.mvc.view.$().css('width', value);
                    this.fireActualWidthChanged();
                }
            },

            setControlHeight: function (value) {
                var oldValue = this.mvc.view.$().css('height');

                if (value !== oldValue) {
                    this.mvc.view.$().css('height', value);
                    this.fireActualHeightChanged();
                }
            },

            getControlWidthPx: function () {
                return this.mvc.view.$().outerWidth();
            },

            getControlInnerWidth: function () {
                return this.mvc.view.$().innerWidth();
            },

            getControlHeightPx: function () {
                return this.mvc.view.$().outerHeight();
            },

            getControlInnerHeight: function () {
                return this.mvc.view.$().innerHeight();
            },


            // TODO Check anchors, logc has been changed
            setLeftAnchor: function (anchor) {

                var oldLeft = this.mvc.view.$().css('left');
                var oldRight = this.mvc.view.$().css('right');

                if (oldLeft !== anchor || oldRight !== 'auto') {
                    this.mvc.view.$().css({ left: anchor, right: 'auto' });
                }
            },

            setRightAnchor: function (anchor) {
                var oldLeft = this.mvc.view.$().css('left');
                var oldRight = this.mvc.view.$().css('right');

                if (oldLeft !== 'auto' || oldRight !== anchor) {

                    this.mvc.view.$().css({ left: 'auto', right: anchor });
                }
            },

            setTopAnchor: function (anchor) {
                var oldTop = this.mvc.view.$().css('top');
                var oldBottom = this.mvc.view.$().css('bottom');

                if (oldTop !== anchor || oldBottom !== 'auto') {
                    this.mvc.view.$().css({ top: anchor, bottom: 'auto' });
                }
            },

            setBottomAnchor: function (anchor) {
                var oldTop = this.mvc.view.$().css('top');
                var oldBottom = this.mvc.view.$().css('bottom');

                if (oldTop !== 'auto' || oldBottom !== anchor) {
                    this.mvc.view.$().css({ top: 'auto', bottom: anchor });
                }
            },

            actualVisible: function () {
                var visible = this.mvc.model.getIsVisible();

                if (!visible) {
                    return false;
                }

                return this.actualParentVisible();
            },

            actualParentVisible: function () {
                var p = this.mvc.parent();
                if (!p) {
                    //control is not on page
                    return false;
                }

                var parentController = p.controller;
                if (!parentController || !parentController.actualVisible) {
                    //may be parent is not regular control, so assume it is visible
                    return true;
                }

                return parentController.actualVisible();
            },

            actualEnabled: function () {
                var enabled = this.mvc.model.getIsEnabled();

                if (enabled === false) {
                    return false;
                }

                return this.actualParentEnabled();
            },

            actualParentEnabled: function () {
                var p = this.mvc.parent();
                if (!p) {
                    //control is not on page
                    return true;
                }

                var parentController = p.controller;
                if (!parentController || !parentController.actualEnabled) {
                    //may be parent is not regular control, so assume it is enabled
                    return true;
                }

                return parentController.actualEnabled();
            },

            applyShadow: function () {

                var shadow = {
                    size: this.mvc.model.get(Roles.SHADOW_SIZE),
                    color: this.getMixedShadowColor(
                        this.mvc.model.getShadowColor(),
                        this.mvc.model.getBackgroundColor())
                };

                Appearence.shadow.applyBoxShadow(this.mvc.view.$(), shadow);
            },


            getMixedShadowColor: function (shadowColor, bgBrush) {
                if (shadowColor == undefined) shadowColor = this.mvc.model.getShadowColor();
                if (this._isARGB(shadowColor) && this._isARGB(bgBrush)) {
                    var shadowObj = Appearence.color.ARGBtoObj(shadowColor);
                    var bgBrushObj = Appearence.color.ARGBtoObj(bgBrush);
                    if (shadowObj && bgBrushObj) {
                        if (shadowObj.a > bgBrushObj.a)
                            shadowObj.a = bgBrushObj.a;
                        return Appearence.color.objToARGB(shadowObj);
                    }
                }
                return Appearence.color.toCssColor(shadowColor);
            },

            _isARGB: function (color) {
                var brushType = Appearence.background.getBrushType(color);
                switch (brushType) {
                case Enums.BrushType.RADIAL_GRADIENT:
                case Enums.BrushType.LINEAR_GRADIENT:
                    return false;
                default:
                    return true;
                }
            },

            applyWidth: function () {
                //abs is for line support
                var width = Math.abs(this.mvc.model.getWidth()),
                    widthUnits = this.mvc.model.getWidthUnits();

                var displayValue;
                if (this.mvc.parent() && this.mvc.parent().controller) {
                    if (widthUnits == Enums.SizeType.Relative) {
                        var parentWidth = Math.abs(this.mvc.parent().model.getWidth()) || 1;
                        displayValue = (100 * width / parentWidth);
                    }
                    else {
                        displayValue = width;
                    }
                } else {
                    displayValue = width;
                    widthUnits = Enums.SizeType.Pixel;
                }
                this.width = displayValue;
                this.setControlWidth(displayValue + Utilites.getMeasureSuffix(widthUnits));
            },

            applyHeight: function () {

                //abs is for line support
                var height = Math.abs(this.mvc.model.getHeight());
                var heightUnits = this.mvc.model.getHeightUnits();

                var displayValue;
                if (this.mvc.parent() && this.mvc.parent().controller) {
                    if (heightUnits == Enums.SizeType.Relative) {//means that parent is not document
                        var parentHeight = Math.abs(this.mvc.parent().model.getHeight()) || 1;
                        displayValue = (100 * height / parentHeight);
                    }
                    else {
                        displayValue = height;
                    }
                } else {
                    displayValue = height;
                    heightUnits = Enums.SizeType.Pixel;
                }
                this.setControlHeight(displayValue + Utilites.getMeasureSuffix(heightUnits));
            },

            applyBorder: function () {
                this.setupBorderTo(this.mvc.view.$());
            },

            applyIsEnabled: function () {
                if (!this.actualEnabled()) {
                    this.mvc.view.$().attr('disabled', 'true');
                    this.mvc.view.$().addClass('disabled');
                    this.mvc.view.$().css('pointer-events', 'none');

                } else {
                    this.mvc.view.$().removeAttr('disabled');
                    this.mvc.view.$().removeClass('disabled');
                    this.mvc.view.$().css('pointer-events', '');
                }
            },

            applyFlash: function () {
                var isFlashEnabled = this.mvc.model.get(Roles.FLASH);
                if (isFlashEnabled) {
                    this.mvc.view.$().addClass(this.flashClass);
                }
                else {
                    this.mvc.view.$().removeClass(this.flashClass);
                }
            },

            /*
            * @method setupBorder used to setup border for element (exactly same as for control)
            * @param el {jQuery} target element(s)
            */
            setupBorderTo: function (el) {
                var border = {
                    penType: this.mvc.model.get(Roles.BORDER_STYLE),
                    size: this.mvc.model.get(Roles.BORDER_THIKNESS),
                    color: this.mvc.model.get(Roles.BORDER_COLOR)
                };

                Appearence.border.applyBorder(el, border);
            },

            find: function (path) {
                var children = this.mvc._container.children;
                for (var i in children) {
                    if (children[i].model.getId() === path) {
                        return children[i];
                    }
                }
            },

            findChilds: function (ctx, className, matches) {
                var that = this;
                if (ctx.model.ClassName == className) {
                    matches.push(ctx.model);
                    return;
                }
                if (ctx.controller.ClassName == className) {
                    matches.push(ctx.controller);
                    return;
                }
                if (ctx.controller.controls) {
                    $.each(ctx.controller.controls, function (key, value) {
                        that.findChilds(value, className, matches)
                        return;
                    });
                    return matches;
                }
            },

            //
            //property changed handlers
            //
            modelPropertyChanged: function (event) {
                if (event.property) {
                    if ((event.newValue) && ((event.newValue.value == null) || (event.newValue.value == NaN) || (event.newValue.value == undefined))) {
                        Error.info(String.format('{0} {1} параметру {2} пришло значение {3} от параметра {4}', this.mvc.model.getRole(), this.mvc.model.getId(), event.property, String(event.newValue.value), event.newValue.itemId));
                    }
                    this.modelPropertyChangedInternal(event);
                }
            },

            modelPropertyChangedInternal: function (event) {
                // обновляем связи типа, если изменились значения свойств экземпляра.
                // Связи экземпляра обновит тип, в котором этот экземпляр находится.
                // Поэтому путь к свойству строится без учета родителей
                var path = '/' + event.property;
                this.mvc.model.getType()._connectionsHandler.onchange(path, this.mvc, event.oldValue);
                // TODO сделать из этого Object, чтобы не перебирать switch'ем. Все-таки время доступа к свойству в объекте О(1)
                var value = this.mvc.model.get(event.property);
                if (this.modelChanged[event.property]) {
                    this.modelChanged[event.property](value);
                }
            },

            onWidthChanged: function (value) {
                this.applyWidth();
            },

            onWidthUnitsChanged: function (value) {
                this.applyWidth();
            },

            onHeightChanged: function (value) {
                this.applyHeight();
            },

            onHeightUnitsChanged: function (value) {
                this.applyHeight();
            },

            onZChanged: function (value) {
                this.mvc.view.$().css('z-index', value);
            },

            onAngleChanged: function (value) {
                var scale = {
                    scaleX: this.mvc.model.get(Roles.SCALE_X),
                    scaleY: this.mvc.model.get(Roles.SCALE_Y),
                    angle: value
                };

                Appearence.position.scale(this.mvc.view.$(), scale);
            },

            onScaleXChanged: function (value) {

                var scale = {
                    scaleX: value,
                    scaleY: this.mvc.model.get(Roles.SCALE_Y),
                    angle: this.mvc.model.get(Roles.ANGLE)
                };

                Appearence.position.scale(this.mvc.view.$(), scale);
            },

            onScaleYChanged: function (value) {

                var scale = {
                    scaleX: this.mvc.model.get(Roles.SCALE_X),
                    scaleY: value,
                    angle: this.mvc.model.get(Roles.ANGLE)
                };

                Appearence.position.scale(this.mvc.view.$(), scale);
            },

            onTabIndexChanged: function (value) {
                //from ParameterProcessor: TabIndex cannot begin with zero, so add one
                if (value === 0) {
                    value++;
                }
                this.mvc.view.$().attr('tabindex', value);
            },

            onToolTipChanged: function (value) {
                this.mvc.view.$().tooltip("option", "content", value);
            },

            onIsVisibleChanged: function (value) {
                if (value) {
                    this.mvc.view.$().css('visibility', '');
                }
                else {
                    this.mvc.view.$().css('visibility', 'hidden');
                }
            },

            onIsEnabledChanged: function (value) {
                this.applyIsEnabled();
                this.fireActualEnabledChanged(this.actualEnabled());
            },

            onFlashChanged: function (value) {
                this.applyFlash();
            },

            onOpacityChanged: function (value) {
                if (value == 0) {
                    this.mvc.view.$().css('pointer-events', 'none');
                }
                else {
                    this.mvc.view.$().css('pointer-events', '');
                }
                this.mvc.view.$().css('opacity', value / 100);
            },

            onShadowSizeChanged: function (value) {
                this.applyShadow();
            },

            onShadowColorChanged: function (value) {
                this.applyShadow();
            },

            onBackgroundColorChanged: function (value) {
                this.onShadowColorChanged();
                this.onShadowSizeChanged();
                Appearence.background.apply(this.mvc.view.$(), value);
            },

            onBackgroundTileChanged: function (value) {
                Appearence.background.applyBackgroundTile(this.mvc.view.$(), value);
            },

            onSkinChanged: function (value) {
                //динамизации образа нет 12965
            },

            onResourceChanged: function (value) {
                if (value) {
                    var resourceUrl = this.mvc.model.resourceManager.resolve(value);
                    //asuming that value is url to resource
                    Appearence.background.applyBgImage(this.mvc.view.$(), resourceUrl);
                }
            },

            onSvgBorderChanged: function (value){ //будем пока что считать что у svg нет границ
                return;
            },

            onCornerRadiusChanged: function (value) {
                // Нам реально надо в половину меньше, чем пришло из редактора.
                // Связано с особенностями отображения в WPF и HTML.
                value = value / 2;
                Appearence.border.applyCornerRadius(this.mvc.view.$(), { radius: value, units: '%' });
            },

            onThemeChanged: function (value) {
                //TODO:
            },

            //
            // Events
            //

            attachEvents: function (elem) {
                var ctx = this;
                elem.contextmenu(function (event) { return ctx.onContextMenu(event); });
            },

            onContextMenu: function (event) {
                event.stopPropagation();
                event.preventDefault();
                this.handleEvent(Enums.EventType.contextMenu, event);
                return true;
            },

            handleEvent: function (eventType, event, force) {
            //    event.stopPropagation();
            //    event.preventDefault();
                if (!(this.actualEnabled() || force === true)) {
                    return Promise.resolve();
                }
                if (this.events[eventType] === undefined) {
                    return Promise.resolve();
                }
                if (!force && this.events[eventType].getIsTriggerOnlyEvent() == false) {
                    //если событие вызвало действие пользователя, 
                    //а не внешний вызов (CallEventAction)
                    //и оно не только, т.е есть параметры
                    this.events[eventType].updateParameters(event, this);
                }
                PermissionChecker.setCurrentEvent(eventType);
                this.askEventPermissions(eventType);

                //TODO event.stopPropagation();
                return Promise.resolve();
            },
            askEventPermissions: function(eventType) {
                var eventActions = this.events[eventType].getActions();
                if (eventActions.length) {
                    this.askPermissions(this.doActions.bind(this,eventActions), this.doNothing, false);
                }
            },
            askConfirmation: function() {
                this.askPermissions(this.allowAct.bind(this), this.denyAct.bind(this), false);
            },

            askAddPermissions: function (function_allow, function_deny, silent) {
                this.askPermissions(function_allow, function_deny, silent);
            },

            askPermissions: function (function_allow, function_deny, silent) {
                var result = PermissionChecker.createResultObj(function_allow, function_deny, silent);
                this.check(result);
            },
            allowAct: function(value) {
                return Promise.resolve();
            },
            denyAct: function() {
                return Promise.resolve();
            },
            doActions: function(eventActions){
                for (var i = 0; i < eventActions.length; i++) {
                    this.actions[eventActions[i]].execute(this);
                }
                return Promise.resolve();
            },
            doNothing: function(){
                return Promise.resolve();
            },
            check: function (result) {
                if (PermissionChecker.needCheck()) {
                    PermissionChecker.check("Control", this, result);
                    
                }else {
                    result.allow_action(); //если пользователей нет, то считается что всё можно
                }
            },
            onPropertyChanging: function (event) {
                this.validateProperty(event.property, event.newProperty);
            },

            validateProperty: function (propertyName, newProperty) {

            },

            callMethod: function (methodName, params) {
                if (this[methodName]) {
                    return this[methodName].apply(this, params);
                }

                return Promise.resolve();
            },

            //
            // custom events

            //----------------------------------------------------------------------
            subscribeActualEnabledChanged: function (handler, context) {
                this.eventTarget.addListener(ACTUAL_ENABLED_CHANGED, handler, context);
            },

            unscubscribeActualEnabledChanged: function (handler) {
                this.eventTarget.removeListener(ACTUAL_ENABLED_CHANGED, handler);
            },

            /*
            * @method fireActualEnabledChanged fires event "actual enabled status of control changed"
            * (even if parent enabled was changed)
            */
            fireActualEnabledChanged: function (newValue) {
                this.eventTarget.fire({ type: ACTUAL_ENABLED_CHANGED, target: this, value: newValue });
            },

            //----------------------------------------------------------------------
            subscribeActualWidthChanged: function (handler, context) {
                this.eventTarget.addListener(ACTUAL_WIDTH_CHANGED, handler, context);
            },

            unscubscribeActualWidthChanged: function (handler) {
                this.eventTarget.removeListener(ACTUAL_WIDTH_CHANGED, handler);
            },
            /*
            * @method fireActualWidthChanged fires when with if control was changed
            * (even if control width is specified in % and parent control width changed
            * @param newValue {number} size in pixels
            */
            fireActualWidthChanged: function (newValue) {
                if (!window.initialized && this.widthType == Enums.SizeType.Pixel) {
                    return;
                }
                this.eventTarget.fire({ type: ACTUAL_WIDTH_CHANGED, target: this, value: newValue });
            },
            //----------------------------------------------------------------------
            subscribeActualHeightChanged: function (handler, context) {
                this.eventTarget.addListener(ACTUAL_HEIGHT_CHANGED, handler, context);
            },

            unscubscribeActualHeightChanged: function (handler) {
                this.eventTarget.removeListener(ACTUAL_HEIGHT_CHANGED, handler);
            },

            /*
            * @method fireActualHeightChanged fires when height if control was changed
            * (even if control height is specified in % and parent control height changed
            * @param newValue {number} size in pixels
            */
            fireActualHeightChanged: function (newValue) {
                this.eventTarget.fire({ type: ACTUAL_HEIGHT_CHANGED, target: this, value: newValue });
            },
            /**
             *  Added to let agility know to not do proxy for controller.
             */
            _noProxy: function () { }
        });

        function mergeActions(actions) {
            if (!actions) {
                return;
            }

            this.actions = this.actions || {};
            var i;
            for (i = 0; i < actions.length; i++) {
                var action = actions[i]; //id уникальны
                //действия экземпляра могут только дополнять пришедшие из типа
                if (this.actions[action.id] == undefined) {
                    this.actions[action.id] = ActionFactory.createByType(action);
                }
            }
        }

        return ControlController;
    });
