﻿// <copyright file="StackPanelController.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


/*
* @class StackPanelController controller class for StackPanel control
*/
define(['common/Enums', 'base/ContainerController'],
function (Enums, ContainerController) {

    var StackPanelController = ContainerController.extend({

        init: function () {
            this._super();

            this.ClassName = 'StackPanelController';         
        },

        modelPropertyChangedInternal: function (event) {

            this._super(event);

            switch (event.property) {
                case Enums.ParameterRoles.ORIENTATION:
                    this.onOrientationChanged(event);
                    break;
                case Enums.ParameterRoles.HORIZONTAL_ELEMENTS_ALIGN:
                    this.onHorizontalElementsAlignChanged(event);
                    break;
                case Enums.ParameterRoles.VERTICAL_ELEMENTS_ALIGN:
                    this.onVerticalElementsAlignChanged(event);
                    break;
            }
        },

        onParentActualWidthChanged: function (event) {
            this._super(event);

            this.fullReposition();
        },

        onParentActualHeightChanged: function (event) {
            this._super(event);

            this.fullReposition();
        },

        onBorderThiknessChanged: function (value) {
            this._super();

            this.fullReposition();
        },

        onOrientationChanged: function (event) {
            this.fullReposition();
        },

        onHorizontalElementsAlignChanged: function (event) {

            if ((event.oldValue && event.newValue) && (event.oldValue.value === Enums.HorizontalElementsAlign.Stretch ||
                    event.newValue.value === Enums.HorizontalElementsAlign.Stretch)) {
                this.setupChildrenWidths();
            }
            
            this.repositionControls(-1);
        },

        onVerticalElementsAlignChanged: function (event) {

            if ((event.oldValue && event.newValue) && (event.oldValue.value === Enums.VerticalElementsAlign.Stretch ||
                    event.newValue.value === Enums.VerticalElementsAlign.Stretch)) {
                this.setupChildrenHeights();
            }           

            this.repositionControls(-1);
        },

        childModelPropertyChangedInternal: function (event) {
            this._super();

            switch (event.property) {
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildWidthChanged(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildHeightChanged(event);
                    break;
            }
        },

        childModelPropertyChanging: function (event) {
            switch (event.property) {
                case Enums.ParameterRoles.WIDTH:
                case Enums.ParameterRoles.WIDTH_UNITS:
                    this.onChildChangingWidth(event);
                    break;
                case Enums.ParameterRoles.HEIGHT:
                case Enums.ParameterRoles.HEIGHT_UNITS:
                    this.onChildChangingHeight(event);
                    break;
            }
        },

        onChildChangingWidth: function (event) {            
            var ha = this.mvc.model.getHorizontalElementsAlign();

            if (ha === Enums.HorizontalElementsAlign.Stretch) {
                event.silent = true;
            }
        },

        onChildChangingHeight: function (event) {
            var va = this.mvc.model.getVerticalElementsAlign();            

            if (va === Enums.VerticalElementsAlign.Stretch) {
                event.silent = true;
            }
        },

        onChildWidthChanged: function (event) {
            //if we come here then it is allowed to change width for children
            //so, reposition corresponding controls
            var childIndex = this.childIndexById(event.target.getId());

            this.repositionControls(childIndex);
        },

        onChildHeightChanged: function (event) {
            //if we come here then it is allowed to change height for children
            //so, reposition corresponding controls
            var childIndex = this.childIndexById(event.target.getId());

            this.repositionControls(childIndex);
        },

        fireActualWidthChanged: function (newValue) {
            this._super(newValue);

                if (this.mvc.model.getHorizontalElementsAlign() === Enums.HorizontalElementsAlign.Stretch) {
                this.setupChildrenWidths();
            }

            if (this.mvc.model.getOrientation() === Enums.OrientationType.Horizontal) {
                if (this.mvc.model.getHorizontalElementsAlign() === Enums.HorizontalElementsAlign.Stretch) {
                    this.repositionControls(-1);
                }
            }
            else {
                if (this.mvc.model.getHorizontalElementsAlign() === Enums.HorizontalElementsAlign.Center) {
                    this.repositionControls(-1);
                }
            }
        },

        fireActualHeightChanged: function (newValue) {
            this._super(newValue);

            if (this.mvc.model.getVerticalElementsAlign() === Enums.VerticalElementsAlign.Stretch) {
                this.setupChildrenHeights();
            }

            if (this.mvc.model.getOrientation() === Enums.OrientationType.Vertical) {
                if (this.mvc.model.getVerticalElementsAlign() === Enums.VerticalElementsAlign.Stretch) {
                    this.repositionControls(-1);
                }
            }
            else {
                if (this.mvc.model.getVerticalElementsAlign() === Enums.VerticalElementsAlign.Center) {
                    this.repositionControls(-1);
                }
            }
        },

        fullReposition: function () {
            this.setupChildrenSize();
            this.repositionControls(-1);
        },

        setupChildrenSize: function () {

            if (this.orderedControls.length === 0) {
                return;
            }

            this.setupChildrenHeights();
            this.setupChildrenWidths();
        },

        /*
        * @method setupChildrenHeights sets height for children controls
        * it is necessary because controls can be stretched or in case if panel changes orientation
        */
        setupChildrenHeights: function () {
            var i,
                child,
                childHeight,
                controlInnerHeight = this.calcClientRect().height;

            if (controlInnerHeight <= 0) {
                //optimization
                return;
            }

            if (this.mvc.model.getOrientation() === Enums.OrientationType.Vertical) {
                if (this.mvc.model.getVerticalElementsAlign() === Enums.VerticalElementsAlign.Stretch) {
                    childHeight = controlInnerHeight / this.orderedControls.length;
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.setControlHeight(childHeight + 'px');
                    }
                }
                else {
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.applyHeight();
                    }
                }
            }
            else {
                if (this.mvc.model.getVerticalElementsAlign() === Enums.VerticalElementsAlign.Stretch) {
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.setControlHeight(controlInnerHeight + 'px');
                    }
                }
                else {
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.applyHeight();
                    }
                }
            }
        },

        /*
       * @method setupChildrenWidths sets width for children controls
       * it is necessary because controls can be stretched or in case if panel changes orientation
       */
        setupChildrenWidths: function () {
            var i,
                child,
                childWidth,
                controlInnerWidth = this.calcClientRect().width;

            if (controlInnerWidth <= 0) {
                //optimization
                return;
            }

            if (this.mvc.model.getOrientation() === Enums.OrientationType.Horizontal) {
                if (this.mvc.model.getHorizontalElementsAlign() === Enums.HorizontalElementsAlign.Stretch) {
                    childWidth = controlInnerWidth / this.orderedControls.length;

                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.setControlWidth(childWidth + 'px');
                    }
                }
                else {
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.applyWidth();
                    }
                }
            }
            else {
                if (this.mvc.model.getHorizontalElementsAlign() === Enums.HorizontalElementsAlign.Stretch) {
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.setControlWidth(controlInnerWidth + 'px');
                    }
                }
                else {
                    for (i = 0; i < this.orderedControls.length; i++) {
                        this.orderedControls[i].controller.applyWidth();
                    }
                }
            }
        },

        /*
         * @method repositionControls used to setup x and y coordinates of controls
         * @param startIndex index of control which caused reposition (to reposition only limited number of controls). 
         *  If -1 then all controls will be repositioned
        */
        repositionControls: function (startIndex) {

            if (this.orderedControls.length === 0) {
                return;
            }

            var emptyRect = this.calcClientRect();

            if (emptyRect.width <= 0 || emptyRect.height <= 0) {
                //optimization
                return;
            }

            if (this.mvc.model.getOrientation() === Enums.OrientationType.Vertical) {
                this.verticalReposition(startIndex, emptyRect);
            }
            else {
                this.horizontalReposition(startIndex, emptyRect);
            }
        },

        /*
         * @method verticalReposition used to setup x and y coordinates of controls if orientation is VERTICAL
         * @param startIndex index of control which caused reposition (to reposition only limited number of controls). 
         *  If -1 then all controls will be repositioned
        */
        verticalReposition: function (startIndex, emptyRect) {
            if (startIndex === -1) {
                startIndex = this.mvc.model.getVerticalElementsAlign() === Enums.VerticalElementsAlign.Bottom ?
                    this.orderedControls.length - 1 : 0;
            }

            var controlsBox = this.calcBoxForVertical(startIndex, emptyRect);

            switch (this.mvc.model.getVerticalElementsAlign()) {
                case Enums.VerticalElementsAlign.Top:
                    this.verticalTopReposition(startIndex, controlsBox);
                    break;
                case Enums.VerticalElementsAlign.Center:
                    this.verticalCenterReposition(startIndex, controlsBox);
                    break;
                case Enums.VerticalElementsAlign.Bottom:
                    this.verticalBottomReposition(startIndex, controlsBox);
                    break;
                case Enums.VerticalElementsAlign.Stretch:
                    //do same as for TOP. it will be called only for all controls
                    this.verticalTopReposition(startIndex, controlsBox);
                    break;
            }
        },

        verticalTopReposition: function (startIndex, emptyRect) {
            var i,
                childHeight,
                childControl,
                horizontalAlign = this.mvc.model.getHorizontalElementsAlign();

            //children are on top, so we can start ajusting from this and move downward
            for (i = startIndex; i < this.orderedControls.length; i++) {
                childControl = this.orderedControls[i];
                childHeight = childControl.controller.getControlHeightPx();
                childControl.controller.setTopAnchor(emptyRect.top + 'px');

                this.applyHorizontalInVertical(childControl, emptyRect, horizontalAlign);

                emptyRect.top += childHeight;
            }
        },

        verticalCenterReposition: function (startIndex, emptyRect) {
            var i,
                childHeight,
                childControl,
                horizontalAlign = this.mvc.model.getHorizontalElementsAlign();

            //children are centered verticaly, so we need to adjust position of all
            for (i = 0; i < this.orderedControls.length; i++) {
                childControl = this.orderedControls[i];
                childHeight = childControl.controller.getControlHeightPx();
                childControl.controller.setTopAnchor(emptyRect.top + 'px');

                this.applyHorizontalInVertical(childControl, emptyRect, horizontalAlign);

                emptyRect.top += childHeight;
            }
        },

        verticalBottomReposition: function (startIndex, emptyRect) {
            var i,
                childHeight,
                childControl,
                horizontalAlign = this.mvc.model.getHorizontalElementsAlign();

            //children are on bottom, so we can start ajusting positions from this and move upward
            for (i = startIndex; i >= 0; i--) {
                childControl = this.orderedControls[i];
                childHeight = childControl.controller.getControlHeightPx();
                childControl.controller.setBottomAnchor(emptyRect.bottom + 'px');

                this.applyHorizontalInVertical(childControl, emptyRect, horizontalAlign);

                emptyRect.bottom += childHeight;
            }
        },

        applyHorizontalInVertical: function (child, controlsBox, horizontalAlign) {
            switch (horizontalAlign) {
                case Enums.HorizontalElementsAlign.Left:
                    child.controller.setLeftAnchor(controlsBox.left + 'px');
                    break;
                case Enums.HorizontalElementsAlign.Center:
                    var left = controlsBox.left + (controlsBox.width - child.controller.getControlWidthPx()) / 2;
                    child.controller.setLeftAnchor(Math.floor(left) + 'px');
                    break;
                case Enums.HorizontalElementsAlign.Right:
                    child.controller.setRightAnchor(controlsBox.right + 'px');
                    break;
                case Enums.HorizontalElementsAlign.Stretch:
                    child.controller.setLeftAnchor(controlsBox.left + 'px');
                    break;
            }
        },

        /*
         * @method horizontalReposition used to setup x and y coordinates of controls if orientation is HORIZONTAL
         * @param startIndex index of control which caused reposition (to reposition only limited number of controls). 
         *  If -1 then all controls will be repositioned
        */
        horizontalReposition: function (startIndex, emptyRect) {
            if (startIndex === -1) {
                startIndex = this.mvc.model.getHorizontalElementsAlign() === Enums.HorizontalElementsAlign.Right ?
                    this.orderedControls.length - 1 : 0;
            }

            var controlsBox = this.calcBoxForHorizontal(startIndex, emptyRect);

            switch (this.mvc.model.getHorizontalElementsAlign()) {
                case Enums.HorizontalElementsAlign.Left:
                    this.horizontalLeftReposition(startIndex, controlsBox);
                    break;
                case Enums.HorizontalElementsAlign.Center:
                    this.horizontalCenterReposition(startIndex, controlsBox);
                    break;
                case Enums.HorizontalElementsAlign.Right:
                    this.horizontalRightReposition(startIndex, controlsBox);
                    break;
                case Enums.HorizontalElementsAlign.Stretch:
                    //do same as for LEFT. it will be called only for all controls
                    this.horizontalLeftReposition(startIndex, controlsBox);
                    break;
            }
        },

        horizontalLeftReposition: function (startIndex, emptyRect) {
            var i,
                childWidth,
                childControl,
                verticalAlign = this.mvc.model.getVerticalElementsAlign();

            //children are on top, so we can start ajusting from this and move downward
            for (i = startIndex; i < this.orderedControls.length; i++) {
                childControl = this.orderedControls[i];
                childWidth = childControl.controller.getControlWidthPx();
                childControl.controller.setLeftAnchor(emptyRect.left + 'px');

                this.applyVerticalInHorizontal(childControl, emptyRect, verticalAlign);

                emptyRect.left += childWidth;
            }
        },

        horizontalCenterReposition: function (startIndex, emptyRect) {
            var i,
                childWidth,
                childControl,
                verticalAlign = this.mvc.model.getVerticalElementsAlign();

            //children are centered verticaly, so we need to adjust position of all
            for (i = 0; i < this.orderedControls.length; i++) {
                childControl = this.orderedControls[i];
                childWidth = childControl.controller.getControlWidthPx();
                childControl.controller.setLeftAnchor(emptyRect.left + 'px');

                this.applyVerticalInHorizontal(childControl, emptyRect, verticalAlign);

                emptyRect.left += childWidth;
            }
        },

        horizontalRightReposition: function (startIndex, emptyRect) {
            var i,
                childWidth,
                childControl,
                verticalAlign = this.mvc.model.getVerticalElementsAlign();

            //children are on bottom, so we can start ajusting positions from this and move upward
            for (i = startIndex; i >= 0; i--) {
                childControl = this.orderedControls[i];
                childWidth = childControl.controller.getControlWidthPx();
                childControl.controller.setRightAnchor(emptyRect.right + 'px');

                this.applyVerticalInHorizontal(childControl, emptyRect, verticalAlign);

                emptyRect.right += childWidth;
            }
        },

        applyVerticalInHorizontal: function (child, controlsBox, verticalAlign) {
            switch (verticalAlign) {
                case Enums.VerticalElementsAlign.Top:
                    child.controller.setTopAnchor(controlsBox.top + 'px');
                    break;
                case Enums.VerticalElementsAlign.Center:
                    var top = controlsBox.top + (controlsBox.height - child.controller.getControlHeightPx()) / 2;
                    child.controller.setTopAnchor(Math.floor(top) + 'px');
                    break;
                case Enums.VerticalElementsAlign.Bottom:
                    child.controller.setBottomAnchor(controlsBox.bottom + 'px');
                    break;
                case Enums.VerticalElementsAlign.Stretch:
                    child.controller.setTopAnchor(controlsBox.top + 'px');
                    break;
            }
        },

        childrenHeight: function (startIndex, endIndex) {
            var i,
                height = 0;

            if (endIndex < startIndex) {
                return 0;
            }

            for (i = startIndex; i <= endIndex; i++) {
                height += this.orderedControls[i].controller.getControlHeightPx();
            }

            return height;
        },

        childrenWidth: function (startIndex, endIndex) {
            var i,
                width = 0;

            if (endIndex < startIndex) {
                return 0;
            }

            for (i = startIndex; i <= endIndex; i++) {
                width += this.orderedControls[i].controller.getControlWidthPx();
            }

            return width;
        },

        calcClientRect: function () {
            var borderWidth = this.mvc.model.getBorderThickness(),
                clientRect = {
                    left: 0,
                    right: 0,
                    top: 0,
                    bottom: 0,
                    width: this.getControlWidthPx() - borderWidth * 2,//this.getControlInnerWidth(), //using border width because innerWidth becomes nonzero too late
                    height: this.getControlHeightPx() - borderWidth * 2,//this.getControlInnerHeight()//this.getControlHeightPx() - borderWidth * 2,
                };

            return clientRect;
        },

        /*
        * @method calcBoxForVertical calculates surraunding box for controls which positions should be adjusted in case if 
        *   control with startIndex will change its height
        * @param {int} startIndex - index of control
        * @param @clientRect - client rect of container control
        */
        calcBoxForVertical: function (startIndex, clientRect) {
            var childrenHeight,
                previousChildrenHeight,
                box = {};
            $.extend(box, clientRect);

            switch (this.mvc.model.getVerticalElementsAlign()) {
                case Enums.VerticalElementsAlign.Top:
                    childrenHeight = this.childrenHeight(0, startIndex - 1);
                    box.top += childrenHeight;
                    break;
                case Enums.VerticalElementsAlign.Center:
                    childrenHeight = this.childrenHeight(0, this.orderedControls.length - 1);
                    box.top += Math.floor((box.height - childrenHeight) / 2);
                    break;
                case Enums.VerticalElementsAlign.Bottom:
                    childrenHeight = this.childrenHeight(0, startIndex);
                    previousChildrenHeight = this.childrenHeight(startIndex + 1, this.orderedControls.length - 1);
                    box.bottom += previousChildrenHeight;
                    box.top += childrenHeight + previousChildrenHeight;//
                    break;
                case Enums.VerticalElementsAlign.Stretch:
                    //return box because we will be here only if we need to rebuild all
                    break;
            }

            return box;
        },

        /*
       * @method calcBoxForHorizontal calculates surraunding box for controls which positions should be adjusted in case if 
       *   control with startIndex will change its height
       * @param {int} startIndex - index of control
       * @param @clientRect - client rect of container control
       */
        calcBoxForHorizontal: function (startIndex, clientRect) {
            var childrenWidth,
                previousChildrenWidth,
                box = {};
            $.extend(box, clientRect);

            switch (this.mvc.model.getHorizontalElementsAlign()) {
                case Enums.HorizontalElementsAlign.Left:
                    childrenWidth = this.childrenWidth(0, startIndex - 1);
                    box.left += childrenWidth;
                    break;
                case Enums.HorizontalElementsAlign.Center:
                    childrenWidth = this.childrenWidth(0, this.orderedControls.length - 1);
                    box.left += Math.floor((box.width - childrenWidth) / 2);
                    break;
                case Enums.HorizontalElementsAlign.Right:
                    childrenWidth = this.childrenWidth(0, startIndex);
                    previousChildrenWidth = this.childrenWidth(startIndex + 1, this.orderedControls.length - 1);
                    box.right += previousChildrenWidth;
                    box.left += childrenWidth + previousChildrenWidth;//
                    break;
                case Enums.HorizontalElementsAlign.Stretch:
                    //return box because we will be here only if we need to rebuild all                    
                    break;
            }

            return box;
        }
    });

    return StackPanelController;
});
