﻿// <copyright file="VariablesSource.js" company="ИнСАТ">
// ИнСАТ, 2014
// </copyright>
// 


define(['common/Enums', 'base/EventTarget', 'common/Error', 'server/VariablesManagersHandler'],
    function (Enums, EventTarget, Error, VariablesManagersHandler) {

        var VariablesSource = Class.extend({
            dataSourceKey: 'dataSourceId',
            getValuesKey: 'getValues',
            setValuesKey: 'setValues',           
            init: function (ownerId) {
                this.ownerId = ownerId; //в VH может лежать одинаковые VH от разных VS
                this.eventTarget = new EventTarget();
                this.variablesManagers = {}; //dataSourceId : VM object      
                this.lastSubscribeDataSourcesVars = {};
                //т.к. источник могут использовать несколько потребителей
                //то не всем им понадобится события на готовность переменных от всех VM
                this.objectIdDataSources = {}; //{id : { s1: false, s2: false}}
                this.vmServerDataReady = {}; //{id: false} //подписаны ли мы уже
                this.vmForSubscribe = {}; //VM, которым были добпвленные переменные,
                //и на которых теперь нужно вызвать subscriиe
            },

            setSourcesOptions: function (options) {
                _.forEach(this.variablesManagers, function (vm) {
                    vm.setOptions(options);
                });
            },

            //обязательно должен вызываться до подписок на события, т.к создает VM
            //objectId - окно (элемент), ожидающий ответа по vars
            addToSubscription: function (vars, objectId) {
                //если переменные только set, то ServerDataReady не будет
                if (vars.getValues && vars.getValues.length > 0 && !this.objectIdDataSources[objectId]) {
                    this.objectIdDataSources[objectId] = {};
                }

                var i;
                for (i = 0; i < vars[this.getValuesKey].length; i++) {
                    //false - показывает, подписались ли мы на ServerDataReady этого источника
                    this.objectIdDataSources[objectId][vars[this.getValuesKey][i][this.dataSourceKey]] =
                        this.objectIdDataSources[objectId][vars[this.getValuesKey][i][this.dataSourceKey]] || false;
                }

                //группируем по datasourceId: {getValues, setValues}
                var lastSubscribeDataSourcesVars = this._getVariablesByDataSource(vars);
                for (var dsId in lastSubscribeDataSourcesVars){
                    this.vmForSubscribe[dsId] = {};
                }
                var ctx = this;
                _.forOwn(lastSubscribeDataSourcesVars, function (dataSourceVars, id) {
                    ctx.getVMById(id).addToSubscription(dataSourceVars, objectId);
                });               
            },

            unsubscribe: function (vars, objectId) {
                var dataSourcesVars = this._getVariablesByDataSource(vars);
                var ctx = this;
                _.forOwn(dataSourcesVars, function (dataSourceVars, id) {
                    ctx.getVMById(id).unsubscribe(dataSourceVars, objectId);
                });
            },            

            _getVariablesByDataSource: function (vars) {
                var i,
                   dataSourcesVars = {};
           
                for (i = 0; i < vars.getValues.length; i++) {
                    this._addSubscribeVariableToContainer(dataSourcesVars, this.getValuesKey, vars.getValues[i]);
                }
             
                for (i = 0; i < vars.setValues.length; i++) {
                    this._addSubscribeVariableToContainer(dataSourcesVars, this.setValuesKey, vars.setValues[i]);
                }

                return dataSourcesVars;
            },

            _addSubscribeVariableToContainer: function (dataSourcesVars, varArrayKey, variable) {
                if (!dataSourcesVars[variable[this.dataSourceKey]]) {
                    dataSourcesVars[variable[this.dataSourceKey]] = {};
                }

                if (!dataSourcesVars[variable[this.dataSourceKey]][varArrayKey]) {
                    dataSourcesVars[variable[this.dataSourceKey]][varArrayKey] = [];
                }

                dataSourcesVars[variable[this.dataSourceKey]][varArrayKey].push(variable);
            },

            getVMById: function (id) {               
                if (this.variablesManagers[id] === undefined) {
                    this.variablesManagers[id] = this._createVM(id);
                }

                return this.variablesManagers[id];
            },

            getVMByType: function(type){
            },           

            _createVM: function (id) {                
                return VariablesManagersHandler.get(id, this.ownerId, Enums.variablesManagersSubTypes.scheme);
            },          

            subscribe: function () {
                var promises = [];
                for (var dsId in this.vmForSubscribe) {
                    var vm = this.getVMById(dsId);
                    promises.push(vm.subscribe());
                }             
                
                this.vmForSubscribe = {};
                return Promise.all(promises);
            },

            _onServerDataReady: function (event) {
                //
                if (this.objectIdDataSources[event.windowId] === undefined) {
                    return;
                }

                if (this.objectIdDataSources[event.windowId][event.sourceId] !== undefined) {
                    delete this.objectIdDataSources[event.windowId][event.sourceId]
                } else {
                    Error.debug('Variables source: unexpected ServerDataReady');
                }

                if (Object.keys(this.objectIdDataSources[event.windowId]).length === 0) {
                    this.fireSourceDataReady(event.windowId);
                    //объект больше не ждет события от источника
                    delete this.objectIdDataSources[event.windowId];
                }

                if (Object.keys(this.objectIdDataSources).length === 0) {
                    //все всё получили - очищаем список подписчиков
                    this.eventTarget.removeAllListeners(Enums.eventType.sourceDataReady);
                    this.vmServerDataReady = {};
                }
            },

            subscribeSourceDataReady: function (handler, context, objectId) {              
                this._subscribeSourceDataReadyInternal(handler, context, objectId);
                this.eventTarget.addListener(Enums.eventType.sourceDataReady, handler, context);
            },

            subscribeSourceDataReadyOnce: function (handler, context, objectId) {
                this._subscribeSourceDataReadyInternal(handler, context, objectId);
                this.eventTarget.bindOnce(Enums.eventType.sourceDataReady, handler, context);
            },

            _subscribeSourceDataReadyInternal: function (handler, context, objectId) {
                if (objectId === undefined) {
                    Error.warn('SubscribeSourceDataReady: objectId is not specified');
                }
                var ctx = this;
                //проходим по всем VM, в которые объект добавлял переменные
                _.forOwn(this.objectIdDataSources[objectId], function (dataSourceVars, id) {
                    //избегаем дублирования подписок на ServerDataReady
                    if (!ctx.vmServerDataReady[id]) {
                        ctx.variablesManagers[id].subscribeServerDataReady(ctx._onServerDataReady,
                            ctx, objectId);
                        ctx.vmServerDataReady[id] = true;
                    }
                });
            },

            _unsubscribeSourceDataReadyInternal: function (handler, objectId) {
                if (objectId === undefined) {
                    Error.warn('UndubscribeSourceDataReady: objectId is not specified');
                }

                var ctx = this;
                //если остались VM, от которых не успел прийти ответ до отписки
                if (this.objectIdDataSources[objectId]) {
                    //проходим по всем VM, в которые объект добавлял переменные
                    _.forOwn(this.objectIdDataSources[objectId], function (dataSourceVars, id) {
                        //избегаем дублирования подписок на ServerDataReady
                        if (ctx.vmServerDataReady[id]) {
                            ctx.variablesManagers[id].unsubscribeServerDataReady(ctx._onServerDataReady,
                                ctx, objectId);
                            ctx.vmServerDataReady[id] = false;
                        }
                    });

                    delete this.objectIdDataSources[objectId];
                }
            },

            fireSourceDataReady: function (objectId) {
                this.eventTarget.fire({ type: Enums.eventType.sourceDataReady, target: this, objectId: objectId });
            },

            unsubscribeSourceDataReady: function (handler, objectId) {
                this._unsubscribeSourceDataReadyInternal(handler, objectId)
                this.eventTarget.removeListener(Enums.eventType.sourceDataReady, handler);
            },

            //проксируем подписки TODO добавить id для лога
            subscribePropertyChanged: function (handler, context) {
                for(var id in this.variablesManagers) {
                    this.variablesManagers[id].subscribePropertyChanged(handler, context);
                };
            },

            subscribePropertyChangedOnce: function (handler, context) {
                for (var id in this.variablesManagers) {
                    this.variablesManagers[id].subscribePropertyChangedOnce(handler, context);
                };
            },

            unsubscribePropertyChanged: function (handler) {
                for (var id in this.variablesManagers) {
                    this.variablesManagers[id].unsubscribePropertyChanged(handler);
                };
            },

            subscribeServerErrorStateChanged: function (handler, context) {
                //this.eventTarget.addListener(SERVER_CONNECTION_ERROR, handler, context);
            },

            unsubscribeServerErrorStateChanged: function (handler) {
               // this.eventTarget.removeListener(SERVER_CONNECTION_ERROR, handler);
            },

            fireServerErrorStateChanged: function (isError) {
               // this.eventTarget.fire({ type: SERVER_CONNECTION_ERROR, target: this, isError: isError });
            },

            getInfoOnWaitingVariables: function () {
                var result="";
                for (var id in this.variablesManagers) {
                    for (var i = 0; i < this.variablesManagers[id].waitingWidows.length; i++) {
                        result += "Окно " + this.variablesManagers[id].waitingWidows[i].id + ":\n";
                        result += "Параметры: " + Object.keys(this.variablesManagers[id].waitingWidows[i].vars).toString() + "\n";
                    }
                };
                return result;
            }
            
        });

        return VariablesSource;
    });